/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene.directory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.util.Objects;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.BlobFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.OakIndexFile;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.lucene.store.DataInput;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OakStreamingIndexFile
implements OakIndexFile,
AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger((String)OakStreamingIndexFile.class.getName());
    private final String name;
    private final NodeBuilder file;
    private long position = 0L;
    private long length;
    private Blob blob;
    private boolean blobModified = false;
    private InputStream blobInputStream;
    private final byte[] uniqueKey;
    private final String dirDetails;
    private final BlobFactory blobFactory;

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    OakStreamingIndexFile(String name, NodeBuilder file, String dirDetails, @NotNull BlobFactory blobFactory) {
        this.name = name;
        this.file = file;
        this.dirDetails = dirDetails;
        this.uniqueKey = OakStreamingIndexFile.readUniqueKey(file);
        this.blobFactory = Objects.requireNonNull(blobFactory);
        PropertyState property = file.getProperty("jcr:data");
        if (property != null) {
            if (property.getType() != Type.BINARY) throw new IllegalArgumentException("Can't load blob for streaming for " + name + " under " + String.valueOf(file));
            this.blob = (Blob)property.getValue(Type.BINARY);
        } else {
            this.blob = null;
        }
        if (this.blob != null) {
            this.length = this.blob.length();
            if (this.uniqueKey != null) {
                this.length = Math.max(0L, this.length - (long)this.uniqueKey.length);
            }
        }
        this.blobInputStream = null;
    }

    private OakStreamingIndexFile(OakStreamingIndexFile that) {
        this.name = that.name;
        this.file = that.file;
        this.dirDetails = that.dirDetails;
        this.uniqueKey = that.uniqueKey;
        this.position = that.position;
        this.length = that.length;
        this.blob = that.blob;
        this.blobModified = that.blobModified;
        this.blobFactory = that.blobFactory;
    }

    private void setupInputStream() throws IOException {
        if (this.blobInputStream == null) {
            this.blobInputStream = this.blob.getNewStream();
            if (this.position > 0L) {
                long pos = this.position;
                this.position = 0L;
                this.seek(pos);
            }
        }
    }

    private void releaseInputStream() {
        if (this.blobInputStream != null) {
            try {
                this.blobInputStream.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.blobInputStream = null;
        }
    }

    @Override
    public OakIndexFile clone() {
        return new OakStreamingIndexFile(this);
    }

    @Override
    public long length() {
        return this.length;
    }

    @Override
    public long position() {
        return this.position;
    }

    @Override
    public void close() {
        IOUtils.closeQuietly((InputStream)this.blobInputStream);
        this.blob = null;
    }

    @Override
    public boolean isClosed() {
        return this.blobInputStream == null && this.blob == null;
    }

    @Override
    public void seek(long pos) throws IOException {
        if (pos < 0L || pos > this.length) {
            String msg = String.format("Invalid seek request for [%s][%s], position: %d, file length: %d", this.dirDetails, this.name, pos, this.length);
            this.releaseInputStream();
            throw new IOException(msg);
        }
        if (this.blobInputStream == null) {
            this.position = pos;
        } else if (pos < this.position) {
            LOG.warn("Seeking back on streaming index file {}. Current position {}, requested position {}. Please make sure that CopyOnRead and prefetch of index files are enabled.", new Object[]{this.dirDetails + "/" + this.getName(), this.position(), pos});
            IOUtils.closeQuietly((InputStream)this.blobInputStream);
            this.blobInputStream = null;
            this.position = pos;
        } else {
            while (this.position < pos) {
                long skipCnt = this.blobInputStream.skip(pos - this.position());
                if (skipCnt <= 0L) {
                    String msg = String.format("Seek request for [%s][%s], position: %d, file length: %d failed. InputStream.skip returned %d", this.dirDetails, this.name, pos, this.length, skipCnt);
                    this.releaseInputStream();
                    throw new IOException(msg);
                }
                this.position += skipCnt;
            }
        }
    }

    @Override
    public void readBytes(byte[] b, int offset, int len) throws IOException {
        Objects.checkFromToIndex(offset, offset + len, Objects.requireNonNull(b).length);
        if (len < 0 || this.position + (long)len > this.length) {
            String msg = String.format("Invalid byte range request for [%s][%s], position: %d, file length: %d, len: %d", this.dirDetails, this.name, this.position, this.length, len);
            this.releaseInputStream();
            throw new IOException(msg);
        }
        this.setupInputStream();
        int readCnt = IOUtils.read((InputStream)this.blobInputStream, (byte[])b, (int)offset, (int)len);
        if (readCnt < len) {
            String msg = String.format("Couldn't read byte range request for [%s][%s], position: %d, file length: %d, len: %d. Actual read bytes %d", this.dirDetails, this.name, this.position, this.length, len, readCnt);
            this.releaseInputStream();
            throw new IOException(msg);
        }
        this.position += (long)len;
    }

    @Override
    public void writeBytes(final byte[] b, final int offset, final int len) throws IOException {
        if (this.blobModified) {
            throw new IllegalArgumentException("Can't do piece wise upload with streaming access");
        }
        InputStream in = new InputStream(){
            int position;
            {
                this.position = offset;
            }

            @Override
            public int available() throws IOException {
                return offset + len - this.position;
            }

            @Override
            public int read() throws IOException {
                int ret;
                if (this.available() <= 0) {
                    return -1;
                }
                return (ret = b[this.position++]) < 0 ? 256 + ret : ret;
            }

            @Override
            public int read(@NotNull byte[] target, int off, int len2) throws IOException {
                if (this.available() <= 0) {
                    return -1;
                }
                int read = (int)Math.min((long)len2, (long)this.available());
                System.arraycopy(b, this.position, target, off, read);
                this.position += read;
                return read;
            }
        };
        this.pushData(in);
    }

    @Override
    public boolean supportsCopyFromDataInput() {
        return true;
    }

    @Override
    public void copyBytes(final DataInput input, final long numBytes) throws IOException {
        InputStream in = new InputStream(){
            long bytesLeftToRead;
            {
                this.bytesLeftToRead = numBytes;
            }

            @Override
            public int read() throws IOException {
                if (this.bytesLeftToRead <= 0L) {
                    return -1;
                }
                --this.bytesLeftToRead;
                int ret = input.readByte();
                return ret < 0 ? 256 + ret : ret;
            }

            @Override
            public int read(@NotNull byte[] b, int off, int len) throws IOException {
                if (this.bytesLeftToRead == 0L) {
                    return -1;
                }
                int read = (int)Math.min((long)len, this.bytesLeftToRead);
                input.readBytes(b, off, read);
                this.bytesLeftToRead -= (long)read;
                return read;
            }
        };
        this.pushData(in);
    }

    private void pushData(InputStream in) throws IOException {
        if (this.uniqueKey != null) {
            in = new SequenceInputStream(in, new ByteArrayInputStream(this.uniqueKey));
        }
        this.blob = this.blobFactory.createBlob(in);
        this.blobModified = true;
    }

    private static byte[] readUniqueKey(NodeBuilder file) {
        if (file.hasProperty("uniqueKey")) {
            String key = file.getString("uniqueKey");
            return StringUtils.convertHexToBytes((String)key);
        }
        return null;
    }

    @Override
    public void flush() throws IOException {
        if (this.blobModified) {
            this.file.setProperty("jcr:lastModified", (Object)System.currentTimeMillis());
            this.file.setProperty("jcr:data", (Object)this.blob, Type.BINARY);
            this.blobModified = false;
        }
    }

    public String toString() {
        return this.name;
    }

    @Override
    public String getName() {
        return this.name;
    }
}

