/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util.io;

import alluxio.conf.PropertyKey;
import alluxio.util.ConfigurationUtils;
import alluxio.util.io.FileUtils;
import alluxio.util.io.PathUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javax.annotation.Nullable;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public class FileUtilsTest {
    private String mWorkerDataFolderPerms = ConfigurationUtils.defaults().get(PropertyKey.WORKER_DATA_FOLDER_PERMISSIONS);
    @Rule
    public TemporaryFolder mTestFolder = new TemporaryFolder();
    @Rule
    public final ExpectedException mException = ExpectedException.none();

    @Test
    public void changeLocalFilePermission() throws IOException {
        Assume.assumeFalse((boolean)System.getProperty("user.name").equals("root"));
        File tempFile = this.mTestFolder.newFile("perm.txt");
        FileUtils.changeLocalFilePermission((String)tempFile.getAbsolutePath(), (String)"---------");
        Assert.assertFalse((tempFile.canRead() || tempFile.canWrite() || tempFile.canExecute() ? 1 : 0) != 0);
        FileUtils.changeLocalFilePermission((String)tempFile.getAbsolutePath(), (String)"rwxrwxrwx");
        Assert.assertTrue((tempFile.canRead() && tempFile.canWrite() && tempFile.canExecute() ? 1 : 0) != 0);
        FileUtils.changeLocalFilePermission((String)tempFile.getAbsolutePath(), (String)"r--r--r--");
        Assert.assertTrue((boolean)tempFile.canRead());
        Assert.assertFalse((boolean)tempFile.canWrite());
        Assert.assertFalse((boolean)tempFile.canExecute());
        this.mException.expect(IOException.class);
        FileWriter fw = new FileWriter(tempFile);
        Assert.fail((String)"opening a read-only file for writing should have failed");
    }

    @Test
    public void changeNonExistentFile() throws IOException {
        File ghostFile = new File(this.mTestFolder.getRoot(), "ghost.txt");
        this.mException.expect(IOException.class);
        FileUtils.changeLocalFilePermission((String)ghostFile.getAbsolutePath(), (String)"rwxrwxrwx");
        Assert.fail((String)"changing permissions of a non-existent file should have failed");
    }

    @Test
    public void changeLocalDirPermissionTests() throws IOException {
        Assume.assumeFalse((boolean)System.getProperty("user.name").equals("root"));
        File tempFile = this.mTestFolder.newFile("perm.txt");
        FileUtils.changeLocalFilePermission((String)this.mTestFolder.getRoot().getAbsolutePath(), (String)"r--r--r--");
        Assert.assertFalse((boolean)tempFile.delete());
        FileUtils.changeLocalFilePermission((String)this.mTestFolder.getRoot().getAbsolutePath(), (String)"rwxr--r--");
        Assert.assertTrue((boolean)tempFile.delete());
    }

    @Test
    public void moveFile() throws IOException {
        File fromFile = this.mTestFolder.newFile("from.txt");
        File toFile = this.mTestFolder.newFile("to.txt");
        FileUtils.move((String)fromFile.getAbsolutePath(), (String)toFile.getAbsolutePath());
        Assert.assertFalse((boolean)fromFile.exists());
        Assert.assertTrue((boolean)toFile.exists());
    }

    @Test
    public void moveNonExistentFile() throws IOException {
        File ghostFile = new File(this.mTestFolder.getRoot(), "ghost.txt");
        File toFile = this.mTestFolder.newFile("to.txt");
        this.mException.expect(IOException.class);
        FileUtils.move((String)ghostFile.getAbsolutePath(), (String)toFile.getAbsolutePath());
        Assert.fail((String)"moving a non-existent file should have failed");
    }

    @Test
    public void deleteFile() throws IOException {
        File tempFile = this.mTestFolder.newFile("fileToDelete");
        File tempFolder = this.mTestFolder.newFolder("dirToDelete");
        FileUtils.delete((String)tempFile.getAbsolutePath());
        FileUtils.delete((String)tempFolder.getAbsolutePath());
        Assert.assertFalse((boolean)tempFile.exists());
        Assert.assertFalse((boolean)tempFolder.exists());
    }

    @Test
    public void deletePathRecursively() throws IOException {
        File tmpDir = this.mTestFolder.newFolder("dir");
        File tmpDir1 = this.mTestFolder.newFolder(new String[]{"dir", "dir1"});
        File tmpDir2 = this.mTestFolder.newFolder(new String[]{"dir", "dir2"});
        File tmpFile1 = this.mTestFolder.newFile("dir/dir1/file1");
        File tmpFile2 = this.mTestFolder.newFile("dir/dir1/file2");
        File tmpFile3 = this.mTestFolder.newFile("dir/file3");
        FileUtils.deletePathRecursively((String)tmpDir.getAbsolutePath());
        Assert.assertFalse((boolean)tmpDir.exists());
        Assert.assertFalse((boolean)tmpDir1.exists());
        Assert.assertFalse((boolean)tmpDir2.exists());
        Assert.assertFalse((boolean)tmpFile1.exists());
        Assert.assertFalse((boolean)tmpFile2.exists());
        Assert.assertFalse((boolean)tmpFile3.exists());
    }

    @Test
    public void deleteNonExistentFile() throws IOException {
        File ghostFile = new File(this.mTestFolder.getRoot(), "ghost.txt");
        this.mException.expect(IOException.class);
        FileUtils.delete((String)ghostFile.getAbsolutePath());
        Assert.fail((String)"deleting a non-existent file should have failed");
    }

    @Test
    public void setLocalDirStickyBit() throws IOException {
        File tempFolder = this.mTestFolder.newFolder("dirToModify");
        if (tempFolder.getAbsolutePath().startsWith("/")) {
            FileUtils.setLocalDirStickyBit((String)tempFolder.getAbsolutePath());
            ArrayList<String> commands = new ArrayList<String>();
            commands.add("/bin/ls");
            commands.add("-ld");
            commands.add(tempFolder.getAbsolutePath());
            try {
                ProcessBuilder builder = new ProcessBuilder(commands);
                Process process = builder.start();
                process.waitFor();
                BufferedReader stdInput = new BufferedReader(new InputStreamReader(process.getInputStream()));
                String line = stdInput.readLine();
                Assert.assertTrue((boolean)line.matches("^d[rwx-]{8}t.*$"));
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    @Test
    public void createBlockPath() throws IOException {
        String absolutePath = PathUtils.concatPath((Object)this.mTestFolder.getRoot(), (Object[])new Object[]{"tmp", "bar"});
        File tempFile = new File(absolutePath);
        FileUtils.createBlockPath((String)tempFile.getAbsolutePath(), (String)this.mWorkerDataFolderPerms);
        Assert.assertTrue((boolean)FileUtils.exists((String)tempFile.getParent()));
    }

    @Test
    public void createFile() throws IOException {
        File tempFile = new File(this.mTestFolder.getRoot(), "tmp");
        FileUtils.createFile((String)tempFile.getAbsolutePath());
        Assert.assertTrue((boolean)FileUtils.exists((String)tempFile.getAbsolutePath()));
        Assert.assertTrue((boolean)tempFile.delete());
    }

    @Test
    public void createDir() throws IOException {
        File tempDir = new File(this.mTestFolder.getRoot(), "tmp");
        FileUtils.createDir((String)tempDir.getAbsolutePath());
        Assert.assertTrue((boolean)FileUtils.exists((String)tempDir.getAbsolutePath()));
        Assert.assertTrue((boolean)tempDir.delete());
    }

    @Test
    public void getLocalFileMode() throws IOException {
        File tmpDir = this.mTestFolder.newFolder("dir");
        File tmpFile777 = this.mTestFolder.newFile("dir/0777");
        tmpFile777.setReadable(true, false);
        tmpFile777.setWritable(true, false);
        tmpFile777.setExecutable(true, false);
        File tmpFile755 = this.mTestFolder.newFile("dir/0755");
        tmpFile755.setReadable(true, false);
        tmpFile755.setWritable(false, false);
        tmpFile755.setExecutable(true, false);
        tmpFile755.setWritable(true, true);
        File tmpFile444 = this.mTestFolder.newFile("dir/0444");
        tmpFile444.setReadOnly();
        Assert.assertEquals((long)511L, (long)FileUtils.getLocalFileMode((String)tmpFile777.getPath()));
        Assert.assertEquals((long)493L, (long)FileUtils.getLocalFileMode((String)tmpFile755.getPath()));
        Assert.assertEquals((long)292L, (long)FileUtils.getLocalFileMode((String)tmpFile444.getPath()));
        FileUtils.deletePathRecursively((String)tmpDir.getAbsolutePath());
    }

    @Test
    public void createStorageDirPath() throws IOException {
        File storageDir = new File(this.mTestFolder.getRoot(), "storageDir");
        File blockFile = new File(storageDir, "200");
        FileUtils.createBlockPath((String)blockFile.getAbsolutePath(), (String)this.mWorkerDataFolderPerms);
        Assert.assertTrue((boolean)FileUtils.exists((String)storageDir.getAbsolutePath()));
        Assert.assertEquals(PosixFilePermissions.fromString("rwxrwxrwx"), Files.getPosixFilePermissions(Paths.get(storageDir.getAbsolutePath(), new String[0]), new LinkOption[0]));
        FileUtils.createBlockPath((String)blockFile.getAbsolutePath(), (String)this.mWorkerDataFolderPerms);
        Assert.assertTrue((boolean)FileUtils.exists((String)storageDir.getAbsolutePath()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void concurrentCreateStorageDirPath() throws Exception {
        int numCreators = 5;
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(5);
        for (int iteration = 0; iteration < 5; ++iteration) {
            ExecutorService executor = Executors.newFixedThreadPool(5);
            CyclicBarrier barrier = new CyclicBarrier(5);
            try {
                File storageDir = new File(this.mTestFolder.getRoot(), "tmp" + iteration);
                for (int i = 0; i < 5; ++i) {
                    File file = new File(storageDir, String.valueOf(i));
                    class ConcurrentCreator
                    implements Callable<Void> {
                        private final String mPath;
                        private final CyclicBarrier mBarrier;

                        ConcurrentCreator(String path, CyclicBarrier barrier) {
                            this.mPath = path;
                            this.mBarrier = barrier;
                        }

                        @Override
                        @Nullable
                        public Void call() throws Exception {
                            this.mBarrier.await();
                            FileUtils.createBlockPath((String)this.mPath, (String)FileUtilsTest.this.mWorkerDataFolderPerms);
                            return null;
                        }
                    }
                    futures.add(executor.submit(new ConcurrentCreator(file.getAbsolutePath(), barrier)));
                }
                for (Future future : futures) {
                    future.get();
                }
                Assert.assertTrue((boolean)FileUtils.exists((String)storageDir.getAbsolutePath()));
                continue;
            }
            finally {
                executor.shutdown();
            }
        }
    }
}

