/**
 * Copyright 2011-2016 GatlingCorp (http://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gatling.core.action

import scala.concurrent.duration.{ Duration, DurationLong }

import io.gatling.commons.util.ClockSingleton.nowMillis
import io.gatling.core.session.{ Expression, Session }
import io.gatling.core.stats.StatsEngine
import io.gatling.core.util.NameGen

import akka.actor.ActorSystem

/**
 * Pace provides a means to limit the frequency with which an action is run, by specifying a minimum wait time between iterations.
 *
 * Originally contributed by James Pickering.
 *
 * @param intervalExpr a function that decides how long to wait before the next iteration
 * @param counter the name of the counter used to keep track of the run state. Typically this would be random, but
 *                can be set explicitly if needed
 * @param statsEngine the StatsEngine
 * @param next the next actor in the chain
 */
class Pace(intervalExpr: Expression[Duration], counter: String, system: ActorSystem, val statsEngine: StatsEngine, val next: Action) extends ExitableAction with NameGen {

  import system._

  override val name = genName("pace")

  /**
   * Pace keeps track of when it can next run using a counter in the session. If this counter does not exist, it will
   * run immediately. On each run, it increments the counter by intervalExpr.
   *
   * @param session the session of the virtual user
   * @return nothing
   */
  override def execute(session: Session): Unit = recover(session) {
    intervalExpr(session) map { interval =>
      val startTimeOpt = session(counter).asOption[Long]
      val startTime = startTimeOpt.getOrElse(nowMillis)
      val nextStartTime = startTime + interval.toMillis
      val waitTime = startTime - nowMillis

        def doNext() = next ! session.set(counter, nextStartTime)

      if (waitTime > 0) {
        scheduler.scheduleOnce(waitTime milliseconds)(doNext())
      } else {
        if (startTimeOpt.isDefined) logger.info(s"Previous run overran by ${-waitTime}ms. Running immediately")
        doNext()
      }
    }
  }
}
