package io.agora.push;

import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Pair;

import io.agora.Error;
import io.agora.chat.ChatClient;
import io.agora.chat.core.EMPreferenceUtils;
import io.agora.cloud.EMHttpClient;
import io.agora.push.common.PushUtil;
import io.agora.push.platform.IPush;
import io.agora.push.platform.fcm.EMFCMPush;
import io.agora.push.platform.hms.EMHMSPush;
import io.agora.push.platform.honor.EMHonorPush;
import io.agora.push.platform.meizu.EMMzPush;
import io.agora.push.platform.mi.EMMiPush;
import io.agora.push.platform.normal.EMNormalPush;
import io.agora.push.platform.oppo.EMOppoPush;
import io.agora.push.platform.vivo.EMVivoPush;
import io.agora.util.DeviceUuidFactory;
import io.agora.util.EMLog;

import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Random;

/**
 * \~english
 * The push help class.
 */
public class PushHelper {
    private static final String TAG = "PushHelper";

    private static final int TIMES_RETRY = 3;
    private static final int MSG_WHAT_REGISTER = 0;
    private static final int MSG_WHAT_UNREGISTER = 1;
    private static final int MSG_WHAT_UPLOAD = 2;

    private Context context;
    private PushConfig pushConfig;
    private IPush pushClient;
    private Handler handler;

    private PushType pushType;
    private String pushToken;
    private boolean unregisterSuccess;

    private final Object bindLock = new Object();
    private final Object unbindLock = new Object();

    private PushListener pushListener;

    private boolean isClientInit = false;

    private PushHelper() {
        final HandlerThread handlerThread = new HandlerThread("token-uploader");
        handlerThread.start();

        handler = new Handler(handlerThread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case MSG_WHAT_REGISTER:
                        synchronized (bindLock) { // 确保task在执行时没有新的tasks加入，避免该task执行完后把新加入的tasks清空。
                            String token = (String) msg.obj;
                            boolean registerSuccess = uploadTokenInternal(pushClient.getNotifierName(), token);
                            if (registerSuccess) {
                                removeMessages(MSG_WHAT_REGISTER);
                                return;
                            }
                            // 当所有task执行完且还没有上传成功时，更换为NORMAL push type。
                            boolean allTasksExecuted = !hasMessages(MSG_WHAT_REGISTER);
                            if (allTasksExecuted) {
                                PushHelper.this.onErrorResponse(pushType, Error.PUSH_BIND_FAILED);
                                register(PushType.NORMAL);
                            }
                        }
                        break;
                    case MSG_WHAT_UNREGISTER:
                        unregisterSuccess = uploadTokenInternal(pushClient.getNotifierName(), "");
                        if (!unregisterSuccess) {
                            PushHelper.this.onErrorResponse(pushType, Error.PUSH_UNBIND_FAILED);
                        }

                        synchronized (unbindLock) {
                            unbindLock.notifyAll();
                        }
                        break;
                    case MSG_WHAT_UPLOAD:
                        Bundle data = msg.getData();
                        if(data == null) {
                            return;
                        }
                        String type = data.getString("type", "");
                        if(TextUtils.isEmpty(type)) {
                            return;
                        }
                        PushType pushType = PushType.getType(type);
                        uploadTokenByTimes(pushType, (String) msg.obj);
                        break;
                    default:
                        super.handleMessage(msg);
                        break;
                }
            }
        };
    }

    /**
     * \~english
     * Gets a singleton of PushHelper.
     * @return The singleton of PushHelper.
     */
    public static PushHelper getInstance() {
        return InstanceHolder.INSTANCE;
    }

    /**
     * \~english
     * Sets the push listener.
     * You can override {@link PushListener#isSupportPush(PushType, PushConfig)} to set the push types. Supported push types are PushType.FCM，PushType.MIPUSH，PushType.HMSPUSH，PushType.MEIZUPUSH，PushType.OPPOPUSH，PushType.VIVOPUSH. If you set a push type beyond this scope, the SDK will choose the Normal push of the chat service without error information without reporting error information.
     * @param callback  The push listener.
     */
    public void setPushListener(PushListener callback) {
        this.pushListener = callback;
    }

    /**
     * \~english
     * Initializes the PushHelper.
     * @param context  The context of Android Activity or Application.
     * @param config   The Push configurations.
     */
    public void init(Context context, PushConfig config) {
        EMLog.e(TAG, TAG + " init, config: " + config.toString());

        if (context == null || config == null) {
            throw new IllegalArgumentException("Null parameters, context=" + context + ", config=" + config);
        }

        this.context = context.getApplicationContext();
        this.pushConfig = config;
    }

    /**
     * \~english
     * Registers the push service.
     * After successful login, the SDK calls the method instead of you.
     */
    public void register() {
        if (context == null || pushConfig == null) {
            EMLog.e(TAG, "PushHelper#init(Context, PushConfig) method not call previously.");
            return;
        }

        PushType pushType = getPreferPushType(pushConfig);
        EMLog.e(TAG, TAG + " register, prefer push type: " + pushType);
        register(pushType);
    }

    /**
     * \~english
     * Unregisters the push service.
     * This method is called by SDK during the call of {@link ChatClient#logout(boolean)}}.  
     * @param unbindToken   Whether to unbind the device.
     * @return Whether the unregistering push succeeds.
     */
    public boolean unregister(boolean unbindToken) {
        EMLog.e(TAG, TAG + " unregister, unbind token: " + unbindToken);

        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered previously, return true directly.");
            return true;
        }

        this.isClientInit = false;
        pushClient.unregister(context);

        // 停止 Token 上传操作。
        handler.removeMessages(MSG_WHAT_UPLOAD);
        handler.removeMessages(MSG_WHAT_REGISTER);

        if (!unbindToken) {
            pushType = null;
            return true;
        }

        if(pushType != null && pushType != PushType.NORMAL) {
            deleteToken();
            // Wait for token delete result.
            synchronized (unbindLock) {
                try {
                    unbindLock.wait();
                } catch (InterruptedException e) {
                }
            }
            if (unregisterSuccess) {
                pushType = null;
            }
        }else {
            unregisterSuccess = true;
            pushType = null;
        }
        EMLog.e(TAG, "Push type after unregister is " + pushType);
        return unregisterSuccess;
    }

    private boolean isTokenChanged(PushType type, final String token) {
        String savedToken = PushHelper.getInstance().getPushTokenWithType(type);
        return (savedToken == null) || !savedToken.equals(token);
    }

    /**
     * \~english
     * Receives and uploads the device token。
     * @param type  The push type.
     * @param token  The device token.
     */
    public void onReceiveToken(PushType type, final String token) {
        EMLog.e(TAG, "onReceiveToken: " + type + " - " + token);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort token upload action.");
            return;
        }
        this.pushToken = token;
        if (isTokenChanged(type, token)) {
            EMLog.d(TAG, "push token changed, upload to server");
            uploadToken(type, token);
            return;
        }
        if (ChatClient.getInstance().getChatConfigPrivate().isNewLoginOnDevice()) {
            EMLog.d(TAG, "push token not change, but last login is not on this device, upload to server");
            uploadToken(type, token);
        } else {
            EMLog.e(TAG, TAG + " not first login, ignore token upload action.");
        }
    }

    /**
     * \~english
     * Occurs when a push error occurs.
     * The SDK triggers this callback when a push error, such as push token binding failure (PUSH_BIND_FAILED), push token unbinding failure (PUSH_UNBIND_FAILED), or unsupported custom types (PUSH_NOT_SUPPORT), occurs. You can have more detailed information on the error from the returned resultCode.
     * If the customized PushListener is not null, pass error info to {@link PushListener#onError(PushType, long)}.
     * @param type          The push type.
     * @param resultCode    The error code.
     */
    public void onErrorResponse(PushType type, long resultCode) {
        EMLog.e(TAG, "onErrorResponse: " + type + " - " + resultCode);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort error response action.");
            return;
        }

        if (resultCode == Error.PUSH_NOT_SUPPORT) {
            register(PushType.NORMAL);
        }

        if (pushListener != null) pushListener.onError(type, resultCode);
    }

    /**
     * \~english
     * Gets the push type.
     * @return  The push type.
     */
    public PushType getPushType() {
        return pushType;
    }

    /**
     * \~english
     * Gets the push token.
     * @return  The push token.
     */
    public String getPushToken() {
        return pushToken;
    }

    /**
     * \~english
     * Gets the FCM's push token.
     * @return  The FCM's push token.
     */
    public String getFCMPushToken() {
        return EMPreferenceUtils.getInstance().getFCMPushToken();
    }

    /**
     * \~english
     * Saves the FCM's push token.
     * @return  The FCM's push token.
     */
    public void setFCMPushToken(String token) {
        EMPreferenceUtils.getInstance().setFCMPushToken(token);
    }

     /**
     * \~english
     * Gets the push token with the push type.
     * 
     * @param type The push type.
     * @return The push token.
     */
    public String getPushTokenWithType(PushType type) {
        return EMPreferenceUtils.getInstance().getPushTokenWithType(type);
    }

     /**
     * \~english
     * Sets the push token with the push type.
     * 
     * @param type The push type.
     * @param token The push token.
     */
    public void setPushTokenWithType(PushType type, final String token) {
        EMPreferenceUtils.getInstance().setPushTokenWithType(type, token);
    }

    /**
     * 注册
     * @param pushType
     * @param hasReceiveToken 是否已经从三方SDK接收到 device token，如果接收到，则不再获取对应的token。
     */
    private void register(@NonNull PushType pushType, boolean hasReceiveToken) {
        if (this.pushType == pushType) {
            EMLog.e(TAG, "Push type " + pushType + " no change, return. ");
            return;
        }

        if (this.pushClient != null) {
            EMLog.e(TAG, pushClient.getPushType() + " push already exists, unregister it and change to " + pushType + " push.");
            pushClient.unregister(context);
        }

        this.pushType = pushType;

        switch (pushType) {
            case FCM:
                pushClient = new EMFCMPush();
                break;
            case MIPUSH:
                pushClient = new EMMiPush();
                break;
            case OPPOPUSH:
                pushClient = new EMOppoPush();
                break;
            case VIVOPUSH:
                pushClient = new EMVivoPush();
                break;
            case MEIZUPUSH:
                pushClient = new EMMzPush();
                break;
            case HMSPUSH:
                pushClient = new EMHMSPush();
                break;
            case HONORPUSH:
                pushClient = new EMHonorPush();
                break;
            case NORMAL:
            default:
                pushClient = new EMNormalPush();
                break;
        }

        this.isClientInit = true;
        if(hasReceiveToken) {
            return;
        }
        pushClient.register(context, pushConfig, pushListener);
    }

    private void register(@NonNull PushType pushType) {
        register(pushType, false);
    }

    private void uploadToken(PushType type, String token) {
        Message message = handler.obtainMessage(MSG_WHAT_UPLOAD, token);
        Bundle bundle = new Bundle();
        bundle.putString("type", type.getName());
        message.setData(bundle);
        handler.sendMessage(message);
    }

    private void uploadTokenByTimes(PushType type, String token) {
        synchronized (bindLock) { // 确保把tasks放入queue时没有task正在执行。以避免task执行完后把刚放进的tasks清空。
            // Cancel all previous upload tasks first.
            handler.removeMessages(MSG_WHAT_REGISTER);
            // Check push type before upload
            register(type, true);
            for (int i = -1; i < TIMES_RETRY; i++) {
                Message msg = handler.obtainMessage(MSG_WHAT_REGISTER, token);
                if (i == -1) {
                    handler.sendMessage(msg);
                } else {
                    int delaySeconds = randomDelay(i);
                    EMLog.i(TAG, "Retry upload after " + delaySeconds + "s if failed.");
                    handler.sendMessageDelayed(msg, delaySeconds * 1000);
                }
            }
        }
    }

    private void deleteToken() {
        handler.obtainMessage(MSG_WHAT_UNREGISTER).sendToTarget();
    }

    private boolean uploadTokenInternal(String notifierName, String token) {
        if(TextUtils.isEmpty(notifierName)) {
            EMLog.e(TAG, "uploadTokenInternal notifierName is null, return. current push type: " + pushType);
            return false;
        }
        String remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, false) + "/users/"
                + ChatClient.getInstance().getCurrentUser() + "/push/binding";
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(ChatClient.getInstance().getContext());
        JSONObject json = new JSONObject();
        try {
            json.put("device_token", token);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal put json exception: " + e.toString());
        }
        int retry_times = 2;
        do {
            try {
                EMLog.e(TAG, "uploadTokenInternal, token=" + token + ", url=" + remoteUrl
                        + ", notifier name=" + notifierName);

                Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                        json.toString(), EMHttpClient.PUT);
                int statusCode = response.first;
                String content = response.second;

                if (statusCode == HttpURLConnection.HTTP_OK) {
                    EMLog.e(TAG, "uploadTokenInternal success.");
                    PushHelper.getInstance().setPushTokenWithType(pushType, token);
                    return true;
                }

                EMLog.e(TAG, "uploadTokenInternal failed: " + content);
                remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + ChatClient.getInstance().getCurrentUser();
            } catch (Exception e) {
                EMLog.e(TAG, "uploadTokenInternal exception: " + e.toString());
                remoteUrl = ChatClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + ChatClient.getInstance().getCurrentUser();
            }
        } while (--retry_times > 0);

        return false;
    }

    private PushType getPreferPushType(PushConfig pushConfig) {
        PushType[] supportedPushTypes = new PushType[]{
                PushType.FCM,
                PushType.MIPUSH,
                PushType.HMSPUSH,
                PushType.MEIZUPUSH,
                PushType.OPPOPUSH,
                PushType.VIVOPUSH,
                PushType.HONORPUSH,
        };

        ArrayList<PushType> enabledPushTypes = pushConfig.getEnabledPushTypes();

        for (PushType pushType : supportedPushTypes) {
            if (enabledPushTypes.contains(pushType) && isSupportPush(pushType, pushConfig)) {
                return pushType;
            }
        }

        return PushType.NORMAL;
    }

    private boolean isSupportPush(PushType pushType, PushConfig pushConfig) {
        boolean support;
        if (pushListener != null) {
            support = pushListener.isSupportPush(pushType, pushConfig);
        } else {
            support = PushUtil.isSupportPush(pushType, pushConfig);
        }
        EMLog.i(TAG, "isSupportPush: " + pushType + " - " + support);
        return support;
    }

    public int randomDelay(int attempts) {
        if (attempts == 0) {
            return new Random().nextInt(5) + 1;
        }

        if (attempts == 1) {
            return new Random().nextInt(54) + 6;
        }

        return new Random().nextInt(540) + 60;
    }

    private static class InstanceHolder {
        static PushHelper INSTANCE = new PushHelper();
    }
}
