package io.agora.notification;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.support.annotation.IdRes;
import android.support.v4.app.NotificationCompat;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.widget.RemoteViews;

import java.lang.ref.SoftReference;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Date;

public class EMNotificationBuilder {

    private Context appContext;
    private boolean sound;
    private boolean vibrate;
    private String channelId;
    private String channelName;
    private int channelLevel;
    private boolean autoCancel = true;
    private String title;
    private String content;
    private SoftReference<Bitmap> iconBitmapRef;
    private EMNotificationDefaultStyle style = new EMNotificationDefaultStyle();
    private PendingIntent pendingIntent;
    private int smallIcon = android.R.drawable.sym_def_app_icon;
    private String ticker;
    private PendingIntent fullScreenIntent;
    private int badgeNum;


    public EMNotificationBuilder(Context context) {
        if (context == null) {
            throw new IllegalArgumentException("context cannot is null");
        }
        appContext = context.getApplicationContext();
    }

    public EMNotificationBuilder setTitle(String title) {
        this.title = title;
        return this;
    }

    public EMNotificationBuilder setContent(String content) {
        this.content = content;
        return this;
    }

    public EMNotificationBuilder setIcon(@IdRes int res) {
        Bitmap tempBitmap = BitmapFactory.decodeResource(appContext.getResources(), res);
        if (tempBitmap != null) {
            iconBitmapRef = new SoftReference<>(tempBitmap);
        }
        return this;
    }

    public EMNotificationBuilder setIcon(Bitmap iconBitmap) {
        if (iconBitmap != null) {
            iconBitmapRef = new SoftReference<>(iconBitmap);
        }
        return this;
    }

    public EMNotificationBuilder setSmallIcon(@IdRes int smallIcon) {
        this.smallIcon = smallIcon;
        return this;
    }

    public EMNotificationBuilder setStyle(EMNotificationDefaultStyle style) {
        this.style = style;
        return this;
    }

    public EMNotificationBuilder setPendingIntent(PendingIntent pendingIntent) {
        this.pendingIntent = pendingIntent;
        return this;
    }

    public EMNotificationBuilder setFullScreenIntent(PendingIntent fullScreenIntent) {
        this.fullScreenIntent = fullScreenIntent;
        return this;
    }

    public EMNotificationBuilder setSound(boolean enable) {
        this.sound = enable;
        return this;
    }

    public EMNotificationBuilder setVibrate(boolean enable) {
        this.vibrate = enable;
        return this;
    }

    public EMNotificationBuilder setLevel(int level) {
        this.channelLevel = level;
        return this;
    }

    public EMNotificationBuilder setTicker(String ticker) {
        this.ticker = ticker;
        return this;
    }

    public EMNotificationBuilder setChannelId(String channelId) {
        if (!TextUtils.isEmpty(channelId)) {
            this.channelId = channelId;
        }
        return this;
    }

    public EMNotificationBuilder setChannelName(String channelName) {
        if (!TextUtils.isEmpty(channelName)) {
            this.channelName = channelName;
        }
        return this;
    }

    public EMNotificationBuilder setAutoCancel(boolean autoCancel) {
        this.autoCancel = autoCancel;
        return this;
    }

    public EMNotificationBuilder setBadgeNum(int badgeNum){
        this.badgeNum = badgeNum;
        return this;
    }

    public Notification build() {

        if (channelId == null) {
            channelId = "chat";
        }

        if (channelName == null) {
            channelName = "消息";
        }

        createNotificationChannel(appContext, vibrate, channelId, channelName, channelLevel);

        int defaults = 0;

        if (sound) {
            defaults = Notification.DEFAULT_SOUND;
        }
        if (vibrate) {
            defaults = Notification.DEFAULT_VIBRATE;
        }

        if (sound && vibrate) {
            defaults = Notification.DEFAULT_ALL;
        }

        NotificationCompat.Builder builder = new NotificationCompat.Builder(appContext);

        try {
            Method method = NotificationCompat.Builder.class.getDeclaredMethod("setChannelId", String.class
            );
            method.setAccessible(true);
            method.invoke(builder, channelId);
        }catch (Exception ignored){

        }

        // 设置小图标
        builder.setSmallIcon(smallIcon);

        if (!sound) {
            builder.setSound(null);
        }

        if (defaults != 0) {
            builder.setDefaults(defaults);
        }

        if (iconBitmapRef != null && iconBitmapRef.get() != null) {
            builder.setLargeIcon(iconBitmapRef.get());
        }

        if(badgeNum > 0){
            builder.setNumber(badgeNum);
        }

        builder.setTicker(ticker)
                .setPriority(channelLevel - 3)
                .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                // 点击后自动清除
                .setAutoCancel(autoCancel)
                // 设置通知标题
                .setContentTitle(title)
                // 设置通知内容
                .setContentText(content)
                // 设置通知的动作
                .setContentIntent(pendingIntent)
                // 设置通知时间，默认为系统发出通知的时间
                .setWhen(System.currentTimeMillis());

        if(style instanceof  EMNotificationBigTextStyle){
            String bigTxt = ((EMNotificationBigTextStyle) style).getBigTxt();
            builder.setStyle(new NotificationCompat.BigTextStyle().bigText(bigTxt));
        }else if (style instanceof  EMNotificationBigPicStyle){
            Bitmap bigPic = ((EMNotificationBigPicStyle) style).getBigPic();
            builder.setStyle(new NotificationCompat.BigPictureStyle().bigPicture(bigPic));
        }

        if (fullScreenIntent != null) {
            builder.setFullScreenIntent(fullScreenIntent, true);
        }

        if (vibrate) {
            // 添加自定义震动提醒
            // 延迟100ms后震动300ms, 再延迟400ms后震动500ms
            long[] pattern = new long[]{100, 200, 300, 400, 500};
            builder.setVibrate(pattern);
        }else{
            builder.setVibrate(null);
        }

        Notification notification = builder.build();
        // 锁屏时显示通知
        builder.setPublicVersion(notification);
        return notification;
    }


    /**
     * 创建通知渠道
     */
    private void createNotificationChannel(Context context, boolean vibrate, String channelId, String channelName, int channelLevel) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            int importance = NotificationManager.IMPORTANCE_DEFAULT;
            if (channelLevel >= 0 && channelLevel <= 5) {
                importance = channelLevel;
            }
            NotificationChannel channel = new NotificationChannel(channelId, channelName, importance);
            if(!sound){
                channel.setSound(null, null);
            }
            if (vibrate) {
                channel.enableVibration(true);
                channel.setVibrationPattern(new long[]{100, 200, 300, 400, 500});
            }else{
                channel.enableVibration(false);
            }
            NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            notificationManager.createNotificationChannel(channel);
        }
    }

    public static class EMNotificationDefaultStyle {
    }

    public static class EMNotificationBigTextStyle extends EMNotificationDefaultStyle {
        private String bigTxt;

        public String getBigTxt() {
            return bigTxt;
        }

        public EMNotificationBigTextStyle setBigTxt(String bigTxt) {
            this.bigTxt = bigTxt;
            return this;
        }
    }

    public static class EMNotificationBigPicStyle extends EMNotificationDefaultStyle {
        private SoftReference<Bitmap> bitmapRef;

        public Bitmap getBigPic() {
            if (bitmapRef != null && bitmapRef.get() != null) {
                return bitmapRef.get();
            }
            return null;
        }

        public EMNotificationBigPicStyle setBigPic(Bitmap bigPic) {
            if (bigPic != null) {
                bitmapRef = new SoftReference<>(bigPic);
            }
            return this;
        }
    }
}
