package io.agora.chat;


import io.agora.CallBack;
import io.agora.PresenceListener;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMAPresence;
import io.agora.chat.adapter.EMAPresenceManager;
import io.agora.chat.adapter.EMAPresenceManagerListener;
import io.agora.util.EMLog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * \~english
 * The Manager that defines how to manage presence states.
 */
public class PresenceManager {
    private final String TAG = getClass().getSimpleName();

    EMAPresenceManager emaObject;
    private List<PresenceListener> listeners = new CopyOnWriteArrayList<PresenceListener>();
    private EMAPresenceManagerListener listenerImpl = new EMAPresenceManagerListener() {
        @Override
        public void onPresenceUpdated(List<EMAPresence> presences) {

            synchronized (listeners) {
                for (PresenceListener listener : listeners) {
                    try {
                        listener.onPresenceUpdated(convertToPresence(presences));
                    } catch (Exception e) {
                        e.printStackTrace();
                        EMLog.e(TAG, "PresenceManager->listenerImpl:onPresenceUpdated() Error:" + e.getMessage());
                    }
                }

            }
        }
    };

    public PresenceManager(EMAPresenceManager presenceManager) {
        emaObject = presenceManager;
        emaObject.addListener(listenerImpl);
    }

    /**
     * \~english
     * Publishes a custom presence state.
     * @param ext            The extension information of the presence state. It can be set as nil.
     * @param callBack       The completion block, which contains the error message if this method fails.
     */
    public void publishPresence( String ext, CallBack callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                emaObject.publishPresence( ext, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess();
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~english
     * Subscribes to a user's presence states. If the subscription succeeds, the subscriber will receive the callback when the user's presence state changes.
     * @param members  The array of IDs of users whose presence states you want to subscribe to.
     * @param expiry   The expiration time of the presence subscription.
     * @param callBack The completion block, which contains the error message if the method fails.
     */
    public void subscribePresences(List<String> members, long expiry, ValueCallBack<List<Presence>> callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<EMAPresence> presences = emaObject.subscribePresences(members, expiry, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(convertToPresence(presences));
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     *\~english
     * Unsubscribes from a user's presence states.
     * @param members  The array of IDs of users whose presence states you want to unsubscribe from.
     * @param callBack The completion block, which contains the error message if the method fails.
     */
    public void unsubscribePresences(List<String> members, CallBack callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                emaObject.unsubscribePresences(members, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess();
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });

    }

    /**
     *\~english
     * Uses pagination to get a list of users whose presence states you have subscribed to.
     * @param pageNum  The current page number, starting from 0.
     * @param pageSize The number of subscribed users on each page.
     * @param callBack The completion block, which contains IDs of users whose presence states you have subscribed to. Returns nil if you subscribe to no user's presence state.
     */
    public void fetchSubscribedMembers(int pageNum, int pageSize, ValueCallBack<List<String>> callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<String> subscribedMembers = emaObject.fetchSubscribedMembers(pageNum, pageSize, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(subscribedMembers);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the current presence state of users.
     * @param members  The array of IDs of users whose current presence state you want to check.
     * @param callBack The completion block, which contains the users whose presence state you have subscribed to.
     */
    public void fetchPresenceStatus(List<String> members, ValueCallBack<List<Presence>> callBack) {
        ChatClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<EMAPresence> presenceStatus = emaObject.fetchPresenceStatus(members, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(convertToPresence(presenceStatus));
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~english
     * Adds a listener.
     * @param listener {@link PresenceListener} The listener to be added.
     */
    public void addListener(PresenceListener listener) {
        if (!listeners.contains(listener)) {
            listeners.add(listener);
        }
    }

    /**
     *\~english
     * Removes a listener.
     * @param listener {@link PresenceListener} The listener to be removed.
     */
    public void removeListener(PresenceListener listener) {
        if (listener != null) {
            listeners.remove(listener);
        }
    }

    /**
     * \~english
     * Removes all listeners.
     */
    public void clearListeners() {
        listeners.clear();
    }

    private List<Presence> convertToPresence(List<EMAPresence> presences) {
        List<Presence> presenceList = new ArrayList<>();
        for (int i = 0; i < presences.size(); i++) {
            presenceList.add(new Presence(presences.get(i)));
        }
        return presenceList;
    }
}
