/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAGroup;
import io.agora.chat.adapter.EMAGroupSetting;
import io.agora.chat.adapter.EMAMucShareFile;

import java.util.ArrayList;
import java.util.List;

/**
 *  \~english
 *  The group class.
 */
public class Group extends EMBase<EMAGroup> {

    /**
     * \~english
     * The enum of the group permission types.
     *
     */
    public enum GroupPermissionType {
        /**
         * \~english
         * The group member.
         */
        member(0),

        /**
         * \~english
         * The group admin.
         */
        admin(1),

        /**
         * \~english
         * The group owner.
         */
        owner(2),

        /**
         * \~english
         * The unknown type.
         */
        none(-1);
        private int permissionType;
        private GroupPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

    public Group(EMAGroup group) {
        emaObject = group;
    }

    /**
     * \~english
     * Gets the group ID.
     * @return The group ID.
     */
    public String getGroupId() {
        return emaObject.groupId();
    }

    /**
     * \~english
     * Gets the group name.
     * @return      The group name.
     */
    public String getGroupName() {
        return emaObject.groupSubject();
    }

    /**
     * \~english
     * Gets the group description.
     * @return  The group description.
     */
    public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~english
     * Gets whether the group is a public group. 
     * @return  If the result is `true`, the group is a public group, otherwise it is a private group.
     */
    public boolean isPublic() {

        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        switch (setting.style()) {
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE:
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE:
                return false;
            default:
                return true;
        }
    }

    /**
     * \~english
     * Gets whether the group member is allowed to invite other users to join the group.
     * @return - `true`: The group member can invite other users to join the group;
     *         - `false`: Do not allow the group member invite other users to join the group.
     */
    public boolean isMemberAllowToInvite()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE) {
            return true;
        }
        return false;
    }

    /**
     * \~english
     * Fetches the group property: whether users can auto join the group VS need requesting or invitation from a group member to join the group.
     * There are four types of group properties used to define the style of a group, and `isMemberOnly` contains three types including: PRIVATE_OWNER_INVITE,  PRIVATE_MEMBER_INVITE, PUBLIC_JOIN_APPROVAL. And do not include {@link GroupManager.GroupStyle#GroupStylePublicOpenJoin}.
     *
     * @return 
     * - `true`: Users can not join the group freely. Needs the invitation from the group owner or members, or the application been approved by the group owner or admins.
     * - `false`: Users can join freely without the group owner or member‘s invitation or the new joiner’s application been approved.
     */
    public boolean isMemberOnly()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
            return true;
        }
        return false;
    }

    /**
     * \~english
     * The max number of group members allowed in a group. The param is set when the group is created.
     * 
     * Be sure to fetch the detail specification of the group from the server first, see {@link GroupManager#getGroupFromServer(String)}. If not, the SDK returns 0.
     * 
     * @return  The allowed max number of group members.
     */
    public int getMaxUserCount() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return 0;
        }
        return setting.maxUserCount();
    }

    /**
     * \~english
     * Gets whether the group message is blocked.
     * 
     * Reference:
     * For blocking group message, see {@link GroupManager#blockGroupMessage(String)},
     * unblocking see {@link GroupManager#unblockGroupMessage(String)}
     * @return   - `true`: The user has blocked the group messages;
     *           - `false`: The user does not blocked the group messages.
     */
    public boolean isMsgBlocked() {
        return emaObject.isMsgBlocked();
    }

    /**
     * \~english
     * Gets the user ID of the group owner.
     * @return    The user ID of the group owner.
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     * \~english
     * The name of the group. 
     * Be sure to fetch the detail specification of the group from the server first, see {@link GroupManager#getGroupFromServer(String)}.
     * @return  The group name.
     */
    public String groupSubject() {
        return emaObject.groupSubject();
    }

    /**
     * \~english
     * Gets the member list of the group.
     * If no member is found from the server, the return might be empty.
     * 
     * Reference:
     * You can get the group members in the following ways:
     * (1) When member's number is less than 200, can use {@link GroupManager#getGroupFromServer(String, boolean)} to fetch,
     * the second parameter pass in true, can get up to 200 members
     * (2) Use the method of {@link GroupManager#fetchGroupMembers(String, String, int)} to get by page,
     * or its asynchronous method of {@link GroupManager#asyncFetchGroupMembers(String, String, int, ValueCallBack)}

     * to get
     * @return  The list of the user IDs of the group members.
     */
    public List<String> getMembers() {
        List<String> members = new ArrayList<String>();
        members.addAll(emaObject.getMembers());
        return members;
    }

    /**
     * \~english
     * Gets the member count of the group.
     * 
     * @return  The count.
     */
    public int getMemberCount() {
        return emaObject.getMemberCount();
    }

    /**
     * \~english
     * Gets the group name, or group ID if the group name is empty.
     * This method can be used for adapter sorting.
     * 
     * @return  The group name, or group ID if the group name is empty.
     */
    public String toString() {
        String str = getGroupName();
        return str != null ? str : getGroupId();
    }

    /**
     * \~english
     * Gets the admin list of the group.
     * 
     * Be sure to fetch the detail specification of the group from the server first, see {@link GroupManager#getGroupFromServer(String)}.
     * 
     * @return  The admin list of the group.
     */
    public List<String> getAdminList() {
        return emaObject.getAdminList();
    }

    /**
     * \~english
     * Gets the blocklist of the group.
     * 
     * If no blocklist is found from the server, the return may be empty.
     * 
     * Reference:
     * To fetch the blocklist, call {@link GroupManager#fetchGroupBlackList(String, int, int)}.
     * 
     * Only the group owner or admin can call this method.
     * 
     * You also can call it's asynchronous method as follows:
     * {@link GroupManager#asyncFetchGroupBlackList(String, int, int, ValueCallBack)}.
     * 
     * @return  The blocklist.
     */
    public List<String> getBlackList() {
        return emaObject.getGroupBans();
    }

    /**
     * \~english
     * Gets the mute list of the group.
     * 
     * If no mute list is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also fetch the mute list by calling {@link GroupManager#fetchGroupMuteList(String, int, int)}.
     * And only the group owner or admin can call this method. 
     * 
     * You can also call it's asynchronous method, see 
     * {@link GroupManager#asyncFetchGroupMuteList(String, int, int, ValueCallBack)}.
     *
     * @return  The mute list of the group.
     */
    public List<String> getMuteList() {
        return emaObject.getGroupMuteList();
    }

    /**
     * \~english
     * Gets the allowlist of the group.
     * 
     * If no allowlist is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also fetch the allowlist by calling {@link GroupManager#fetchGroupWhiteList(String, ValueCallBack)}.
     * And only the group owner or admin can call this method.
     * @return  The group allowlist.
     */
    public List<String> getWhiteList() { return emaObject.getWhiteList(); }

    /**
     * \~english
     * Gets whether all members are muted.
     * 
     * This method has limitations and is recommended to be used with caution.
     * 
     * The state is updated when a all-muted/all-unmuted callback is received, but only for the in-memory object.
     * After the in-memory object is collected and pulled again from the database or server, the state becomes unreliable.
     * @return  Whether all members are muted.
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }

    /**
     * \~english
     * Gets the customized extension of the group.
     * @return  The customized extension of the group.
     */
    public String getExtension() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return "";
        }
        return setting.extension();
    }

    /**
     * \~english
     * Gets the group announcement.
     * If no announcement is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also call the method: {@link GroupManager#fetchGroupAnnouncement(String)} to get the group announcement.
     * As for its asynchronous method, see {@link GroupManager#asyncFetchGroupAnnouncement(String, ValueCallBack)}.
     * @return  The content of the group announcement.
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     *  \~english
     *  The list of the shared file.
     */
    private List<MucSharedFile> shareFileList;

    /**
     * \~english
     * Gets the group shared file list.
     * If no shared file is found from the server, the return might be empty.
     * 
     * Reference:
     * You can also call {@link GroupManager#fetchGroupSharedFileList(String, int, int)} to fetch the shared file list.
     * And for the asynchronous method, see {@link GroupManager#asyncFetchGroupSharedFileList(String, int, int, ValueCallBack)}.
     * @return  The group shared file list.
     */
    public List<MucSharedFile> getShareFileList()
    {
        if (shareFileList == null) {
            shareFileList = new ArrayList<MucSharedFile>();
        }
        else {
            shareFileList.clear();
        }
        List<EMAMucShareFile> afiles = emaObject.getShareFiles();
        for (EMAMucShareFile afile : afiles){
            shareFileList.add(new MucSharedFile(afile));
        }
        return shareFileList;
    }

    /**
     * \~english
     * Gets the current user's role in group.
     * @return   Returns the permission type of the user in a group, see `GroupPermissionType`.
     */
    public GroupPermissionType getGroupPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == GroupPermissionType.member.permissionType) {
            return GroupPermissionType.member;
        }else if(permissionType == GroupPermissionType.admin.permissionType) {
            return GroupPermissionType.admin;
        }else if(permissionType == GroupPermissionType.owner.permissionType) {
            return GroupPermissionType.owner;
        }else {
            return GroupPermissionType.none;
        }
    }

    /**
     * \~english
     * Gets whether the group is disabled.
     *
     * @return  - `true`: Group has been disabled;
     *          - `false`: Group is not disabled.
     */
    public boolean isDisabled() { return emaObject.isDisabled(); }

}
