/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import android.net.Uri;

import io.agora.chat.adapter.message.EMAFileMessageBody;
import io.agora.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import io.agora.chat.adapter.message.EMAImageMessageBody;
import io.agora.chat.adapter.message.EMAMessageBody;
import io.agora.chat.adapter.message.EMAVideoMessageBody;
import io.agora.chat.adapter.message.EMAVoiceMessageBody;
import io.agora.util.FileHelper;

import java.io.File;

/**
 * \~english
 * The base class of file type message.
 */
public abstract class FileMessageBody extends MessageBody {

	/**
	 * \~english
	 * The download status of the attachment file .
	 */
	public enum EMDownloadStatus {
        DOWNLOADING,/** \~chinese 正在下载文件消息。  \~english The SDK is downloading the file message. */
        SUCCESSED,/** \~chinese 下载文件消息成功。 \~english The SDK successfully downloads the file message. */
        FAILED,/** \~chinese 下载文件消息失败。  \~english The SDK fails to download the file message.*/
        PENDING /** \~chinese 文件消息下载正在等待中。  \~english File message download is pending.*/
    }

	/**
	 * \~english
	 * The constructor method of the file message body.
	 * @param localPath	The absolute path to the attachment file or the path to the URI as a string.
	 */
	public FileMessageBody(String localPath) {
    	this(FileHelper.getInstance().formatInUri(localPath));
	}

	/**
	 * \~english
	 * The constructor method of the file message body.
	 * @param localPath  The localPath URI of the attachment file.
	 */
	public FileMessageBody(Uri localPath) {
		this(localPath, EMAMessageBody.EMAMessageBodyType_FILE);
	}

	/**
	 * @param localPath The localPath URI of the attachment file.
	 * @param type  The file type.
	 */
	FileMessageBody(String localPath, int type) {
    	this(Uri.fromFile(new File(localPath)), type);
	}

	FileMessageBody(Uri localPath, int type) {
		switch (type) {
		case EMAMessageBody.EMAMessageBodyType_FILE:
			emaObject = new EMAFileMessageBody(localPath, type);
			setFileLength(FileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_IMAGE:
			emaObject = new EMAImageMessageBody(localPath, null);
			setFileLength(FileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VIDEO:
			emaObject = new EMAVideoMessageBody(localPath, null);
			setFileLength(FileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VOICE:
		    emaObject = new EMAVoiceMessageBody(localPath, 0);
			setFileLength(FileHelper.getInstance().getFileLength(localPath));
		    break;
		default:
			break;
		}
	}

	public FileMessageBody(EMAFileMessageBody body) {
		emaObject = body;
	}
	

	/**
	 * \~english
	 * Gets the file name.
	 * @return  The file name.
	 */
	public String getFileName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~english
	 * Sets the file name.
	 * @param fileName  The file name.
	 */
	public void setFileName(String fileName) {
		((EMAFileMessageBody)emaObject).setDisplayName(fileName);
	}

	/**
	 * \~english
	 * The local path or the string in the form of URI of the file.
	 *
	 * @return The local file path or the string form of URI.
	 */
	public String getLocalUrl() {
		return FileHelper.getInstance().formatOutLocalUrl(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~english
	 * Sets the local path of the file. 
	 * @param localUrl	The absolute path to the file or the string form of URI.
	 */
	public void setLocalUrl(String localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(FileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~english
	 * Gets the URI of the local image, voice and so on.
	 * 
	 * @return  The URI of the local image, voice and so on.
	 */
	public Uri getLocalUri() {
		return FileHelper.getInstance().formatOutUri(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~english
	 * Sets the URI of the local file.
	 * @param localUrl	The URI of file.
	 */
	public void setLocalUrl(Uri localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(FileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~english
	 * Gets the path of the attachment file in the server.
	 * @return  The path of the attachment file in the server.
	 */
	public String getRemoteUrl() {
		return ((EMAFileMessageBody)emaObject).getRemoteUrl();
	}

	/**
	 * \~english
	 * Sets the path of the attachment file in the server.
	 * @param remoteUrl  The path of the attachment file in the server.
	 */
	public void setRemoteUrl(String remoteUrl) {
		((EMAFileMessageBody)emaObject).setRemotePath(remoteUrl);
	}

	/**
	 * \~english
	 * Sets the file's token, this method is typically invoked by the SDK.
	 * @param secret The file's token.
	 */
	public void setSecret(String secret) {
		((EMAFileMessageBody)emaObject).setSecretKey(secret);
	}

	/**
	 * \~english
	 * Gets the token for downloading the message attachment.
	 * @return  The file token generated by the server。
	 */
	public String getSecret() {
		return ((EMAFileMessageBody)emaObject).getSecret();
	}

	/**
	 * \~english
	 * Gets the filename.
	 * @return The filename.
	 */
	public String displayName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~english
	 * Sets the data length (bytes) of the attachment file.
	 * @param length  The data length (bytes) of the attachment file.
	 */
	public void setFileLength(long length) {
		((EMAFileMessageBody)emaObject).setFileLength(length);
	}

	/**
	 * \~english
	 * Gets the download status of the attachment file.
	 * @return The download status of the attachment file.
	 */
	public EMDownloadStatus downloadStatus() {
	    EMADownloadStatus _status = ((EMAFileMessageBody)emaObject).downloadStatus();
	    switch (_status) {
		/** \~chinese 正在下载文件消息。  \~english The SDK is downloading the file message. */
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
		/** \~chinese 下载文件消息成功。 \~english The SDK successfully downloads the file message. */
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
		/** \~chinese 下载文件消息失败。  \~english The SDK fails to download the file message.*/
	    case FAILED:
	        return EMDownloadStatus.FAILED;
		/** \~chinese 文件消息下载正在等待中。  \~english File message download is pending.*/
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }
		/** \~chinese 该状态已废弃。请改用 {@link EMDownloadStatusSucceed}。  \~english This state is deprecated. Use {@link EMDownloadStatusSucceed} instead.*/


	    return EMDownloadStatus.SUCCESSED;
	}

	/**
	 * \~english
	 * Sets the download status of the attachment file.
	 * @param status	The downloading status of the attachment file.
	 */
	public void setDownloadStatus(EMDownloadStatus status) {
		((EMAFileMessageBody)emaObject).setDownloadStatus(EMADownloadStatus.valueOf(status.name()));
	}

}
