package io.agora.chat;


import io.agora.chat.adapter.EMAConversation;
import io.agora.chat.adapter.EMAFetchMessageOption;
import io.agora.util.EMLog;

import java.util.List;

/**
 * \~english
 * The parameter configuration class for pulling historical messages from the server.
 */
public class FetchMessageOption extends EMBase<EMAFetchMessageOption> {

    /**
     * \~english
     * The constructor.
     */
    public FetchMessageOption() {
        emaObject = new EMAFetchMessageOption();
    }

    /**
     * \~english
     * Sets whether the retrieved messages are saved to the database.
     *
     * @param save Whether to save the retrieved messages to the database:
     *             - `true`: Yes.
     *             - (Default)`false`: No.
     */
    public void setIsSave(boolean save) {
        emaObject.setSave(save);
    }

    /**
     * \~english
     * Sets the message search direction.
     *
     * @param direction  The message search direction. See {@link Conversation.SearchDirection}.
     *                   - (Default) `UP`: Messages are retrieved in the descending order of the Unix timestamp included in them.
     * 					 - `DOWN`: Messages are retrieved in the ascending order of the Unix timestamp included in them.
     *
     */
    public void setDirection(Conversation.SearchDirection direction) {
        EMAConversation.EMASearchDirection d = direction == Conversation.SearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;
        emaObject.setDirection(d);
    }

    /**
     * \~english
     * Sets the message sender in the group conversation.
     *
     * @param from The user ID of the message sender in the group conversation.
     */
    public void setFrom(String from) {
        emaObject.setFrom(from);
    }

    /**
     *  \~english
     *  Sets the types of message for query. 
     * 
     * @param msgTypes The list of message types for query. See {@link ChatStatisticsManager.SearchMessageType}.
     *                 You can set several message types. The default value is `null`, indicating that all types of messages are retrieved.
     */
    public void setMsgTypes(List<ChatMessage.Type> msgTypes) {
        if(msgTypes!=null) {
            for (Object type : msgTypes) {
                if (!(type instanceof ChatMessage.Type)) {
                    EMLog.e(getClass().getSimpleName(),"setMsgTypes-->"+"Invalid argument type:" + type.getClass().getName());
                    throw new IllegalArgumentException("Invalid argument type: " + type.getClass().getName());
                }
            }
            emaObject.setMsgTypes(msgTypes);
        }
    }

    /**
     *  \~english
     * Sets the start time for message query. 
     *
     * @param startTime  The start time for message query. The time is a UNIX time stamp in milliseconds. The default value is `-1`, indicating that this parameter is ignored during message query. 
     *                   - If the start time is set to a specific time spot and the end time uses the default value `-1`, the SDK returns messages that are sent and received in the period that is from the start time to the current time.
     *                   - If the start time uses the default value `-1` and the end time is set to a specific time spot, the SDK returns messages that are sent and received in the period that is from the timestamp of the first message to the current time.
     */
    public void setStartTime(long startTime) {
        emaObject.setStartTime(startTime);
    }

    /**
     *  \~english
     * Sets the end time for message query. 
     *
     * @param endTime  The end time for message query. The time is a UNIX time stamp in milliseconds. The default value is `-1`, indicating that this parameter is ignored during message query. 
     *                  - If the start time is set to a specific time spot and the end time uses the default value `-1`, the SDK returns messages that are sent and received in the period that is from the start time to the current time.
     *                  - If the start time uses the default value `-1` and the end time is set to a specific time spot, the SDK returns messages that are sent and received in the period that is from the timestamp of the first message to the current time.
     */
    public void setEndTime(long endTime) {
        emaObject.setEndTime(endTime);
    }
}
