/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import android.text.TextUtils;

import io.agora.CallBack;
import io.agora.chat.ChatMessage.Type;
import io.agora.chat.adapter.EMAConversation;
import io.agora.chat.adapter.EMAConversation.EMAConversationType;
import io.agora.chat.adapter.EMAConversation.EMASearchDirection;
import io.agora.chat.adapter.message.EMAMessage;
import io.agora.util.EMLog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * \~english
 * The conversation class, which defines one-to-one conversations, group conversations, and chat room conversations.
 *
 * Each type of conversation involves messages that are sent and received.
 *
 * The following code shows how to get the number of the unread messages from the conversation.
 * ```java
 *     // ConversationId can be the ID of the peer user, the group ID, or the chat room ID:
 *     Conversation conversation = ChatClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * ```
 *
*/

public class Conversation extends EMBase<EMAConversation> {

    private static final String TAG = "conversation";
    private static final int LIST_SIZE = 512;

    Conversation(EMAConversation conversation) {
        emaObject = conversation;
    }

    /**
     * \~english
     * The conversation types.
     */
    public enum ConversationType {
        /**
         *\~english
         * One-to-one chat.
         */
        Chat,

        /**
         *\~english
         * Group chat.
         */
        GroupChat,

        /**
         *\~english
         *Chat room.
         */
        ChatRoom,

        /**
         *\~english
         * Discussion group. Currently, this type of conversation is not available.

         */
        DiscussionGroup,

        /**
         * \~english
         * Help desk.
         */
        HelpDesk
    }

    /**
     *  \~english
     *  The message search direction.
     *
     * The message research is based on the Unix timestamp included in messages. Each message contains two Unix timestamps:
     * - The Unix timestamp when the message is created;
     * - The Unix timestamp when the message is received by the server.
     *
     * Which Unix timestamp is used for message search depends on the setting of {@link ChatOptions#setSortMessageByServerTime(boolean)}.
     *
     */
    public enum SearchDirection {
        /**
         *\~english
         * Messages are retrieved in the descending order of the Unix timestamp included in them.
         */
        UP,     

        /**
         *\~english
         * Messages are retrieved in the ascending order of the Unix timestamp included in them.
         */
        DOWN    
    }

    /**
     * \~english
     * The conversation ID, which depends on the conversation type.
     * - One-to-one chat/help desk: The conversation ID is the user ID of the peer user.
     * - Group chat: The conversation ID is the group ID.
     * - Chat room: The conversation ID is the chat room ID.
     *
     * @return The conversation ID.
     */
    public String conversationId() {
        return emaObject.conversationId();
    }

    /**
     * \~english
     * Gets the conversation type.

     * @return  The conversation type.
     */
    public ConversationType getType() {
        EMAConversationType t = emaObject._getType();
        if (t == EMAConversationType.CHAT) {
            return ConversationType.Chat;
        }
        if (t == EMAConversationType.GROUPCHAT) {
            return ConversationType.GroupChat;
        }
        if (t == EMAConversationType.CHATROOM) {
            return ConversationType.ChatRoom;
        }
        if (t == EMAConversationType.DISCUSSIONGROUP) {
            return ConversationType.DiscussionGroup;
        }
        if (t == EMAConversationType.HELPDESK) {
            return ConversationType.HelpDesk;
        }
        return ConversationType.Chat;
    }

    /**
     * \~english
     * Gets the number of unread messages in the conversation.
     *
     * @return The unread message count in the conversation.
     */
    public int getUnreadMsgCount() {
        return emaObject.unreadMessagesCount();
    }

    /**
     *  \~english
     *  Marks all unread messages as read.
     */
    public void markAllMessagesAsRead() {
        emaObject.markAllMessagesAsRead(true);
    }

    /**
     * \~english
     * Gets the number of all unread messages in the conversation in the local database.
     *
     * @return The number of all unread messages in the conversation.
     */
    public int getAllMsgCount() {
        return emaObject.messagesCount();
    }

    /**
     * \~english
     * Checks whether the current conversation is a thread conversation.
     *
     * @return Whether the conversation is a chat thread conversation.
     *         - `true`: Yes.
     *         - `false`: No.
     */
    public boolean isChatThread() {
        return emaObject.isChatThread();
    }

    /**
     * \~english
     * Loads messages from the local database.
     *
     * The loaded messages will be put in the conversation in the memory according to the timestamp in them and will be returned when you call {@link #getAllMessages()}.
     *
     * @param startMsgId    The starting message ID for query. The SDK loads messages, starting from the specified one, in the descending order of the timestamp included in them.
     *                      If this parameter is set as `null` or an empty string, the SDK retrieves messages, from the latest one, according to the descending order of the timestamp included in them.
     * @param pageSize      The number of messages that you expect to get on each page. The value range is [1,400].
     * @return              The list of retrieved messages (excluding the one with the starting ID).
     */
    public List<ChatMessage> loadMoreMsgFromDB(String startMsgId, int pageSize) {
        return loadMoreMsgFromDB(startMsgId, pageSize, SearchDirection.UP);
    }

    /**
     * \~english
     * Loads the messages from the local database, starting from a specific message ID.
     *
     * The loaded messages will be put in the conversation in the memory according to the timestamp included in them.
     *
     * @param startMsgId    The starting message ID for query. After this parameter is set, the SDK retrieves messages, from the specified one, according to the message search direction.
     *                      If this parameter is set as `nil`, the SDK retrieves messages according to the search direction while ignoring this parameter.
 *                          - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the descending order of the timestamp included in them.
 *                          - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the ascending order of the timestamp included in them.
     * @param pageSize      The number of messages that you expect to get on each page. The value range is [1,400].
     * @param direction     The message search direction. See {@link SearchDirection}.
	 *                       - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					     - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.

     * @return              The list of retrieved messages (excluding the one with the starting ID).
     */
    public List<ChatMessage> loadMoreMsgFromDB(String startMsgId, int pageSize, SearchDirection direction) {
        EMASearchDirection d = direction == SearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;
        List<EMAMessage> msgs = emaObject.loadMoreMessages(startMsgId, pageSize, d);
        List<ChatMessage> result = new ArrayList<ChatMessage>();
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        getCache().addMessages(result);
        return result;
    }

    /**
     * \~english
     * Retrieves messages in the local database based on the Unix timestamp included in them.
     *
     * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
     * @param maxCount   The maximum number of message to retrieve each time. The value range is [1,400].
     * @param direction  The message search direction. See {@link SearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     *
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     *
     */
    public List<ChatMessage> searchMsgFromDB(long timeStamp, int maxCount, SearchDirection direction) {
        EMASearchDirection d = direction == SearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(timeStamp, maxCount, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<ChatMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<ChatMessage>();
        } else {
            result = new ArrayList<ChatMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~english
     * Gets messages of certain types that a specified user sends in the conversation.
     *
     * @param type       The message type. See {@link Type}.
     * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If this parameter is set as a negative value, the SDK retrieves from the latest message.
     * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link SearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<ChatMessage> searchMsgFromDB(ChatMessage.Type type, long timeStamp, int maxCount, String from, SearchDirection direction) {
        EMASearchDirection d = direction == SearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<ChatMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<ChatMessage>();
        } else {
            result = new ArrayList<ChatMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~english
     * Gets messages with keywords that the specified user sends in the conversation.
     *
     * @param keywords   The keywords for query.
     * @param timeStamp  The starting Unix timestamp for search. The unit is millisecond.
     *                   If this parameter is set as a negative value, the SDK retrieves from the current time.
     * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link SearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<ChatMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, SearchDirection direction) {
        EMASearchDirection d = direction == SearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<ChatMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<ChatMessage>();
        } else {
            result = new ArrayList<ChatMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~english
     * Gets a certain quantity of messages sent or received in a certain period from the local database.
     *
     * @param startTimeStamp    The starting Unix timestamp in the message for query. The unit is millisecond. See {@link ChatOptions#setSortMessageByServerTime}.
     * @param endTimeStamp      The ending Unix timestamp in the message for query. The unit is millisecond. See {@link ChatOptions#setSortMessageByServerTime}.
     * @param maxCount          The maximum number of messages to retrieve each time. The value range is [1,400].
     * @return                  The list of retrieved messages (excluding the ones with the starting timestamp and ending timestamp).
     */
    public List<ChatMessage> searchMsgFromDB(long startTimeStamp, long endTimeStamp, int maxCount) {
        List<EMAMessage> msgs = emaObject.searchMessages(startTimeStamp, endTimeStamp, maxCount);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<ChatMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<ChatMessage>();
        } else {
            result = new ArrayList<ChatMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~english
     * Gets custom messages with keywords that the specified user sends in the conversation.
     *
     * Note: Be cautious about memory usage when the maxCount is large.
     *
     * @param keywords   The keywords for search.
     * @param timeStamp  TThe starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If this parameter is set as a negative value, the SDK retrieves from the current time.
     * @param maxCount   The maximum number of messages to retrieve. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link SearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<ChatMessage> searchCustomMsgFromDB(String keywords, long timeStamp, int maxCount, String from, SearchDirection direction) {
        EMASearchDirection d = direction == SearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchCustomMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<ChatMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<ChatMessage>();
        } else {
            result = new ArrayList<ChatMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new ChatMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~english
     * Gets the message with message ID.
     *
     * The SDK first retrieves the message from the memory. If no message is found, the SDK will retrieve it from the local database and load it.
     *
     * @param  messageId     The message ID.
     * @param  markAsRead    Whether to mark the retrieved message as read.
     *                       - `true`: Yes.
     *                       - `false`: No.
     * @return message       The retrieved message instance.
     */
    public ChatMessage getMessage(String messageId, boolean markAsRead) {
        EMLog.d(TAG, "getMessage messageId: "+messageId+" markAsRead: "+markAsRead);
        ChatMessage msg = getCache().getMessage(messageId);
        if (msg == null) {
            EMAMessage emaMsg = emaObject.loadMessage(messageId);
            if (emaMsg == null) {
                return null;
            }
            msg = new ChatMessage(emaMsg);
        }
        emaObject.markMessageAsRead(messageId, markAsRead);
        return msg;
    }

    /**
     * \~english
     * Marks a message as read.
     *
     * You can also call {@link ChatMessage#setUnread(boolean)} to mark the message as read.
     *
     * @param messageId The message ID.
     */
    public void markMessageAsRead(String messageId) {
        emaObject.markMessageAsRead(messageId, true);
    }

    /**
     * \~english
     * Gets all messages of the conversation in the local memory.
     *
     * If no message is found in the memory, the SDK will load the latest message from the local database.
     *
     * @return The message list.
     */
    public List<ChatMessage> getAllMessages() {
        if (getCache().isEmpty()) {
            EMAMessage lastMsg = emaObject.latestMessage();
            List<ChatMessage> msgs = new ArrayList<ChatMessage>();
            if (lastMsg != null) {
                msgs.add(new ChatMessage(lastMsg));
            }
            getCache().addMessages(msgs);
        }
        return getCache().getAllMessages();
    }

    /**
     * \~english
     * Deletes a message in the local database.
     *
     *
     * @param messageId    The ID of the message to delete.
     */
    public void removeMessage(String messageId) {
        EMLog.d(TAG, "remove msg from conversation: " + messageId);
        emaObject._removeMessage(messageId);
        getCache().removeMessage(messageId);
    }

    /**
     * \~english
     * Gets the latest message in the conversation.
     *
     * A call to this method has no impact on the number of unread messages.
     *
     * The SDK first retrieves the message from the memory. If no message is found, the SDK will retrieve it from the local database and load it.
     *
     * @return  The message instance.
     */
    public ChatMessage getLastMessage() {
        if (getCache().isEmpty()) {
            EMAMessage _msg = emaObject.latestMessage();
            ChatMessage msg = _msg == null ? null : new ChatMessage(_msg);
            getCache().addMessage(msg);
            return msg;
        } else {
            return getCache().getLastMessage();
        }
    }

    /**
     * \~english
     * Gets the latest received message in the conversation.
     *
     * @return  The message instance.
     */
    public ChatMessage getLatestMessageFromOthers() {
        EMAMessage _msg = emaObject.latestMessageFromOthers();
        ChatMessage msg = _msg == null ? null : new ChatMessage(_msg);
        getCache().addMessage(msg);
        return msg;
    }

    /**
     * \~english
     * Deletes all the messages in the conversation.
     *
     * This method only deletes all the messages in a conversation in the memory, but not the messages in the local database.
     *
     * When exiting a conversation, you need to clear the memory to reduce the memory consumption.
     */
    public void clear() {
        getCache().clear();
    }

    /**
     * \~english
     * Deletes all the messages in the conversation from the memory and local database.
     */
    public void clearAllMessages() {
        emaObject.clearAllMessages();
        getCache().clear();
    }

    /**
     * \~english
     * Sets the extension field of the conversation.
     *
     * The extension field is only stored in the local database, but not synchronized to the server.
     *
     * The extension field of a chat thread conversation is not saved in the local database.
     *
     * @param ext       The extension field of the conversation.
     */
    public void setExtField(String ext) {
        if(!isChatThread()) {
            emaObject._setExtField(ext);
        }
    }

    /**
     * \~english
     * Gets the extension field of the conversation.
     *
     * The extension field is only stored in the local database, but not synchronized to the server.
     *
     * The extension field of a chat thread conversation is not saved in the local database.
     *
     * @return The extension content.
     */
    public String getExtField() {
        return emaObject.extField();
    }

    /**
     * \~english
     * Checks whether the conversation is pinned.
     *
     * @return Whether the conversation is pinned.
     *         - `true`: Yes.
     *         - `false`: No. The conversation is unpinned.
     */
    public boolean isPinned() {
        return emaObject.isPinned();
    }

    /**
     * \~english
     * Gets the time when the conversation is pinned.
     * 
     * @return The UNIX timestamp when the conversation is pinned. The unit is millisecond. This value is `0` when the conversation is not pinned.
     */
    public long getPinnedTime() {
        return emaObject.getPinnedTime();
    }

    /**
     * \~english
     * Converts a message type to a conversation type.
     *
     * @param       id    The message ID.
     * @param       type  The message type.
     * @return      The conversation type.
     */
    public static ConversationType msgType2ConversationType(String id, ChatMessage.ChatType type) {
        switch (type) {
            case GroupChat:
                return ConversationType.GroupChat;
            case ChatRoom:
                return ConversationType.ChatRoom;
            default:
                return ConversationType.Chat;
        }
    }

    /**
     * \~english
     * Checks whether it is a group conversation or a chat room conversation, or a conversation of another type.
     *
     * @return  Whether it is a group conversation or a chat room conversation, or a conversation of another type.
     *          - `true`: It is a group conversation or a chat room conversation.
     *          - `false`: It is not a group conversation or a chat room conversation.
     */
    public boolean isGroup() {
        ConversationType type = getType();
        // 需要排除Thread会话
        return !isChatThread() && (type == ConversationType.GroupChat ||
                type == ConversationType.ChatRoom);
    }

    /**
     *  \~english
     * Inserts a message to the conversation in the local database。
     *
     * To insert the message correctly, ensure that the conversation ID of the message is the same as that of the conversation.
     *
     * The message will be inserted based on the Unix timestamp included in it. Upon message insertion, the SDK will automatically update attributes of the conversation, including `latestMessage`.
     *
     *  @param msg  The message instance.
     */
    public boolean insertMessage(ChatMessage msg) {
        EMLog.d(TAG, "insertMessage msg: {msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
        boolean result = emaObject.insertMessage(msg.emaObject);
        if(result) {
            getCache().addMessage(msg);
        }
        return result;
    }

    /**
     *  \~english
     * Inserts a message to the end of the conversation in the local database.
     *
     * To insert the message correctly, ensure that the conversation ID of the message is the same as that of the conversation.
     *
     * After a message is inserted, the SDK will automatically update attributes of the conversation, including `latestMessage`.
     *
     *  @param msg The message instance.
     */
	public boolean appendMessage(ChatMessage msg) {
        EMLog.d(TAG, "appendMessage msg: {msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
	    boolean result = emaObject.appendMessage(msg.emaObject);
	    if(result) {
            getCache().addMessage(msg);
	    }
	    return result;
	}

    /**
     *  \~english
     * Updates a message in the local database.
     *
     * After you update a message, the message ID remains unchanged and the SDK automatically updates attributes of the conversation, like `latestMessage`.
     *
     *  @param msg  The message to update.
     */
    public boolean updateMessage(ChatMessage msg) {
        EMLog.d(TAG, "updateMessage msg{ msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
        boolean updateMessage = emaObject.updateMessage(msg.emaObject);
        if(updateMessage) {
            getCache().addMessage(msg);
        }
        return updateMessage;
    }

    /**
     * \~english
     * Gets the path to save attachments in the conversation.
     *
     * Before clearing conversation data, you can call this method to check the path to save attachments related to the conversation. If necessary, implement exception protection.
     *
     * @return The path to save attachments in the conversation.
     */
    public String getMessageAttachmentPath() {
        String downloadPath = ChatClient.getInstance().getChatConfigPrivate().getDownloadPath();
        return downloadPath + "/" + ChatClient.getInstance().getCurrentUser()
                + "/" + conversationId();
    }

    /**
     * \~english
     * 
     * Deletes historical messages from the server unidirectionally by message ID.
     * 
     * @param msgIdList 		The list of IDs of messages to be deleted from the server unidirectionally.
     * @param callBack			The result callback which contains the error information if the method fails. See {@link CallBack}.
     */
    public void removeMessagesFromServer(List<String> msgIdList, CallBack callBack){
        ChatClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),msgIdList,callBack);
    }

    /**
     * \~english
     * Deletes historical messages from the server unidirectionally by timestamp.
     *
     * @param beforeTimeStamp	 The starting UNIX timestamp for unidirectional message deletion. The unit is millisecond. 
     *                           After the timestamp is set, this method deletes messages that are received by the server before the specified timestamp. 
     * @param callBack			 The result callback which contains the error information if the method fails. See {@link CallBack}.
     */
    public void removeMessagesFromServer(long beforeTimeStamp, CallBack callBack){
        ChatClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),beforeTimeStamp,callBack);
    }

    /**
     * \~english
     * Deletes messages sent or received in a certain period from the local database.
     * @param startTime The starting Unix timestamp for message deletion. The unit is millisecond.
     * @param endTime The ending Unix timestamp for message deletion. The unit is millisecond.
     * @return Whether the message deletion succeeds: 
     *         - If the operation succeeds, the SDK returns `true`.
     *         - If the operation fails, the SDK returns `false`.
     */
    public boolean removeMessages(long startTime,long endTime) {
        EMLog.d(TAG, "remove msgs from conversation by startTime: " + startTime+" ,endTime: "+endTime);
        boolean result;
        result=emaObject._removeMessage(startTime, endTime);
        //Removes messages in the memory
        if(result) {
            getCache().removeMessages(startTime,endTime);
        }
        return result;
    }

    // ====================================== Message cache ======================================

    MessageCache getCache() {
        MessageCache cache;
        synchronized (ChatClient.getInstance().chatManager().caches) {
            cache = ChatClient.getInstance().chatManager().caches.get(emaObject.conversationId());
            if (cache == null) {
                cache = new MessageCache();
            }
            ChatClient.getInstance().chatManager().caches.put(emaObject.conversationId(), cache);
        }
        return cache;
    }

    static class MessageCache {

        TreeMap<Long, Object> sortedMessages = new TreeMap<Long, Object>(new MessageComparator());
        Map<String, ChatMessage> messages = new HashMap<String, ChatMessage>();
        Map<String, Long> idTimeMap = new HashMap<String, Long>();
        boolean hasDuplicateTime = false;

        final boolean sortByServerTime = ChatClient.getInstance().getChatConfigPrivate().getOptions().isSortMessageByServerTime();

        class MessageComparator implements Comparator<Long> {

            @Override
            public int compare(Long time0, Long time1) {
                long val = time0 - time1;
                if (val > 0) {
                    return 1;
                } else if (val == 0) {
                    return 0;
                } else {
                    return -1;
                }
            }
        }

        public synchronized ChatMessage getMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return null;
            }
            return messages.get(msgId);
        }

        public synchronized void addMessages(List<ChatMessage> msgs) {
            for (ChatMessage msg : msgs) {
                addMessage(msg);
            }
        }

        public synchronized void addMessage(ChatMessage msg) {
            if (msg == null || msg.emaObject == null || msg.getMsgTime() == 0 || msg.getMsgTime() == -1 || msg.getMsgId() == null
                    || msg.getMsgId().isEmpty() || msg.getType() == Type.CMD) {
                return;
            }
            String id = msg.getMsgId();
            // override message
            if (messages.containsKey(id)) {
                Long time = idTimeMap.get(id);
                if (time != null) {
                    removeMessageByTime(id, time);
                }
                messages.remove(id);
                idTimeMap.remove(id);
            }
            // messages share same time stamp
            long time = sortByServerTime ? msg.getMsgTime() : msg.localTime();
            if (sortedMessages.containsKey(time)) {
                hasDuplicateTime = true;
                Object v = sortedMessages.get(time);
                if (v != null) {
                    if (v instanceof ChatMessage) {
                        if(TextUtils.equals(((ChatMessage) v).getMsgId(),msg.getMsgId())) {
                            return;
                        }
                        List<ChatMessage> msgs = new LinkedList<>();
                        msgs.add((ChatMessage) v);
                        msgs.add(msg);
                        sortedMessages.put(time, msgs);
                    } else if (v instanceof List) {
                        List<ChatMessage> msgs = (List<ChatMessage>) v;
                        msgs.add(msg);
                    }
                }
            } else {
                sortedMessages.put(time, msg);
            }
            messages.put(id, msg);
            idTimeMap.put(id, time);
        }

        public synchronized void removeMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            ChatMessage msg = messages.get(msgId);
            if (msg != null) {
                Long time = idTimeMap.get(msgId);
                if (time != null) {
                    removeMessageByTime(msgId, time);
                    idTimeMap.remove(msgId);
                }
                messages.remove(msgId);
            }
        }

        public synchronized void removeMessages(long startTime, long endTime) {
            if (startTime > endTime) {
                EMLog.e(TAG, " removeMessages error: startTime > endTime");
                return;
            }
            //Iterate through the sortedMessages to get all the MessageIds to remove
            List<String> removedMsgIds = new ArrayList<>();
            for (Long timeStamp : sortedMessages.keySet()) {
                if (timeStamp >= startTime && timeStamp <= endTime) {
                    Object v = sortedMessages.get(timeStamp);
                    if (v != null && v instanceof List) {
                        List<ChatMessage> msgs = (List) v;
                        for (ChatMessage m : msgs) {
                            if (m != null && m.getMsgId() != null) {
                                removedMsgIds.add(m.getMsgId());
                            }
                        }
                    } else if (v != null && v instanceof ChatMessage) {
                        removedMsgIds.add(((ChatMessage) v).getMsgId());
                    }
                }
            }
            //remove message
            for (int i = 0; i < removedMsgIds.size(); i++) {
                removeMessage(removedMsgIds.get(i));
            }
        }

        synchronized void replaceMsgId(String oldMsgId, String newMsgId) {
            if(TextUtils.isEmpty(oldMsgId) || TextUtils.isEmpty(newMsgId)) {
                return;
            }
            ChatMessage message = messages.get(oldMsgId);
            if(message != null) {
                Long time = idTimeMap.get(oldMsgId);
                if(time != null) {
                    Object v = sortedMessages.get(time);
                    sortedMessages.remove(time);
                    idTimeMap.remove(oldMsgId);
                    long newTime = sortByServerTime ? message.getMsgTime() : message.localTime();
                    sortedMessages.put(newTime, v);
                    idTimeMap.put(newMsgId, newTime);
                }
                messages.remove(oldMsgId);
                messages.put(newMsgId, message);
            }
        }

        private synchronized void removeMessageByTime(String msgId, long time) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            if (hasDuplicateTime && sortedMessages.containsKey(time)) {
                Object v = sortedMessages.get(time);
                if (v != null && v instanceof List) {
                    List<ChatMessage> msgs = (List)v;
                    for (ChatMessage m : msgs) {
                        if (m != null && m.getMsgId() != null && m.getMsgId().equals(msgId)) {
                            msgs.remove(m);
                            break;
                        }
                    }
                } else {
                    sortedMessages.remove(time);
                }
            } else {
                sortedMessages.remove(time);
            }
        }

        public synchronized List<ChatMessage> getAllMessages() {
            List<ChatMessage> list = new ArrayList<ChatMessage>();
            if (hasDuplicateTime == false) {
                for (Object v : sortedMessages.values()) {
                    list.add((ChatMessage)v);
                }
            } else {
                for (Object v : sortedMessages.values()) {
                    if (v != null) {
                        if (v instanceof List) {
                            list.addAll((List<ChatMessage>)v);
                        } else {
                            list.add((ChatMessage)v);
                        }
                    }
                }
            }
            return list;
        }

        public synchronized ChatMessage getLastMessage() {
            if (sortedMessages.isEmpty()) {
                return null;
            }

            Object o = sortedMessages.lastEntry().getValue();
            if (o == null) {
                return null;
            }
            if (o instanceof ChatMessage) {
                return (ChatMessage)o;
            } else if (o instanceof List){
                List<ChatMessage> msgs = (List<ChatMessage>)o;
                if (msgs.size() > 0) {
                    return msgs.get(msgs.size() - 1);
                }
                return null;
            }
            return null;
        }

        public synchronized void clear() {
            //no need to keep the last message
            sortedMessages.clear();
            messages.clear();
            idTimeMap.clear();
        }

        public synchronized boolean isEmpty() {
            return sortedMessages.isEmpty();
        }
    }
}
