package io.agora.chat;

import io.agora.chat.adapter.EMAMessageStatistics;
import io.agora.chat.adapter.EMAStatisticsManager;

/**
 * \~english
 * The class for traffic statistics management of local messages.
 *
 * This class contains methods that are used to calculate the number of local messages of certain types sent and/or received in a specified period, as well as their traffic.
 *
 * This traffic statistics function is disabled by default. To use this function, you need to enable it by setting {@link ChatOptions#setEnableStatistics(boolean)} prior to the SDK initialization.
 *
 * The SDK only calculates the traffic of messages that are sent and received within the last 30 days after the traffic statistics function is enabled.
 *
 *  The message traffic is calculated as follows:
 *  - For a text, command, location or custom message, the message traffic is the traffic of the message body.
 *  - For an image or video message, the message traffic is the traffic sum of the message body, the image or video file, and the thumbnail.
 *  - For a file or voice message, the message traffic is the traffic sum of the message body and the attachment.
 *
 * **Note**
 * 1. For messages with attachments, the traffic will be calculated only if the download is successful.
 *    If an attachment is downloaded multiple times, its download traffic will be accumulated.
 * 2. For roaming messages pulled from the server, if they already exist in the local database, the SDK ignores them during traffic calculation.
 *
 * The SDK only measures the traffic of local messages, but not the actual message traffic. Generally, the calculated traffic volume is smaller than the actual traffic because of the following:
 *   - The traffic of the common protocol data generated during message sending is not considered;
 *   - For the received messages, the server aggregates them and adds common fields. During traffic statistics, the SDK only calculates the traffic of individual messages, but ignoring the traffic of common fields.
 *
 */
public class ChatStatisticsManager extends EMBase<EMAStatisticsManager> {

    /**
     * \~english
     * The message directions for querying.
     */
    public enum SearchMessageDirect {
        /**
         * \~english
         * This message is sent from the local client.
         */
        SEND(0),
        /**
         * \~english
         * The message is received by the local client.
         */
        RECEIVE(1),
        /**
         * \~english
         * The message is sent from or received by the local client.
         */
        ALL(100);

        private final int index;
        private SearchMessageDirect(int index) {
            this.index = index;
        }

        public int getIndex() {
            return index;
        }
    }

    /**
     * \~english
     * The message types for querying.
     */
    public enum SearchMessageType {
        /**
         * \~english
         * Text message.
         */
        TXT(0),
        /**
         * \~english
         * Image message.
         */
        IMAGE(1),
        /**
         * \~english
         * Video message.
         */
        VIDEO(2),
        /**
         * \~english
         * Location message.
         */
        LOCATION(3),
        /**
         * \~english
         * Voice message.
         */
        VOICE(4),
        /**
         * \~english
         * File message.
         */
        FILE(5),
        /**
         * \~english
         * Command message.
         */
        CMD(6),
        /**
         * \~english
         * Custom message.
         */
        CUSTOM(7),
        /**
         * \~english
         * All message types.
         */
        ALL(100);

        private final int index;
        private SearchMessageType(int index) {
            this.index = index;
        }

        public int getIndex() {
            return index;
        }
    }

    ChatClient mClient;

    ChatStatisticsManager(ChatClient client, EMAStatisticsManager manager) {
        emaObject = manager;
        mClient = client;
    }

    /**
     * \~english
     * Gets message traffic statistics by message ID.
     *
     * @param messageId The message ID.
     * @return  The message traffic statistics. See {@link MessageStatistics}.
     */
    public MessageStatistics getMessageStatistics(String messageId) {
        EMAMessageStatistics statistics = emaObject.getMsgStatistics(messageId);
        if(statistics != null) {
            return new MessageStatistics(statistics);
        }
        return null;
    }

    /**
     * \~english
     * Gets the count of messages of certain types that are sent and/or received in a specified period.
     *
     * @param startTimestamp    The starting timestamp for statistics. The unit is millisecond.
     * @param endTimestamp      The ending timestamp for statistics. The unit is millisecond.
     * @param direct            The message direction.
     * @param type              The message type.
     * @return                  The count of messages that meet the statistical conditions. `0` is returned in the case of a call failure.
     */
    public int getMessageCount(long startTimestamp, long endTimestamp, SearchMessageDirect direct, SearchMessageType type) {
        return emaObject.searchMsgStatisticsNumber(startTimestamp, endTimestamp, direct.getIndex(), type.getIndex());
    }

    /**
     * \~english
     * Gets the total traffic volume of messages of certain types that are sent and/or received in a specified period.
     *
     * The traffic is measured in bytes.
     *
     *  The message traffic is calculated as follows:
     *  - For a text, command, location, or custom message, the message traffic is the traffic of the message body.
     *  - For an image or video message, the message traffic is the traffic sum of the message body, the image or video file, and the thumbnail.
     *  - For a file or voice message, the message traffic is the traffic sum of the message body and the attachment.
     *
     * @param startTimestamp    The starting timestamp for statistics. The unit is millisecond.
     * @param endTimestamp      The ending timestamp for statistics. The unit is millisecond.
     * @param direct            The message direction.
     * @param type              The message type.
     * @return                  The total traffic volume of messages that meet the statistical conditions. `0` is returned in the case of a call failure.
     */
    public long getMessageSize(long startTimestamp, long endTimestamp, SearchMessageDirect direct, SearchMessageType type) {
        return emaObject.searchMsgStatisticsSize(startTimestamp, endTimestamp, direct.getIndex(), type.getIndex());
    }

    /**
     * \~english
     * Deletes the traffic data of local messages that are received by the server before the specified timestamp.
     *
     * @param timestamp The Unix timestamp threshold for traffic data deletion, in milliseconds. The traffic data of local messages that are received by the server before the specified timestamp will be deleted.
     * @return The deletion result:
     *         - `true`: The deletion succeeds;
     *         - `false`: The deletion fails.
     */
    boolean removeMessagesBeforeTime(long timestamp) {
        return emaObject.removeMsgStatisticsBeforeTime(timestamp);
    }


}
