/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import static io.agora.Error.CHATROOM_NOT_EXIST;
import static io.agora.Error.GENERAL_ERROR;

import android.text.TextUtils;

import android.support.annotation.NonNull;

import io.agora.CallBack;
import io.agora.ChatRoomChangeListener;
import io.agora.Error;
import io.agora.ResultCallBack;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAChatRoom;
import io.agora.chat.adapter.EMAChatRoomManager;
import io.agora.chat.adapter.EMAChatRoomManagerListener;
import io.agora.chat.adapter.EMAError;
import io.agora.exceptions.ChatException;
import io.agora.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * \~english
 * The chat room manager, which manages users joining and existing the chat room and getting the chat room list, and manages member privileges.
 * For example, joining a chat room:
 * ```java
 *   ChatClient.getInstance().chatroomManager().joinChatRoom(conversationId, new ValueCallBack<ChatRoom>() {
 *    	public void onSuccess(ChatRoom value) {
 *    	    //Processing logic for successful chat room joining.
 *        }
 *    	public void onError(int error, String errorMsg) {
 *    	    //Processing logic for chat room joining failures.
 *        }
 * 	});
 * ```
 */
public class ChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<ChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<ChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<ChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<ChatRoom>());
	
	ChatClient mClient;
	public ChatRoomManager(ChatClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~english
	 * Adds a chat room event listener.
	 * 
	 * @note
	 * Chat room destruction, member entry and exit, mute, and allow list changes among other chat room operations can be listened for by setting
	 * {@link ChatRoomChangeListener}.
	 *
	 * Chat room event listeners added with this method can be removed by calling {@link #removeChatRoomListener(ChatRoomChangeListener)}.
	 *
	 * @param listener A chat room event listener. See {@link ChatRoomChangeListener}.
	 */
	public void addChatRoomChangeListener(ChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
	 * \~english
	 * Removes a chat room event listener.
	 * 
	 * @note
	 * This method removes a chat room event listener added with {@link #addChatRoomChangeListener(ChatRoomChangeListener)}.
	 * 
	 * Currently, {@link #removeChatRoomListener(ChatRoomChangeListener)} is recommended to remove a chat room event listener.
	 *
	 * @param listener	The chat room listener to be removed.
	 * @deprecated  	Deprecated. Please use {@link ChatRoomManager#removeChatRoomListener(ChatRoomChangeListener)} instead.
	 */
    @Deprecated
	public void removeChatRoomChangeListener(ChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~english
	 * Removes the chat room event listener.
	 * 
	 * @note
	 * This method removes the chat room event listener added with {@link #addChatRoomChangeListener(ChatRoomChangeListener)}.
	 *
	 * @param listener	The chat room event listener to be removed.
	 */
	public void removeChatRoomListener(ChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~english
     * Joins a chat room.
	 * 
	 * @note
	 * To exit the chat room, call {@link #leaveChatRoom(String, CallBack)} .
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to join.
	 * @param callback 	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the chat room object;
	 *                  if this call fails, calls {@link ValueCallBack#onError(int, String)}, where the first parameter is the error code and
	 *                  the second is the error message.
	 */
	public void joinChatRoom(final String roomId, final ValueCallBack<ChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = ChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				ChatRoom room = new ChatRoom(r);
				if(callback!=null) {
					if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
						callback.onSuccess(room);
					} else {
						callback.onError(error.errCode(), error.errMsg());
					}
				}
			}
		});

	}
	
	/**
	 * \~english
	 * Exits a chat room.
	 * 
	 * @note
	 * A user that joins a chat room using {@link #joinChatRoom(String, ValueCallBack)} can call the leaveChatRoom method to exit the chat room.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to exit.
	 *
	 */
	public void leaveChatRoom(final String roomId) {
		ChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        ChatOptions options = ChatClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // The following is a temporary workaround to delete the conversation in case of leaving the room,
        // but deleteConversation could be a time-consuming operation.
		boolean delete = options.isDeleteMessagesAsExitChatRoom();
        if(delete) {
			ChatClient.getInstance().chatManager().deleteConversation(roomId,true);
        }
		threadPool.submit(new Runnable() {

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				ChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}
	/**
	 * \~english
	 * Exits a chat room.
	 *
	 * A user that joins a chat room using {@link #joinChatRoom(String, ValueCallBack)} can call the leaveChatRoom method to exit the chat room.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to exit.
	 *
	 * @param callback		The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *                      if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void leaveChatRoom(final String roomId, CallBack callback) {
		ChatRoom room = getChatRoom(roomId);
		if (room == null) {
			callback.onError(CHATROOM_NOT_EXIST,"chatroom not exist");
		    return;
		}

        ChatOptions options = ChatClient.getInstance().getOptions();

        boolean allowed = options.canChatroomOwnerLeave();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
			callback.onError(GENERAL_ERROR,"The owner can not allowed to leave chatroom");
            return;
        }

        // The following is a temporary workaround to delete the conversation in case of leaving the room,
        // but deleteConversation could be a time-consuming operation.
		boolean delete = options.deleteMessagesOnLeaveChatroom();
        if(delete) {
			ChatClient.getInstance().chatManager().deleteConversation(roomId,true);
        }
		threadPool.submit(new Runnable() {

			@Override
			public void run() {
				EMAError error =  new EMAError();
				ChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
				if(callback!=null) {
					if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == CHATROOM_NOT_EXIST) {
						callback.onSuccess();
					} else {
						callback.onError(error.errCode(), error.errMsg());
					}
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets chat room data from the server with pagination.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * The asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, ValueCallBack)} can be used.
	 *
	 * @param pageNum 				The page number, starting from 1.
	 * @param pageSize 				The number of records that you expect to get on each page. The value range is [1,50].
	 * @return      				Chat room data. See {@link PageResult}.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 *
	 */
	public PageResult<ChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws ChatException {
		EMAError error = new EMAError();
		PageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
		handleError(error);

		List<EMAChatRoom> rooms = result.getData();
		int count = result.getPageCount();

		PageResult<ChatRoom> ret = new PageResult<ChatRoom>();
		List<ChatRoom> data = new ArrayList<ChatRoom>();
		for (EMAChatRoom room : rooms) {
			data.add(new ChatRoom(room));
		}
		ret.setPageCount(count);
		ret.setData(data);

		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}

	/**
	 * \~english
	 * Gets chat room data from the server with pagination.
	 * 
	 * @note
	 * When {@link CursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 * 
	 * As this method is time-consuming, the asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, ValueCallBack)} can be used.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageSize 				The number of records per page. The value range is [1,50].
	 * @param cursor 				The cursor position from which to start getting data.
	 * @return 					    Chat room data. See {@link CursorResult}.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 * @deprecated 					Deprecated. Please use {@link #fetchPublicChatRoomsFromServer(int, int)} instead.
	 *
	 */
	@Deprecated
	public CursorResult<ChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<EMAChatRoom> cursorResult = emaObject.fetchChatroomsWithCursor(cursor, pageSize, error);
		handleError(error);
		
		CursorResult<ChatRoom> ret = new CursorResult<ChatRoom>();
		List<ChatRoom> data = new ArrayList<ChatRoom>();
		for (EMAChatRoom room : cursorResult.getData()) {
			data.add(new ChatRoom(room));
		}
		ret.setCursor(cursorResult.getCursor());
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
	 * \~english
	 * Gets chat room data from the server with pagination.
	 *
	 * @note
	 * When {@link CursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageSize 	The number of records that you expect to get on each page. The value range is [1,50].
	 * @param cursor 	The cursor position from which to start getting data.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                  If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 *
	 * @deprecated 		Deprecated. Please use {@link #asyncFetchPublicChatRoomsFromServer(int, int, ValueCallBack)} instead.
	 * 
	 */
	@Deprecated
	public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final ValueCallBack<CursorResult<ChatRoom>> callback) {
	    ChatClient.getInstance().execute(new Runnable() {
	        
	        @Override
            public void run() {
                try {
                    CursorResult<ChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~english
     * Gets chat room data from the server with pagination.
	 *
     * This is an asynchronous method.
     *
     * @param pageNum 	The page number, starting from 1.
     * @param pageSize 	The number of records that you expect to get on each page. The value range is [1,50].
     * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)}.
	 *                  If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final ValueCallBack<PageResult<ChatRoom>> callback) {
        ChatClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    PageResult<ChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~english
	 * Gets details of a chat room from the server.
	 * 
	 * @note
	 * This method does not get the member list.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * For the asynchronous method, see {@link #asyncFetchChatRoomFromServer(String, ValueCallBack)}.
	 *
	 * @param roomId				The chat room ID.
	 * @return ChatRoom			The chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public ChatRoom fetchChatRoomFromServer(String roomId) throws ChatException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
	 * \~english
	 * Gets details of a chat room from the server.
	 * 
	 * @note
	 * The member list, if required, can contain at most 200 members by default. For more members, call {@link ChatRoomManager#fetchChatRoomMembers(String, String, int)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param roomId 				The chat room ID.
	 * @param fetchMembers 			Whether to get chat room members.
	 * @return ChatRoom			The chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
 	public ChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws ChatException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new ChatRoom(room);
 	}
	
	/**
     * \~english
     * Gets details of a chat room from the server.
	 * 
	 * @note
	 * This method does not get the chat room member list.
	 *	 
	 * This is an asynchronous method.
	 *
     * For the synchronous method, see {@link #fetchChatRoomFromServer(String)}.
	 *
     * @param roomId	The chat room ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                  if this call fails, calls{@link ValueCallBack#onError(int, String)}.
     */
    public void asyncFetchChatRoomFromServer(final String roomId, final ValueCallBack<ChatRoom> callback) {
        ChatClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    ChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~english
	 * Gets the chat room in the memory.
	 *
	 * @param roomId		The chat room ID.
	 * @return 				The chat room instance. The SDK returns null if the chat room is not found in the memory.
	 */
	public ChatRoom getChatRoom(String roomId) {
		EMAChatRoom room = emaObject.getChatroom(roomId);
		if(room == null){
			return null;
		}
		return new ChatRoom(room);
	}
	
	/**
	 * \~english
	 * Gets the list of chat rooms in the memory.
	 * 
	 * @note
	 * Data can be returned upon the call of this method only after one of the following methods is called:
	 * - {@link #fetchPublicChatRoomsFromServer(int, int)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, ValueCallBack)}
	 * - {@link #fetchPublicChatRoomsFromServer(int, String)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, ValueCallBack)}
	 *  The returned data is the data last obtained by using one of the preceding methods.
	 * @return	The list of chat rooms maintained by ChatRoomManager.
	 * @deprecated Deprecated.
	 */
	@Deprecated
	public List<ChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}

	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            ChatClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
			    try {
			        for (ChatRoomChangeListener listener : chatRoomListeners) {
			            if (leaveReason == DESTROYED) {
			                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
			            } else {
							listener.onRemovedFromChatRoom(leaveReason, chatroom.getId(), chatroom.getName(), ChatClient.getInstance().getCurrentUser());
						}
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
				    for (ChatRoomChangeListener listener : chatRoomListeners) {
					    listener.onMemberJoined(chatroom.getId(), member);
				    }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
			        for (ChatRoomChangeListener listener : chatRoomListeners) {
			            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		public void onAddMuteList(EMAChatRoom chatRoom, List<String> mutes, long expireTime) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListAdded(chatRoom.getId(), mutes, expireTime);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveMutes(EMAChatRoom chatRoom, List<String> mutes) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListRemoved(chatRoom.getId(), mutes);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListAdded(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListAdded(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListRemoved(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListRemoved(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAllMemberMuteStateChanged(EMAChatRoom chatRoom, final boolean isMuted) {
			synchronized (chatRoomListeners) {
				try {
					for(ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAllMemberMuteStateChanged(chatRoom.getId(), isMuted);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAddAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminAdded(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminRemoved(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onOwnerChanged(EMAChatRoom chatRoom, String newOwner, String oldOwner) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onOwnerChanged(chatRoom.getId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAChatRoom chatRoom, String announcement) {
			synchronized (chatRoomListeners) {
				try {
					for (ChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAnnouncementChanged(chatRoom.getId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateSpecificationFromChatroom(EMAChatRoom chatroom) {
			synchronized (chatRoomListeners) {
				for (ChatRoomChangeListener listener : chatRoomListeners) {
					try {
						if (null != chatroom) {
							listener.onSpecificationChanged(new ChatRoom(chatroom));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onAttributesUpdate(String chatRoomId, String attributes, String from) {
			synchronized (chatRoomListeners) {
				for (ChatRoomChangeListener listener : chatRoomListeners) {
					Map<String,String> attributeMap = parseJsonUpdate(attributes);
					try {
						if (null != attributeMap && attributeMap.size() > 0)
						listener.onAttributesUpdate(chatRoomId,attributeMap,from);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onAttributesRemoved(String chatRoomId, String attributes, String from) {
			synchronized (chatRoomListeners) {
				for (ChatRoomChangeListener listener : chatRoomListeners) {
					List<String> keyList = parseJsonRemove(attributes);
					try {
						if (null != keyList && keyList.size() > 0)
							listener.onAttributesRemoved(chatRoomId,keyList,from);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}
	};


	/**
	 * \~english
	 * Creates a chat room.
	 * 
	 * @Note
	 * For the asynchronous method, see {@link #asyncCreateChatRoom(String, String, String, int, List, ValueCallBack)}. 
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param subject           	The chat room name.
	 * @param description       	The chat room description.
	 * @param welcomeMessage    	A welcome message inviting users to join the chat room.
	 * @param maxUserCount      	The maximum number of members allowed in the chat room.
	 * @param members           	The list of users invited to join the chat room.
	 * @return ChatRoom		    The created chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom createChatRoom(String subject, String description, String welcomeMessage,
	                                 int maxUserCount, List<String> members)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.createChatRoom(subject, description, welcomeMessage, ChatRoom.ChatRoomStyle.ChatRoomStylePublicOpenJoin.ordinal(), maxUserCount, members, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Creates a chat room.
	 *
	 * For the synchronous method, see {@link #createChatRoom(String, String, String, int, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param subject           The chat room name.
	 * @param description       The chat room description.
	 * @param welcomeMessage    A welcome message for inviting users to join the chat room.
	 * @param maxUserCount      The maximum number of members allowed in the chat room.
	 * @param members           The list of users invited to join the chat room.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncCreateChatRoom(final String subject, final String description, final String welcomeMessage,
	                                final int maxUserCount, final List<String> members, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(createChatRoom(subject, description, welcomeMessage, maxUserCount, members));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Destroys a chat room.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncDestroyChatRoom(String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public void destroyChatRoom(String chatRoomId)
			throws ChatException {
		EMAError error = new EMAError();
		emaObject.destroyChatroom(chatRoomId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Destroys a chat room.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #destroyChatRoom(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 *                      if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncDestroyChatRoom(final String chatRoomId, final CallBack callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyChatRoom(chatRoomId);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Changes the chat room name.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeChatRoomSubject(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newSubject			The modified chat room name.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}
     */
	public ChatRoom changeChatRoomSubject(String chatRoomId, String newSubject)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomSubject(chatRoomId, newSubject, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Changes the chat room name.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeChatRoomSubject(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param newSubject	The modified chat room name.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatRoomSubject(final String chatRoomId, final String newSubject, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatRoomSubject(chatRoomId, newSubject));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Modifies the chat room description.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeChatroomDescription(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newDescription		The modified chat room description.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom changeChatroomDescription(String chatRoomId, String newDescription)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomDescription(chatRoomId, newDescription, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Modifies the chat room description.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeChatroomDescription(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param newDescription	The modified chat room description.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatroomDescription(final String chatRoomId, final String newDescription, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatroomDescription(chatRoomId, newDescription));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the chat room member list.
	 * 
	 * @note
	 * When {@link CursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMembers(String, String, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param cursor				The cursor position from which to start getting data. At the first call, if you set the cursor as "null", the SDK gets chat room members in the reverse chronological order of when members joined the chat room.
	 * @param pageSize				The number of members that you expect to get on each page. The value range is [1,50].
	 * @return						The list of chat room members and the cursor for the next query. See {@link CursorResult}.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public CursorResult<String> fetchChatRoomMembers(String chatRoomId, String cursor, int pageSize)
			throws ChatException {
		EMAError error = new EMAError();
		CursorResult result = emaObject.fetchChatroomMembers(chatRoomId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~english
	 * Gets the chat room member list.
	 * 
	 * @note
	 * When {@link CursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param cursor		The cursor position from which to start getting data. At the first call, if you set the cursor as "null", the SDK gets chat room members in the reverse chronological order of when members joined the chat room.
	 * @param pageSize		The number of members that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomMembers(final String chatRoomId, final String cursor, final int pageSize, final ValueCallBack<CursorResult<String>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMembers(chatRoomId, cursor, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Mutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncMuteChatRoomMembers(String, List, long, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param muteMembers 			The list of members to be muted.
	 * @param duration 				The mute duration in milliseconds. The value `-1` indicates that the members are muted permanently.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom muteChatRoomMembers(String chatRoomId, List<String> muteMembers, long duration)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.muteChatroomMembers(
				chatRoomId, muteMembers, duration, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Mutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #muteChatRoomMembers(String, List, long)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param muteMembers 	The list of members to be muted.
	 * @param duration 		The mute duration in milliseconds. The value `-1` indicates that the members are muted permanently. 
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncMuteChatRoomMembers(final String chatRoomId, final List<String> muteMembers, final long duration, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteChatRoomMembers(chatRoomId, muteMembers, duration));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Unmutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncUnMuteChatRoomMembers(String, List, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be unmuted.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom unMuteChatRoomMembers(String chatRoomId, List<String> members)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unmuteChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Unmutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #unMuteChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param members		The list of members to be unmuted.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncUnMuteChatRoomMembers(final String chatRoomId, final List<String> members,
	                                             final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteChatRoomMembers(chatRoomId, members));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Transfers the chat room ownership.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeOwner(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			A chat room ID.
	 * @param newOwner				The new owner of the chat room.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public ChatRoom changeOwner(String chatRoomId, String newOwner)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.transferChatroomOwner(chatRoomId, newOwner, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Transfers the chat room ownership.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeOwner(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param newOwner		The new owner of the chat room.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeOwner(final String chatRoomId, final String newOwner, final ValueCallBack<ChatRoom> callBack)
			throws ChatException {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeOwner(chatRoomId, newOwner));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncAddChatRoomAdmin(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The member to be set as the chat room admin.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom addChatRoomAdmin(String chatRoomId, String admin)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.addChatroomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Adds a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #addChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The member to be set as the chat room admin.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncAddChatRoomAdmin(final String chatRoomId, final String admin, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addChatRoomAdmin(chatRoomId, admin));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes administrative privileges of a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomAdmin(String, String, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The chat room admin whose administrative privileges are to be removed.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom removeChatRoomAdmin(String chatRoomId, String admin)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Removes administrative privileges of a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #removeChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The chat room admin whose administrative privileges are to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAdmin(final String chatRoomId, final String admin,
	                                final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomAdmin(chatRoomId, admin));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the list of muted chat room members from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMuteList(String, int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of muted members that you expect to get on each page. The value range is [1,50].
	 * @return 						A map which contains the muted member ID and mute duration, where the key of each entry
	 * 								is the ID of a muted member and the value is the mute duration in milliseconds.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Map<String, Long> fetchChatRoomMuteList(String chatRoomId, int pageNum, int pageSize)
			throws ChatException {
		EMAError error = new EMAError();
		Map<String, Long> members = emaObject.fetchChatRoomMuteList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~english
	 * Gets the list of muted chat room members from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomMuteList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of muted members that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomMuteList(final String chatRoomId, final int pageNum, final int pageSize,
	                                       final ValueCallBack<Map<String, Long>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMuteList(chatRoomId, pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes members from a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomMembers(String, List, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members			The list of members to be removed.
	 * @return				The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom removeChatRoomMembers(String chatRoomId, List<String> members)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Removes members from a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #removeChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomMembers(final String chatRoomId, final List<String> members,
	                                       final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomMembers(chatRoomId, members));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds members to the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * - A member, once added to the chat room block list, will be removed from the chat room by the server.
	 * - The method {@link ChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} is triggered, 
	 *  where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}.  
	 * - Members added to the block list are banned from rejoining the chat room until they are removed from the block list.
	 * 
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be added to block list.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom blockChatroomMembers(String chatRoomId, List<String> members)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.blockChatroomMembers(chatRoomId, members, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Adds members to the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * - A member, once added to the chat room block list, will be removed from the chat room by the server.
	 * - The method {@link ChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} can be called to send a callback notification,
	 *    where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}. 
	 * - Members added to the block list are banned from rejoining the chat room until they are removed from the block list.
	 * For the synchronous method, see {@link #blockChatroomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be added to block list.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncBlockChatroomMembers(final String chatRoomId, final List<String> members, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(blockChatroomMembers(chatRoomId, members));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes members from the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be removed from the block list.
	 * @return						The modified chat room instance.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     */
	public ChatRoom unblockChatRoomMembers(String chatRoomId, List<String> members)
			throws ChatException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unblockChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new ChatRoom(chatRoom);
	}

	/**
	 * \~english
	 * Removes members from the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #unblockChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed from the block list.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncUnBlockChatRoomMembers(final String chatRoomId, final List<String> members,
	                                        final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unblockChatRoomMembers(chatRoomId, members));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the chat room block list with pagination.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomBlackList(String, int, int, ValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of members on the block list that you expect to get on each page. The value range is [1,50].
	 * @return						The block list of the chat room.
	 * @throws ChatException	A description of the exception. See {@link Error}.
     * 
	 * 
     */
	public List<String> fetchChatRoomBlackList(String chatRoomId, int pageNum, int pageSize)
			throws ChatException {
		EMAError error = new EMAError();
		List<String> blockList = emaObject.fetchChatRoomBlackList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return blockList;
	}

	/**
	 * \~english
	 * Gets the chat room block list with pagination.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomBlackList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of members on the block list that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the block list of the chat room;
	 * 						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
     * 		
	 */
	public void asyncFetchChatRoomBlackList(final String chatRoomId, final int pageNum, final int pageSize,
	                                        final ValueCallBack<List<String>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomBlackList(chatRoomId, pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds members to the allow list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * When {@link #muteAllMembers(String, ValueCallBack)} executed by the chat room owner or admin, does not work for members on the allow list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be added to the allowlist.
	 * @param callBack				The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 								if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void addToChatRoomWhiteList(final String chatRoomId, final List<String> members, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.addToWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new ChatRoom(chatRoom));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes members from the chat room block list.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For members removed from the block list, {@link #muteAllMembers(String, ValueCallBack)} works.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be removed from the chat room block list.
	 * @param callBack              The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)}; 
	 *                              if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void removeFromChatRoomWhiteList(final String chatRoomId, final List<String> members, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.removeFromWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new ChatRoom(chatRoom));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Checks whether the current member is on the chat room block list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} to show whether the member is on the block list;
	 * 						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void checkIfInChatRoomWhiteList(final String chatRoomId, ValueCallBack<Boolean> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~english
	 * Checks whether the current member is muted or not.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} to show whether the member is be muted or not;
	 * 						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncCheckIfInMuteList(final String chatRoomId, ValueCallBack<Boolean> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfBeMuted(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the chat room allow list from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void fetchChatRoomWhiteList(final String chatRoomId, final ValueCallBack<List<String>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchChatRoomWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Mutes all members.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * This method does not work for the chat room owner, admin, and members on the chat room block list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId  	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void muteAllMembers(final String chatRoomId, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.muteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new ChatRoom(chatRoom));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Unmutes all members.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void unmuteAllMembers(final String chatRoomId, final ValueCallBack<ChatRoom> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.unmuteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new ChatRoom(chatRoom));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Updates the chat room announcement.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncUpdateChatRoomAnnouncement(String, String, CallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param announcement 			The modified announcement content.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void updateChatRoomAnnouncement(String chatRoomId, String announcement) throws ChatException {
		EMAError error = new EMAError();
		emaObject.updateChatRoomAnnouncement(chatRoomId, announcement, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Updates the chat room announcement.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #updateChatRoomAnnouncement(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID
	 * @param announcement 	The modified announcement content.
	 * @param callBack 		The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 						if this call fails, calls {@link CallBack#onError(int, String)}. 
	 */
	public void asyncUpdateChatRoomAnnouncement(final String chatRoomId, final String announcement, final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatRoomAnnouncement(chatRoomId, announcement);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the chat room announcement from the server.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomAnnouncement(String, ValueCallBack)}.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * @param chatRoomId 			The chat room ID.
	 * @return						The chat room announcement.
	 * @throws ChatException	A description of the exception. See {@link Error}
	 */
	public String fetchChatRoomAnnouncement(String chatRoomId) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatRoomAnnouncement(chatRoomId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~english
	 * Gets the chat room announcement from the server.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomAnnouncement(String)}.
	 * 
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack  	The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the chat room announcement;
	 * 						if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomAnnouncement(final String chatRoomId, final ValueCallBack<String> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomAnnouncement(chatRoomId));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Adds custom chat room attributes.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncSetChatroomAttributes(String,Map,boolean, ResultCallBack)}
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`: Yes.
	 * - `false`: No.
	 *
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	private Map<String,Integer> setChatroomAttribute(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete) throws ChatException{
		EMAError error = new EMAError();
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),false,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> setChatroomAttributes(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete,EMAError error) {
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),false,error);
		return parseCodeJson(result);
	}

	/**
	 * \~english
	 * Adds custom chat room attributes.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The new chat room attributes Key-Value. A single key cannot exceed 128 characters; the total number of keys in a chat room cannot exceed 100. A single Value can not exceed 4096 characters. The total attribute can not exceed 10 GB.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. calls {@link ResultCallBack#onResult(int, Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link Error}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncSetChatroomAttributes(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete ,@NonNull final ResultCallBack<Map<String,Integer>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				if (null == attributeMap || attributeMap.size() == 0){
					callBack.onResult(Error.INVALID_PARAM,new HashMap<>());
					return;
				}
				EMAError error = new EMAError();
				Map<String,Integer> resultMap = setChatroomAttributes(chatRoomId,attributeMap,autoDelete,error);
				callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~english
	 * Sets a custom chat room attribute.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param key				The chat room attribute key that specifies the attribute name. The attribute name can contain 128 characters at most. 
	 *                          A chat room can have a maximum of 100 custom attributes. The following character sets are supported:
     * - 26 lowercase English letters (a-z)
     * - 26 uppercase English letters (A-Z)
     * - 10 numbers (0-9)
     * - "_", "-", "."
	 * @param value				The chat room attribute value. The attribute value can contain a maximum of 4096 characters. The total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attribute set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncSetChatroomAttribute(String chatRoomId,String key,String value,boolean autoDelete,@NonNull final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key) ){
						callBack.onError(Error.INVALID_PARAM,"add Attributes key Cannot be an empty string");
						return;
					}
					Map<String,String> attributeMap = new HashMap<>();
					attributeMap.put(key,value);
					setChatroomAttribute(chatRoomId,attributeMap,autoDelete);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Sets custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 * 
	 * For the synchronous method, see {@link #asyncSetChatroomAttributesForced(String, Map, boolean, ResultCallBack)}}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new custom chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 *
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	private Map<String,Integer> setChatroomAttributeForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),true,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> setChatroomAttributesForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete,EMAError error){
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),true,error);
		return parseCodeJson(result);
	}

	/**
	 * \~english
	 * Sets custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. calls {@link ResultCallBack#onResult(int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link Error}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncSetChatroomAttributesForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete ,@NonNull final ResultCallBack<Map<String,Integer>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if (null == attributeMap || attributeMap.size() == 0 ){
						callBack.onResult(Error.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = setChatroomAttributesForced(chatRoomId,attributeMap,autoDelete,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~english
	 * Sets a custom chat room attribute forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param key				The chat room attribute key that specifies the attribute name. The attribute name can contain 128 characters at most. 
	 *                          A chat room can have a maximum of 100 custom attributes. The following character sets are supported:
     * - 26 lowercase English letters (a-z)
     * - 26 uppercase English letters (A-Z)
     * - 10 numbers (0-9)
     * - "_", "-", "."
	 * @param value				The chat room attribute value. The attribute value can contain a maximum of 4096 characters. The total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncSetChatroomAttributeForced(String chatRoomId,String key,String value,boolean autoDelete ,@NonNull final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key) ){
						callBack.onError(Error.INVALID_PARAM,"add Attributes key Cannot be an empty string");
						return;
					}
					Map<String,String> attributeMap = new HashMap<>();
					attributeMap.put(key,value);
					setChatroomAttributeForced(chatRoomId,attributeMap,autoDelete);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the list of custom chat room attributes based on the attribute key list.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncFetchChatroomAttributesFromServer(String,List,ValueCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The list of chat room attribute keys. If you pass `null`, the SDK returns all custom attributes of the chat room.
	 * @throws ChatException 	A description of the exception. See {@link Error}.
	 */
	private Map<String,String> fetchChatroomAttributes(String chatRoomId , List<String> keyList) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatroomAttributes(chatRoomId,keyList,error);
		handleError(error);
		return parseJson(result);
	}

	/**
	 * \~english
	 * Gets the list of custom chat room attributes based on the attribute key list.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #fetchChatroomAttributes(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The list of chat room attribute keys. If you pass `null`, the SDK returns all custom attributes of the chat room.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatroomAttributesFromServer(String chatRoomId,List<String> keyList,@NonNull final ValueCallBack<Map<String,String>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatroomAttributes(chatRoomId,keyList));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets all the custom attributes of a chat room.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #fetchChatroomAttributes(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomAllAttributesFromServer(String chatRoomId,@NonNull final ValueCallBack<Map<String,String>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatroomAttributes(chatRoomId,null));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes custom chat room attributes by chat room ID and attribute key list.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncRemoveChatRoomAttributesFromServer(String,List, ResultCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The key list of the chat room attributes to remove. If you pass `null`, the SDK returns all the custom attributes of the chat room.
	 * @throws ChatException 	A description of the exception. See {@link Error}
	 */
	private Map<String,Integer> removeChatroomAttribute(String chatRoomId, List<String> keyList) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,false,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> removeChatroomAttributes(String chatRoomId, List<String> keyList,EMAError error) {
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,false,error);
		return parseCodeJson(result);
	}

	/**
	 * \~english
	 * Removes custom chat room attributes by the attribute key list.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The key list of the custom chat room attributes to remove.
	 * @param callBack			The completion callback. calls {@link ResultCallBack#onResult(int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link Error}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncRemoveChatRoomAttributesFromServer(String chatRoomId, List<String> keyList,@NonNull final ResultCallBack<Map<String,Integer>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if ( null == keyList || keyList.size() == 0 ){
						callBack.onResult(Error.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = removeChatroomAttributes(chatRoomId,keyList,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~english
	 * Removes a custom chat room attribute.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttribute(String,List)}
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param key				The key of the custom chat room attribute to remove.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAttributeFromServer(String chatRoomId, String key,@NonNull final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key)){
						callBack.onError(Error.INVALID_PARAM,"remove Attribute key Cannot be an empty string");
						return;
					}
					List<String> keyList = new ArrayList<>();
					keyList.add(key);
					removeChatroomAttribute(chatRoomId,keyList);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Removes custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncRemoveChatRoomAttributesFromServerForced(String,List, ResultCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The key list of the custom chat room attributes to remove.
	 * @throws ChatException 	A description of the exception. See {@link Error}.
	 */
	private Map<String,Integer> removeChatroomAttributeForced(String chatRoomId, List<String> keyList) throws ChatException {
		EMAError error = new EMAError();
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,true,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> removeChatroomAttributesForced(String chatRoomId, List<String> keyList,EMAError error) {
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,true,error);
		return parseCodeJson(result);
	}


	/**
	 * \~english
	 * Removes custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttributeForced(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The key list of the custom chat room attributes to remove.
	 * @param callBack			The completion callback. calls {@link ResultCallBack#onResult(int, Object)} (int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link Error}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncRemoveChatRoomAttributesFromServerForced(String chatRoomId, List<String> keyList,@NonNull final ResultCallBack<Map<String,Integer>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if (null == keyList || keyList.size() == 0 ){
						callBack.onResult(Error.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = removeChatroomAttributesForced(chatRoomId,keyList,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~english
	 * Removes a custom chat room attribute forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttributeForced(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param key				The key of the custom chat room attribute to remove.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link CallBack#onSuccess()};
	 * 							if this call fails, calls {@link CallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAttributeFromServerForced(String chatRoomId, String key,@NonNull final CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key)){
						callBack.onError(Error.INVALID_PARAM,"remove Attribute key Cannot be an empty string");
						return;
					}
					List<String> keyList = new ArrayList<>();
					keyList.add(key);
					removeChatroomAttributeForced(chatRoomId,keyList);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}



	// ============================= chatRoom_reform new add api end
	private void handleError(EMAError error)  throws ChatException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new ChatException(error);
	    }
	}

	private void handlePartialError(EMAError error)  throws ChatException {
		if (error.errCode() != EMAError.EM_NO_ERROR && error.errCode() != EMAError.PARTIAL_SUCCESS) {
			throw new ChatException(error);
		}
	}

	private Map<String,String> parseJson(String jsonString) throws ChatException {
		if (TextUtils.isEmpty(jsonString)) return null;
		Map<String, String> resultMap = new HashMap<>();
		try {
			JSONObject resultJson = new JSONObject(jsonString);
			Iterator<String> iterator = resultJson.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				String v = resultJson.getString(k);
				resultMap.put(k,v);
			}
			return resultMap;
		} catch (JSONException e) {
			e.printStackTrace();
			throw new ChatException(Error.SERVER_UNKNOWN_ERROR,"Unknown server error");
		}
	}

	private Map<String,Integer> parseCodeJson(String jsonString){
		Map<String, Integer> resultMap = new HashMap<>();
		if (TextUtils.isEmpty(jsonString)) return resultMap;
		try {
			JSONObject resultJson = new JSONObject(jsonString);
			Iterator<String> iterator = resultJson.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				int v = resultJson.getInt(k);
				resultMap.put(k,v);
			}
		} catch (JSONException e) {
			EMLog.e("parseCodeJson",e.getMessage());
		}
		return resultMap;
	}

	private Map<String,String> parseJsonUpdate(String jsonString){
		if (TextUtils.isEmpty(jsonString)) return null;
		Map<String, String> resultMap = new HashMap<>();
		try {
			List<String> successList = new ArrayList<>();
			JSONObject resultJson = new JSONObject(jsonString);
			JSONObject result = resultJson.getJSONObject("result");
			JSONArray jsonArray = result.getJSONArray("successKeys");
			for (int i = 0; i < jsonArray.length(); i++) {
				successList.add(jsonArray.getString(i));
			}
			JSONObject jsonObject = resultJson.getJSONObject("properties");
			Iterator<String> iterator = jsonObject.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				String v = jsonObject.getString(k);
				if (successList.contains(k)){
					resultMap.put(k,v);
				}
			}
			return resultMap;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	private List<String> parseJsonRemove(String jsonString){
		if (TextUtils.isEmpty(jsonString)) return null;
		try {
			List<String> successList = new ArrayList<>();
			JSONObject resultJson = new JSONObject(jsonString);
			JSONObject result = resultJson.getJSONObject("result");
			JSONArray jsonArray = result.getJSONArray("successKeys");
			for (int i = 0; i < jsonArray.length(); i++) {
				successList.add(jsonArray.getString(i));
			}
			return successList;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	private String toJsonString(Map<String,String> extJson,boolean autoDelete){
		JSONObject resultJson = null;
		try {
			resultJson = new JSONObject();
			resultJson.put("metaData",new JSONObject(extJson));
			if(autoDelete){
				resultJson.put("autoDelete","DELETE");
			}else {
				resultJson.put("autoDelete","NO_DELETE");
			}
			return resultJson.toString();
		} catch (JSONException e) {
			e.printStackTrace();
			return "";
		}
	}
	
	void onLogout() {
	}
}
