/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2015 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import static io.agora.chat.ChatMessage.self;

import android.graphics.BitmapFactory.Options;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import io.agora.CallBack;
import io.agora.ConversationListener;
import io.agora.Error;
import io.agora.MessageListener;
import io.agora.ValueCallBack;
import io.agora.chat.Conversation.ConversationType;
import io.agora.chat.ChatMessage.Status;
import io.agora.chat.ChatMessage.Type;
import io.agora.chat.adapter.EMAChatManager;
import io.agora.chat.adapter.EMAChatManagerListener;
import io.agora.chat.adapter.EMAConversation;
import io.agora.chat.adapter.EMAConversation.EMAConversationType;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMAFetchMessageOption;
import io.agora.chat.adapter.EMAGroupReadAck;
import io.agora.chat.adapter.EMAMessageReaction;
import io.agora.chat.adapter.EMAMessageReactionChange;
import io.agora.chat.adapter.EMAReactionManager;
import io.agora.chat.adapter.EMAReactionManagerListener;
import io.agora.chat.adapter.message.EMAFileMessageBody;
import io.agora.chat.adapter.message.EMAImageMessageBody;
import io.agora.chat.adapter.message.EMAMessage;
import io.agora.chat.adapter.message.EMAMessageBody;
import io.agora.exceptions.ChatException;
import io.agora.notification.core.EMNotificationHelper;
import io.agora.util.FileHelper;
import io.agora.util.EMLog;
import io.agora.util.ImageUtils;
import io.agora.util.PathUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * \~english
 * The chat manager class, responsible for sending and receiving messages, managing conversations (including loading and deleting conversations), and downloading attachments.
 *
 * The sample code for sending a text message is as follows:
 * 
 * ```java
 * 	ChatMessage message = ChatMessage.createTxtSendMessage(content, toChatUsername);
 * 	ChatClient.getInstance().chatManager().sendMessage(message);
 * ```
 */
public class ChatManager {

	EMAChatManager emaObject;
	EMAReactionManager emaReactionObject;

	private static final String TAG = "ChatManager";

	private final static String INTERNAL_ACTION_PREFIX = "em_";

	ChatClient mClient;

	Map<String, Conversation.MessageCache> caches = new Hashtable<String, Conversation.MessageCache>();

	protected ChatManager(){}

	protected ChatManager(ChatClient client, EMAChatManager manager, EMAReactionManager reactionManager) {
		mClient = client;

		emaObject = manager;
		emaObject.addListener(chatManagerListenerImpl);

		emaReactionObject = reactionManager;
		emaReactionObject.addListener(mReactionManagerListenerImpl);
	}

	private List<MessageListener> messageListeners = new CopyOnWriteArrayList<MessageListener>();
	private List<ConversationListener> conversationListeners = Collections.synchronizedList(new ArrayList<ConversationListener>());

	EMAChatManagerListener chatManagerListenerImpl = new EMAChatManagerListener() {
//		private final Object lockObj = new Object();
//
//        List<MessageListener> cloneSyncedList(List<MessageListener> list) {
//            if (list == null) {
//                return new ArrayList<>();
//            } else {
//                synchronized (lockObj) {
//                    return list.subList(0, list.size());
//                }
//            }
//        }

		@Override
		public void onReceiveMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<ChatMessage> msgs = new ArrayList<ChatMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new ChatMessage(msg));
					}

					List<ChatMessage> remainingMsgs = new ArrayList<ChatMessage>();
					for (ChatMessage msg : msgs) {
				    	/*if(msg.getChatType() == ChatMessage.ChatType.ChatRoom){
                            ChatRoom room = ChatClient.getInstance().chatroomManager().getChatRoom(msg.conversationId());
                            if(room == null){
                            	continue;
                            }
				    	}*/

	                    Conversation conv = getConversation(msg.conversationId(), Conversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
	                    if(conv == null){
	                    	EMLog.d(TAG, "no conversation");
	                    	continue;
	                    }
	                    
	                    // Command messages are not put into the cache.
	                    if (msg.getType() != Type.CMD) {
	                        conv.getCache().addMessage(msg);
	                    }
	                    remainingMsgs.add(msg);
				    }
				    
				    if(remainingMsgs.size() <= 0){
						EMLog.d(TAG, "no remainingMsgs");
				    	return;
				    }

					for (MessageListener l : messageListeners) {
						try { // Do not break the loop if one listener has a problem
							EMLog.d(TAG, "onMessageReceived： " + l);
							l.onMessageReceived(remainingMsgs);
						} catch (Exception e) {
							EMLog.d(TAG, "onMessageReceived has problem: " + e.getMessage());
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onReceiveCmdMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<ChatMessage> msgs = new ArrayList<ChatMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new ChatMessage(msg));
					}
                    try {
                        for (MessageListener l : messageListeners) {
                            l.onCmdMessageReceived(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

				}
			});
		}

		@Override
		public void onMessageStatusChanged(final EMAMessage message, final EMAError error) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    try {
                        ChatMessage msg = new ChatMessage(message);
                        for (MessageListener l : messageListeners) {
                            l.onMessageChanged(msg, null);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
	    public void onMessageAttachmentsStatusChanged(final EMAMessage message, final EMAError error) {
            try {
                ChatMessage msg = new ChatMessage(message);
                for (MessageListener l : messageListeners) {
                    l.onMessageChanged(msg, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
		}
		
		
        @Override
        public void onReceiveRecallMessages(final List<EMAMessage> messages) {
            mClient.executeOnMainQueue(new Runnable() {

                @Override
                public void run() {
                    List<ChatMessage> msgs = new ArrayList<ChatMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new ChatMessage(msg));
                        getConversation(msg.conversationId()).getCache().removeMessage(msg.msgId());
                    }

                    try {
                        for (MessageListener l : messageListeners) {
                            l.onMessageRecalled(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
		@Override
		public void onReceiveHasReadAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    List<ChatMessage> msgs = new ArrayList<ChatMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new ChatMessage(msg));
                    }

                    try {
                        for (MessageListener l : messageListeners) {
                            l.onMessageRead(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveHasDeliveredAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<ChatMessage> msgs = new ArrayList<ChatMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new ChatMessage(msg));
					}

                    try {
                        for (MessageListener l : messageListeners) {
                            l.onMessageDelivered(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveReadAckForConversation(String fromUsername, String toUsername) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveConversationHasReadAcks");
					synchronized (conversationListeners) {
						if(!TextUtils.equals(fromUsername, ChatClient.getInstance().getCurrentUser())) {
							//1v1聊天中收到对方发送的channel ack消息，将本地的会话的发送消息置为已读（对方已读）,需要重新load数据
							Conversation conversation = ChatClient.getInstance().chatManager().getConversation(fromUsername);
							if(conversation != null) {
								conversation.loadMoreMsgFromDB(null, conversation.getAllMessages().size());
							}
						}
						try {
							for (ConversationListener l : conversationListeners) {
								l.onConversationRead(fromUsername, toUsername);
							}
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onUpdateConversationList(final List<EMAConversation> conversations) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateConversationList");
                    synchronized (conversationListeners) {
	                    try {
	                        for (ConversationListener l : conversationListeners) {
	                            l.onConversationUpdate();
	                        }
	                    } catch (Exception e) {
		                    e.printStackTrace();
	                    }
                    }
				}
			});
		}

		@Override
		public void onReceivePrivateMessages(List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onPrivateMessages");
					for (EMAMessage msg : messages) {
						ChatMessage message = new ChatMessage(msg);
						EMNotificationHelper.getInstance().analyzeCmdMessage(message);
					}
				}
			});
		}

		@Override
		public void onReceiveReadAcksForGroupMessage(List<EMAGroupReadAck> acks) {
            mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveReadAcksForGroupMessage");

					List<GroupReadAck> groupAcks = new ArrayList<GroupReadAck>();
					for (EMAGroupReadAck ack : acks) {
						groupAcks.add(new GroupReadAck(ack));
					}

					try {
						for (MessageListener l : messageListeners) {
							l.onGroupMessageRead(groupAcks);
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

		@Override
		public void onUpdateGroupAcks() {
        	mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateGroupAcks");
					try {
						for (MessageListener l : messageListeners) {
							l.onReadAckForGroupMessageUpdated();
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

		@Override
		public void onMessageIdChanged(String convId, String oldId, String newId) {
			EMAMessage message = emaObject.getMessage(newId);
			if(message != null) {
				deleteImageScaledPath(message);
			}
			Conversation conv = getConversation(convId);
			if(conv != null) {
				conv.getCache().replaceMsgId(oldId, newId);
			}
		}

		@Override
		public void onMessageContentChanged(EMAMessage message, String operatorId, long operationTime) {

			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {

					ChatMessage msg = new ChatMessage(message);

					Conversation conv = getConversation(msg.conversationId(), Conversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
					if(conv != null){
						if (msg.getType() != Type.CMD) {
							conv.getCache().removeMessage(msg.getMsgId());
							conv.getCache().addMessage(msg);
						}
					}else{
						EMLog.d(TAG, "onMessageContentChanged:no conversation");
					}

					for (MessageListener l : messageListeners) {
						try { // Do not break the loop if one listener has a problem
							EMLog.d(TAG, "onMessageContentChanged： " + l);
							l.onMessageContentChanged(msg,operatorId,operationTime);
						} catch (Exception e) {
							EMLog.d(TAG, "onMessageContentChanged has problem: " + e.getMessage());
							e.printStackTrace();
						}
					}
				}
			});
		}
	};

	EMAReactionManagerListener mReactionManagerListenerImpl = new EMAReactionManagerListener() {
		@Override
		public void onMessageReactionDidChange(List<EMAMessageReactionChange> reactionChangeList) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onMessageReactionDidChange");
					List<MessageReactionChange> list = new ArrayList<>(reactionChangeList.size());
					MessageReactionChange reactionChange;
					for (EMAMessageReactionChange emaReactionChange : reactionChangeList) {
						reactionChange = new MessageReactionChange(emaReactionChange);
						list.add(reactionChange);
					}
					for (MessageListener l : messageListeners) {
						try {
							l.onReactionChanged(list);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});

		}
	};

	/**
	 * \~english
	 * Sends a message。
	 *  
	 * For a voice or image message or a message with an attachment, the SDK will automatically upload the attachment.
	 *
	 * You can determine whether to upload the attachment to the chat sever by setting {@link ChatOptions#setAutoTransferMessageAttachments(boolean)}.
	 *
	 * To listen for the message sending status, call {@link ChatMessage#setMessageStatusCallback(CallBack)}.
	 * 
	 * @param msg    The message object to be sent. Make sure to set the param. 
	 */
	public void sendMessage(final ChatMessage msg) {
		msg.makeCallbackStrong();

		boolean createConv = msg.getType() != Type.CMD;
		final Conversation conv = getConversation(msg.conversationId(), Conversation.msgType2ConversationType(msg.getTo(), msg.getChatType()), createConv, msg.isChatThreadMessage());

		//Add the message at first
		if (conv != null) {
			boolean exists = conv.getCache().getMessage(msg.getMsgId()) != null;
			if (!exists) {
				// The message sending time is lastMsgTime + 1 ms
				long lastMsgTime = System.currentTimeMillis();
				ChatMessage lastMsg = conv.getLastMessage();
				if (lastMsg != null) {
					lastMsgTime = lastMsgTime < lastMsg.getMsgTime() ? lastMsg.getMsgTime() : lastMsgTime;
				}
				msg.setMsgTime(lastMsgTime + 1);

				conv.getCache().addMessage(msg);
			}
		}

		class HandleError {
			HandleError(int code, String desc) {
				ChatMessage.EMCallbackHolder holder = msg.messageStatusCallBack;
				if (holder != null) {
					holder.onError(code, desc);
				}
			}
		}

    mClient.executeOnSendQueue(new Runnable() {
            
        @Override
        public void run() {
				try {
					// If do not use Chat Server, do not deal with the logic of pictures and videos
					if(mClient.getOptions().getAutoTransferMessageAttachments()) {
						// If message body is image, check scale request, set size, and file name
						if (msg.getType() == Type.IMAGE) {
							//the default status is fail status,
							//which lead to message show fail and then show success in need scale image
							msg.setStatus(Status.INPROGRESS);
							ImageMessageBody body = (ImageMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(Error.GENERAL_ERROR, "Message body can not be null");
								return;
							}

							String localUri = body.getLocalUrl();
							String remoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(remoteUrl)){
								if(!FileHelper.getInstance().isFileExist(localUri)) {
									new HandleError(Error.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							if(!body.isSendOriginalImage() && FileHelper.getInstance().isFileExist(localUri)) {
								String scaledImagePath = ImageUtils.getScaledImageByUri(mClient.getContext(), localUri);
								if(!TextUtils.equals(scaledImagePath, localUri)) {
									body.setOriginalLocalPath(localUri);
									long originalSize = FileHelper.getInstance().getFileLength(localUri);
									long scaledSize = FileHelper.getInstance().getFileLength(scaledImagePath);
									if (originalSize == 0) {
										EMLog.d(TAG, "original image size:" + originalSize);
										new HandleError(Error.FILE_INVALID, "original image size is 0");
										return;
									}
									EMLog.d(TAG, "original image size:" + originalSize + " scaled image size:" + scaledSize
											+ " ratio:" + (int) (scaledSize / originalSize) + "%");
									localUri = scaledImagePath;
									body.setLocalUrl(FileHelper.getInstance().formatInUri(localUri));
									body.setFileLength(scaledSize);
								}
								body.setFileName(FileHelper.getInstance().getFilename(localUri));
							}
							// get image width and height
							Options options = ImageUtils.getBitmapOptions(mClient.getContext(), localUri);
							if(options != null) {
								int width = options.outWidth;
								int height = options.outHeight;
								body.setSize(width, height);
							}
						}else if (msg.getType() == Type.VIDEO){
							msg.setStatus(Status.INPROGRESS);
							VideoMessageBody body = (VideoMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(Error.GENERAL_ERROR, "Message body can not be null");
								return;
							}
							Uri filePathUri = body.getLocalUri();
							String fileRemoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(fileRemoteUrl)){
								if(!FileHelper.getInstance().isFileExist(filePathUri)) {
									new HandleError(Error.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							// get video width and height
							String thumbPath = FileHelper.getInstance().getFilePath(body.getLocalThumbUri());
							if(!TextUtils.isEmpty(thumbPath)) {
								Options options = ImageUtils.getBitmapOptions(thumbPath);
								int width = options.outWidth;
								int height = options.outHeight;
								body.setThumbnailSize(width, height);
							}
						}
					}

					String oldId = msg.getMsgId();
					//set callback to replace old id
					setMessageSendCallback(msg, conv, oldId);
					emaObject.sendMessage(msg.emaObject);
				} catch (Exception e) {
					e.printStackTrace();
					new HandleError(Error.GENERAL_ERROR, "send message failed");
				}
			}
		});
	}

	private void setMessageSendCallback(final ChatMessage msg,
										final Conversation conv,
										final String oldId) {
		if (msg == null) {
			return;
		}

		msg.setInnerCallback(new CallBack(){

			@Override
			public void onSuccess() {
				deleteImageScaledPath(msg.emaObject);
				if (conv != null) {
					conv.getCache().replaceMsgId(oldId, msg.getMsgId());
				}
			}

			@Override
			public void onError(int code, String error) {

			}
		});
	}

	private void deleteImageScaledPath(final EMAMessage msg) {
		if(msg == null) {
		    return;
		}
		List<EMAMessageBody> bodies = msg.bodies();
		if(bodies.size() <= 0) {
		    return;
		}
		EMAMessageBody body = msg.bodies().get(0);
		if (body instanceof EMAImageMessageBody) {
			String originImagePath = ((EMAImageMessageBody) body).getOriginalLocalPath();
			if(!FileHelper.getInstance().isFileExist(originImagePath)) {
				return;
			}
			String scaleImagePath = ((EMAImageMessageBody)body).getLocalUrl();
			EMLog.d(TAG, "origin: + " + originImagePath + ", scale:" + scaleImagePath);
			//if scaleImagePath is not origin image path, should delete scale image file
			if(scaleImagePath != null && !scaleImagePath.equals(originImagePath)){
				boolean isDeleted = FileHelper.getInstance().deletePrivateFile(scaleImagePath);
				EMLog.d(TAG, "Deleted the scale image file: "+ isDeleted + " the scale image file path: "+scaleImagePath);
			}
			((EMAImageMessageBody)body).setLocalPath(originImagePath);
			Conversation conversation = getConversation(msg.conversationId());
			if(conversation != null) {
			    conversation.emaObject.updateMessage(msg);
			}
		}
	}

	/**
	 * \~english
	 * Sends the conversation read receipt to the server.
	 * 
	 * This method is only for one-to-one chat conversations.
	 *
	 * This method will inform the sever to set the unread message count of the conversation to `0`.
	 * The message sender will receive the {@link ConversationListener#onConversationRead(String, String)} callback.
	 * The message recipient that has logged in to multiple devices, will also receive the callback.
	 *
	 * If you want to send a read receipt for a group message, call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * @param conversationId			The conversation ID.
	 * @throws ChatException	    The possible exceptions are as follows: {@link Error#USER_NOT_LOGIN}, {@link Error#SERVER_NOT_REACHABLE}, and
	 * 									{@link Error#MESSAGE_INVALID}. See {@link Error}.
	 */
	public void ackConversationRead(String conversationId) throws ChatException {
		EMAError error = new EMAError();
		emaObject.sendReadAckForConversation(conversationId, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Sends the read receipt for a message to the server.
	 *
	 * **Note**
	 * 
	 * This method applies to one-to-one chats only.
	 * 
	 * This method only takes effect if you set {@link ChatOptions#setRequireAck(boolean)} as `true`.
	 *
	 * To send the read receipt for a group message, call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * We recommend that you call {@link #ackConversationRead(String)} when opening the chat page, and call this method in other cases to reduce the number of method calls.
	 *
	 * @param to			The message for which the read receipt is to be sent.
	 * @param messageId		The message ID.
	 * @throws ChatException  A description of the exception, see  {@link Error}.
	 */
	public void ackMessageRead(String to, String messageId) throws ChatException {
		ChatOptions chatOptions = ChatClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "As the chat option SetRequireAck is set to false, the read receipt is not sent.");
			return;
		}
		if(TextUtils.isEmpty(to)) {
			EMLog.e(TAG, "The to parameter cannot be null.");
		    return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			emaObject.sendReadAckForMessage(msg);
		} else { // just for Xinju since there is no local storage
			EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, ChatMessage.ChatType.Chat.ordinal());
			_msg.setMsgId(messageId);
			_msg.setFrom(to);
			//set conversationId because native code need it
			_msg.setConversationId(to);

			emaObject.sendReadAckForMessage(_msg);
		}
	}

	/**
	 * \~english
	 * Sends the read receipt for a group message to the server.
	 * 
	 * **Note**
	 *
	 * You can only call this method after setting {@link ChatOptions#setRequireAck(boolean)} and {@link ChatMessage#setIsNeedGroupAck(boolean)} to `true`.
	 * 
	 * To send the read recipient for a one-to-one chat message to the server, call {@link #ackMessageRead(String, String)}.

	 * To send the conversation read receipt to the server, call {@link #ackConversationRead(String)}.
	 *
	 * @param to					The conversation ID.
	 * @param messageId				The message ID.
	 * @param ext					The extension information, which is a custom keyword that specifies a custom action or command.
	 * @throws ChatException	A description of the exception, see {@link Error}.
	 */
	public void ackGroupMessageRead(String to, String messageId, String ext) throws ChatException {
		ChatOptions chatOptions = ChatClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			if (msg.isNeedGroupAck()) {
				emaObject.sendReadAckForGroupMessage(msg, ext);
			} else {
				EMLog.d(TAG, "normal group message, do not ack it");
			}
		}
	}
	
    /**
	 * \~english
	 * Recalls the sent message.
	 *
	 * If the recipient is offline when the message is delivered and recalled, the recipient only receives the callback {@link MessageListener#onMessageRecalled} instead of the message.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param message The message instance.
	 *
	 * @throws ChatException A description of the exception. See {@link Error}.
     */
    public void recallMessage(ChatMessage message) throws ChatException {
        EMAError error = new EMAError();
        if (message == null) {
            throw new ChatException(Error.MESSAGE_INVALID, "The message was not found");
        }
        emaObject.recallMessage(message.emaObject, error);
        handleError(error);
        // 指定是否是Thread消息
		Conversation conversation = getConversation(message.getTo(), Conversation.msgType2ConversationType(message.getMsgId(), message.getChatType()), true, message.isChatThreadMessage());
		if(conversation != null) {
		    conversation.getCache().removeMessage(message.getMsgId());
		}
    }

	/**
	 * \~english
	 * Recalls the sent message.
	 *
	 * This is an asynchronous method.
	 *
	 * @param message	The message object.
	 * @param callback  A object of the CallBack class. See {@link CallBack}.
	 */
	public void asyncRecallMessage(final ChatMessage message,
                                   final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    recallMessage(message);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~english
	 * Gets the message by message ID.
	 *
	 * @param messageId 	The message ID.
	 * @return  The message object with the specified ID. The SDK returns `null` if the message does not exist.
	 */
	public ChatMessage getMessage(String messageId) {
		synchronized (caches) {
			for(Conversation.MessageCache cache : caches.values()) {
				ChatMessage msg = cache.getMessage(messageId);
				if (msg != null) {
					return msg;
				}
			}
		}
		EMAMessage message =  emaObject.getMessage(messageId);
		if (message == null) {
			return null;
		}
		ChatMessage msg = new ChatMessage(message);
		return msg;
	}

	ChatMessage getMessage(long nativeHandler) {
		EMAMessage message = emaObject.getMessage(nativeHandler);
		if(message == null) {
		    return null;
		}
		return new ChatMessage(message);
	}


	/**
	 * \~english
	 * Gets the conversation object by conversation ID.
	 * 
	 * The SDK will return `null` if the conversation is not found.
	 *
	 * @param id 	The conversation ID.
	 * @return 		The conversation with the specified ID. The SDK returns `null` if the conversation is not found.
	 */
	public Conversation getConversation(String id){
		EMAConversation conversation = emaObject.conversationWithType(id, EMAConversationType.CHAT, false, false);
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, false);
		}
		// 判断完群组会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, false);
		}
		// 判断完聊天室会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.DISCUSSIONGROUP, false, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.HELPDESK, false, false);
		}

		return conversation == null ? null : new Conversation(conversation);
	}

	/**
	 * \~english
	 * Gets the conversation by conversation ID and type.
	 * 
	 * The SDK will return `null` if the conversation is not found.
	 *
	 * @param id 		The conversation ID:
	 *                  - One-to-one chat: The ID of the peer user;
	 *                  - Group chat: The group ID;
	 *                  - Chat room chat: The chat room ID;
	 *                  - Message thread: The message thread ID.
	 * @param type  	The conversation type. See {@link ConversationType}
	 * @return 			The conversation object found by ID and type. The SDK returns `null` if the conversation is not found.
	 */
	public Conversation getConversation(String id, ConversationType type) {
		Conversation conversation = getConversation(id, type, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(id, type, false, true);
		}
		return conversation;
	}

	/**
	 * \~english
	 * Gets the conversation object by user ID or group ID and conversation type.
	 *
	 * @param username 			The conversation ID:
	 *                          - One-to-one chat: The ID of the peer user;
	 *                          - Group chat: The group ID;
	 *                          - Chat room chat: The chat room ID;
	 *                          - Message thread: The message thread ID.
	 * @param type 				The conversation type. See {@link ConversationType}.
	 * @param createIfNotExists Whether to create a conversation if the specified conversation is not found:
	 *                          -  `true`: Yes;
	 *                          -  `false`: No.
	 * @return					The retrieved conversation object. The SDK returns `null` if the conversation is not found.
	 */
	public Conversation getConversation(String username, ConversationType type, boolean createIfNotExists) {
		Conversation conversation = getConversation(username, type, createIfNotExists, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(username, type, createIfNotExists, true);
		}
		return conversation;
	}

	/**
	 * \~english
	 * Get conversation object by conversation id and conversation type.
	 *
	 * If the conversation is not found, you can create a new object based on the value of CreateIFNotExists or an empty object.
	 *
	 * @param username 			The conversation ID.
	 * *                        - One-to-one chat: The ID of the peer user;
	 *                          - Group chat: The group ID;
	 *                          - Chat room chat: The chat room ID;
	 *                          - Message thread: The message thread ID.
	 * @param type 				The conversation type. See {@link ConversationType}.
	 * @param createIfNotExists Whether to create a conversation if the specified conversation is not found:
	 *                          - `true`: Yes;
	 *                          -  `false`: No.
	 * @param isChatThread 		Whether to search for the chat thread conversation.
	 *                          - `true`: Yes;
	 * 							- `false`: No.
	 * @return					The retrieved conversation object. The SDK returns `null` if the conversation is not found.
	 */
	public Conversation getConversation(String username, ConversationType type, boolean createIfNotExists, boolean isChatThread) {
		EMAConversationType t = EMAConversationType.CHAT;
		switch (type) {
			case Chat:
				t = EMAConversationType.CHAT;
		        break;
			case GroupChat:
				t = EMAConversationType.GROUPCHAT;
		        break;
			case ChatRoom:
				t = EMAConversationType.CHATROOM;
		        break;
			case DiscussionGroup:
				t = EMAConversationType.DISCUSSIONGROUP;
		        break;
			case HelpDesk:
				t = EMAConversationType.HELPDESK;
		        break;
		}
		EMAConversation conversation = emaObject.conversationWithType(username, t, createIfNotExists, isChatThread);
		if (conversation == null) {
			return null;
		}
		Log.d(TAG, "convID:" + conversation.conversationId());
		return new Conversation(conversation);
	}


	/**
	 * \~english
	 * Marks all conversations as read.
	 * 
	 * This method is for the local conversations only.
	 */
	public void markAllConversationsAsRead() {
		List<EMAConversation> conversations = emaObject.loadAllConversationsFromDB();
		for (EMAConversation conversation : conversations) {
			conversation.markAllMessagesAsRead(true);
		}
	}

	/**
	 * \~english
	 * Gets the unread message count.
	 *
	 * @return	The count of unread messages.
	 */
	public int getUnreadMessageCount()
	{
		List<EMAConversation> conversations = emaObject.getConversations();
		int unreadCount = 0;
		for (EMAConversation conversation : conversations) {
			if (conversation._getType() != EMAConversationType.CHATROOM) {
				unreadCount += conversation.unreadMessagesCount();
			}
		}
		return unreadCount;
	}

	/**
	 * \~english
	 * Saves the message to the memory and local database.
	 * 
	 * The command messages (type: {@link Type#CMD}, message body: {@link CmdMessageBody}) are not saved locally.
	 *
	 * @param message	The message to store.
	 */
	public void saveMessage(ChatMessage message) {
		ChatMessage.ChatType type = message.getChatType();
		ConversationType t = ConversationType.Chat;
		switch (type) {
			case Chat:
				t = ConversationType.Chat;
				break;
			case GroupChat:
				t = ConversationType.GroupChat;
				break;
			case ChatRoom:
				t = ConversationType.ChatRoom;
				break;
		}
		String convId = message.getTo();
		//for group, chatroom, conversation id is group id for both receive and send message
		if (t == ConversationType.Chat && message.direct() == ChatMessage.Direct.RECEIVE) {
			convId = message.getFrom();
		}
		if (message.getType() == Type.CMD) {
			return;
		}
		Conversation conv = getConversation(convId, t, true, message.isChatThreadMessage());
		if(conv == null) {
			EMLog.e(TAG, "Failed to save message because conversation is null, convId: "+convId);
		    return;
		}
		// when send message out, appendMessage will update time to lastMsgTime + 1ms
		conv.insertMessage(message);
	}

	/**
	 * \~english
	 * Updates the local message.
	 * 
	 * This method updates the message in both the memory and the local database at the same time.
	 *
	 * @param message The message object to update.
	 */
	public boolean updateMessage(ChatMessage message) {
		String id = message.direct() == ChatMessage.Direct.RECEIVE ? message.getFrom() : message.getTo();
		if (message.getType() == Type.CMD) {
			return false;
		}
		Conversation conv = getConversation(message.conversationId(), Conversation.msgType2ConversationType(id, message.getChatType()), true, message.isChatThreadMessage());
		return conv.updateMessage(message);
	}


	/**
	 * \~english
	 * Modifies a message.
     *
	 * After this method is called to modify a message, both the local message and the message on the server are modified.
     *
	 * This method can only modify a text message in one-to-one chats or group chats, but not in chat rooms.
	 * 
	 * Upon a message modification, the callback {@link io.agora.MessageListener#onMessageContentChanged(io.agora.chat.ChatMessage, java.lang.String, long)} will be received by the message recipient(s) and in multi-device login scenarios.
	 *
	 * @param messageId  The ID of the message to modify.
	 * @param messageBodyModified The modified message body.
	 * @param callBack The message modification callback. A object of the ValueCallBack class. See {@link ValueCallBack}，contains the message modified.
	 */
	public void asyncModifyMessage(String messageId,MessageBody messageBodyModified,final ValueCallBack<ChatMessage> callBack){
		if(TextUtils.isEmpty(messageId)||messageBodyModified==null) {
			callBack.onError(Error.GENERAL_ERROR,"messageId or messageBody is empty");
		    return;
		}
		mClient.executeOnSendQueue(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error=new EMAError();
					EMAMessage emaMessage= emaObject.modifyMessage(messageId,messageBodyModified.emaObject,error);
					handleError(error);
					if(emaMessage!=null) {
						callBack.onSuccess(new ChatMessage(emaMessage));
					}else{
						callBack.onError(Error.GENERAL_ERROR,"emaMessage is empty");
					}
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(),e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Downloads the message attachment.
	 * 
	 * You can also call this method if the attachment fails to be downloaded automatically.
	 *
	 * @param msg 	The ID of the message with the attachment to be downloaded.
	 */
	public void downloadAttachment(final ChatMessage msg) {
        if (msg == null) {
            return;
        }
        msg.makeCallbackStrong();
		checkContentAttachmentExist(msg);
		emaObject.downloadMessageAttachments(msg.emaObject);
	}

	/**
	 * \~english
	 * Checks whether the attachment exists in the message.
	 * 
	 * @param msg  The message object.
	 */
	private void checkContentAttachmentExist(ChatMessage msg) {
		List<EMAMessageBody> bodies = msg.emaObject.bodies();
		if(bodies != null && !bodies.isEmpty()) {
		    for (EMAMessageBody body: bodies) {
		    	switch (body.type()) {
		    	    case EMAMessageBody.EMAMessageBodyType_IMAGE :
		    	    case EMAMessageBody.EMAMessageBodyType_VIDEO :
		    	    case EMAMessageBody.EMAMessageBodyType_VOICE :
		    	    case EMAMessageBody.EMAMessageBodyType_FILE :
						EMAFileMessageBody fileBody = (EMAFileMessageBody) body;
						String localUrl = fileBody.getLocalUrl();
						EMLog.d(TAG, "download before check path = "+localUrl);
						// If local url does not exit or is not in private app path, new file path will be created
						if(TextUtils.isEmpty(localUrl)
								|| !localUrl.contains("/Android/data/" + mClient.getContext().getPackageName() + "/")) {
							String filename = fileBody.displayName();
							String newLocalPath = null;
							switch (body.type()) {
							    case EMAMessageBody.EMAMessageBodyType_IMAGE :
									newLocalPath = PathUtil.getInstance().getImagePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VIDEO :
									newLocalPath = PathUtil.getInstance().getVideoPath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VOICE :
									newLocalPath = PathUtil.getInstance().getVoicePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_FILE :
									newLocalPath = PathUtil.getInstance().getFilePath()+File.separator+filename;
							        break;
							}
							if(!TextUtils.isEmpty(newLocalPath)) {
							    fileBody.setLocalPath(newLocalPath);
							    updateMessage(msg);
								EMLog.d(TAG, "download:create new path , path is "+newLocalPath);
							}
						}
						break;
		    	}
			}
		}
	}

	/**
	 * \~english
	 * Downloads the message thumbnail.
	 *
	 * @param msg  The message with the thumbnail to be downloaded. Only the image messages and video messages have a thumbnail.
	 */
	public void downloadThumbnail(final ChatMessage msg) {
        msg.makeCallbackStrong();
		emaObject.downloadMessageThumbnail(msg.emaObject);
	}

	/**
	 * \~english
	 * Downloads and parses the attachment of the combined message.
	 *
	 * For a combined message, this method downloads and parses the attachment and returns the message list at the first call;
	 * for subsequent calls, there are two cases:
	 * 1.If the attachment already exits, this method directly parses it and returns the message list.
	 * 2.If the attachment does not exist, this method downloads it before parsing it and returning the message list.
	 *
	 * @param message	The combined message to download and parse.
	 * @param callback  The completion callback. If this call succeeds, calls {@link ValueCallBack#onSuccess(Object)} and returns the parsed message list;
	 * 					If this call fails, calls {@link ValueCallBack#onError(int, String)}.
	 */
	public void downloadAndParseCombineMessage(ChatMessage message, ValueCallBack<List<ChatMessage>> callback) {
		ChatClient.getInstance().execute(()-> {
			if(message == null) {
			    callback.onError(Error.INVALID_PARAM, "Message cannot be null.");
				return;
			}
			if(message.getType() != Type.COMBINE) {
			    callback.onError(Error.INVALID_PARAM, "Only combine message are supported.");
				return;
			}
			EMAError error = new EMAError();
			List<EMAMessage> msgs = emaObject.downloadCombineMessages(message.emaObject, error);
			try {
				handleError(error);
			} catch (ChatException e) {
				callback.onError(e.getErrorCode(), e.getDescription());
				return;
			}
			List<ChatMessage> messageList = new ArrayList<>();
			for (EMAMessage msg : msgs) {
				messageList.add(new ChatMessage(msg));
			}
			callback.onSuccess(messageList);
		});
	}

	/**
	 * \~english
	 * Imports messages to the local database.
	 * 
	 * You can only import messages that you sent or received.
	 * 
	 * It is recommended that you import at most 1,000 messages each time.
	 *
	 * @param msgs The messages to import.
	 */
	public synchronized void importMessages(List<ChatMessage> msgs) {
		List<EMAMessage> dummy = new ArrayList<EMAMessage>();
		for (ChatMessage msg : msgs) {
			dummy.add(msg.emaObject);
		}
		ChatClient.getInstance().getChatConfigPrivate().importMessages(dummy);
	}

	/**
	 * \~english
	 * Gets the list of conversations by conversation type.
	 *
	 * @param type	The conversation type. See {@link ConversationType}.
	 * @return 		The list of conversations of the specified type.
	 */
	public List<Conversation> getConversationsByType(ConversationType type) {
		List<EMAConversation> conversations = emaObject.getConversations();
		List<Conversation> result = new ArrayList<Conversation>();
		for (EMAConversation conv : conversations) {
			if (type.ordinal() == conv._getType().ordinal()) {
				result.add(new Conversation(conv));
			}
		}
		return result;
	}

	/**
	 * \~english
	 * Gets all local conversations.
	 * 
	 * Conversations will be first retrieved from the memory. If no conversation is found, the SDK retrieves from the local database.
	 *
	 * @return The retrieved conversations.
	 */
	public Map<String, Conversation> getAllConversations() {
		List<EMAConversation> conversations = emaObject.getConversations();
		Hashtable<String, Conversation> result = new Hashtable<String, Conversation>();
		for (EMAConversation conversation : conversations) {
			/*if (conversation._getType() != EMAConversationType.CHATROOM) {
				result.put(conversation.conversationId(), new Conversation(conversation));
			}*/
			result.put(conversation.conversationId(), new Conversation(conversation));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets all local conversations.
	 * 
	 * Pinned conversations are listed at the top of the list and followed by unpinned conversations.
	 * 
	 * The SDK returns conversations in the descending order of the timestamp of the latest message in them, with the pinned ones at the top of the list and followed by the unpinned ones.
	 * 
	 * Conversations will be first retrieved from the memory. If no conversation is found, the SDK retrieves from the local database.
	 * 
	 * @return The sorted conversation list.
	 */
	public List<Conversation> getAllConversationsBySort() {
		List<EMAConversation> conversations = emaObject.getAllConversationsBySort();
		List<Conversation> result = new ArrayList<>();
		for (EMAConversation conversation : conversations) {
			result.add(new Conversation(conversation));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets the conversation list from the server.
	 * 
	 * You can pull 10 conversations within 7 days by default (each conversation contains the latest historical message).
     *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @return    The conversation list of the current user.
	 * @deprecated Deprecated. Use {@link #asyncFetchConversationsFromServer(int, String, ValueCallBack)} instead.
	 */
	@Deprecated
	public Map<String, Conversation> fetchConversationsFromServer() throws ChatException {
		EMAError error = new EMAError();

		List<EMAConversation> conversations = emaObject.fetchConversationsFromServer(error);
		//emaObject.fetchConversationsFromServer(error);
		handleError(error);

		Hashtable<String, Conversation> result = new Hashtable<String, Conversation>();
		for (EMAConversation conversation : conversations) {
			result.put(conversation.conversationId(), new Conversation(conversation));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets the conversation list from the server.
     *
	 * This is an asynchronous method.
	 *
	 * @return	The conversation list of the current user.
	 * @deprecated Deprecated. Use {@link #asyncFetchConversationsFromServer(int, String, ValueCallBack)} instead.
	 */
	@Deprecated
	public void asyncFetchConversationsFromServer(final ValueCallBack<Map<String, Conversation>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchConversationsFromServer());
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	Map<String, Conversation> fetchConversationsFromServer(int pageNum, int pageSize) throws ChatException {
		EMAError error = new EMAError();

		List<EMAConversation> conversations = emaObject.fetchConversationsFromServerWithPage(error, pageNum, pageSize);
		//emaObject.fetchConversationsFromServer(error);
		handleError(error);

		Hashtable<String, Conversation> result = new Hashtable<String, Conversation>();
		for (EMAConversation conversation : conversations) {
			result.put(conversation.conversationId(), new Conversation(conversation));
		}
		return result;
	}

	/**
	 * \~english
	 * Gets the conversation list from the server with pagination.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param pageNum The current page number, starting from 1.
	 * 
	 * @param pageSize The number of conversations to get on each page. The value range is [1,20].
	 *
	 * @return	The conversation list of the current user.
	 * @deprecated Deprecated. Use {@link #asyncFetchConversationsFromServer(int, String, ValueCallBack)} instead.
	 */
	@Deprecated
	public void asyncFetchConversationsFromServer(int pageNum, int pageSize, final ValueCallBack<Map<String, Conversation>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchConversationsFromServer(pageNum, pageSize));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Get the list of pinned conversations from the server with pagination.
	 * 
	 * The SDK returns pinned conversations in the reverse chronological order of their pinning.
	 * 
	 * @param limit         The number of conversations that you expect to get on each page. The value range is [1,50].
	 * @param cursor        The position from which to start getting data. If you pass in `null` or an empty string (""), the SDK retrieves the pinned conversations from the latest pinned one.
	 * @param callback      The query result {@link CursorResult}, including the cursor for getting data next time and the list of pinned conversations.
	 * 						For the last page, the return value of cursor is an empty string.
	 */
	public void asyncFetchPinnedConversationsFromServer(final int limit, final String cursor, final ValueCallBack<CursorResult<Conversation>> callback) {
		mClient.execute(()-> {
			EMAError error = new EMAError();
			CursorResult<EMAConversation> _cursorResult = emaObject.fetchPinnedConversationsFromServer(limit, cursor, error);
			try {
				handleError(error);
			} catch (ChatException e) {
				callback.onError(e.getErrorCode(), e.getDescription());
				return;
			}
			CursorResult<Conversation> cursorResult = new CursorResult<>();
			cursorResult.setCursor(_cursorResult.getCursor());
			List<Conversation> conversationList = new ArrayList<>();
			for (EMAConversation conv : _cursorResult.getData()) {
				conversationList.add(new Conversation(conv));
			}
			cursorResult.setData(conversationList);
			callback.onSuccess(cursorResult);
		});
	}

	/**
	 * \~english
	 * Gets the list of conversations from the server with pagination.
	 * 
	 * The SDK retrieves the list of conversations in the reverse chronological order of their active time (generally the timestamp of the last message). 
	 *
	 * If there is no message in the conversation, the SDK retrieves the list of conversations in the reverse chronological order of their creation time.
	 * 
	 *  @param limit        The number of conversations that you expect to get on each page. The value range is [1,50].
	 * @param cursor        The position from which to start to get data. If you pass in `null` or an empty string (""), the SDK retrieves conversations from the latest active one.
	 * @param callback      The query result {@link CursorResult}, including the cursor for getting data next time and the conversation list.
	 * 						For the last page, the return value of cursor is an empty string.
	 */
	public void asyncFetchConversationsFromServer(final int limit, final String cursor, final ValueCallBack<CursorResult<Conversation>> callback) {
		mClient.execute(()-> {
			EMAError error = new EMAError();
			CursorResult<EMAConversation> _cursorResult = emaObject.fetchConversationsFromServerWithCursor(limit, cursor, error);
			try {
				handleError(error);
			} catch (ChatException e) {
				callback.onError(e.getErrorCode(), e.getDescription());
				return;
			}
			CursorResult<Conversation> cursorResult = new CursorResult<>();
			cursorResult.setCursor(_cursorResult.getCursor());
			List<Conversation> conversationList = new ArrayList<>();
			for (EMAConversation conv : _cursorResult.getData()) {
				conversationList.add(new Conversation(conv));
			}
			cursorResult.setData(conversationList);
			callback.onSuccess(cursorResult);

		});
	}

	/**
	 * \~english
	 * Sets whether to pin a conversation.
	 * 
	 * @param conversationId 	The conversation ID.
	 * @param isPinned 			Whether to pin the conversation: 
	 *                          - `true`：Yes. 
	 *              			- `false`: No. The conversation is unpinned.
	 * @param callback 			The callback for setting whether to pin the conversation.
	 */
	public void asyncPinConversation(final String conversationId, boolean isPinned, final CallBack callback) {
		mClient.execute(()-> {
			EMAError error = new EMAError();
			emaObject.pinConversation(conversationId, isPinned, error);
			try {
				handleError(error);
				callback.onSuccess();
			} catch (ChatException e) {
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	/**
	 * \~english
	 * Loads all conversations from the local database into the memory.
	 *
	 * Generally, this method is called upon successful login to speed up the loading of the conversation list.
	 */
	public void loadAllConversations() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~english
	 * Deletes a conversation and its local messages from the local database.
	 *
	 * @param username 			The conversation ID.
	 * @param deleteMessages 	Whether to delete the historical messages with the conversation.
	 *                          - `true`: Yes.
     *                          - `false`: No.
	 * @return 					Whether the conversation is successfully deleted.
     *                          - `true`: Yes.
     *                          - `false`: No.
	 */
	public boolean deleteConversation(String username, boolean deleteMessages) {
		Conversation conv = getConversation(username);
		if (conv == null) {
			return false;
		}
		if (!deleteMessages) {
			conv.clear();
		} else {
			conv.clearAllMessages();
		}
		emaObject.removeConversation(username, deleteMessages, conv.isChatThread());
		return true;
	}

	/**
	 * \~english
	 * Deletes the specified conversation and its historical messages from the server.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 			The conversation ID.
	 * @param type              The conversation type. See {@link ConversationType}.
	 * @param isDeleteServerMessages 	Whether to delete the historical messages with the conversation.
	 *                                  - `true`: Yes
	 *                                  - `false`: No
	 * @param deleteCallBack  The callback for the deletion of the conversation and its historical messages.
	 *                          - `true`: Success;
	 *                          - `false`: Failure.
	 */
	public void deleteConversationFromServer(String username,ConversationType type, boolean isDeleteServerMessages,CallBack deleteCallBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				EMAConversationType t = EMAConversationType.CHAT;
				switch (type) {
					case Chat:
						t = EMAConversationType.CHAT;
				        break;
					case GroupChat:
						t = EMAConversationType.GROUPCHAT;
				        break;
					case ChatRoom:
						t = EMAConversationType.CHATROOM;
				        break;
					case DiscussionGroup:
						t = EMAConversationType.DISCUSSIONGROUP;
				        break;
					case HelpDesk:
						t = EMAConversationType.HELPDESK;
				        break;
				}
				EMAError error=emaObject.deleteConversationFromServer(username,t, isDeleteServerMessages);
				if(error.errCode()==EMAError.EM_NO_ERROR) {
					deleteCallBack.onSuccess();
				}else{
					deleteCallBack.onError(error.errCode(), error.errMsg());
				}
			}
		});
	}

	/**
	 * \~english
	 *
	 * Adds the message listener.
	 * 
	 * After a message listener is added, you can listen for new messages when they arrive.
	 *
	 * You can call {@link #removeMessageListener(MessageListener)} to remove the message listener when it is not required.
	 *
	 * @param listener The message listener to add. See {@link MessageListener}.
	 */
	public void addMessageListener(MessageListener listener) {
		if(listener == null){
			EMLog.d(TAG, "addMessageListener: listener is null");
			return;
		}

		if(!messageListeners.contains(listener)){
			EMLog.d(TAG, "add message listener: " + listener);
			messageListeners.add(listener);
		}
	}

	/**
	 * \~english
	 * Removes the message listener.
	 * 
	 * After adding a message listener with {@link #addMessageListener(MessageListener)}, you can call this method to remove it when it is not required.
	 * 
	 * @param listener The message listener to remove.
	 */
	public void removeMessageListener(MessageListener listener) {
		if(listener == null){
			return;
		}

		messageListeners.remove(listener);
	}

	/**
	 * \~english
	 * Adds the conversation listener. 
	 * 
	 * After a conversation listener is added, you can listen for conversation changes and conversation read receipts.
	 * 
	 * You can call the {@link #removeConversationListener(ConversationListener)} method to remove the conversation listener when it is not required.
	 *
	 * @param listener The conversation listener to add. See {@link ConversationListener}.
	 */
	public void addConversationListener(ConversationListener listener){
		if(!conversationListeners.contains(listener)){
			conversationListeners.add(listener);
		}
	}

	/**
	 * \~english
	 * Removes the conversation listener.
	 * 
	 * After adding a conversation listener with {@link #addConversationListener(ConversationListener)}, you can call this method to remove it when it is not required.
	 *
	 * @param listener The conversation listener to remove.
	 */
	public void removeConversationListener(ConversationListener listener){
		if(listener == null){
			return;
		}

		conversationListeners.remove(listener);
	}

	/**
	 * \~english
	 * Marks a voice message as listened.
	 *
	 * @param message The message object.
	 */
	public void setVoiceMessageListened(ChatMessage message)
	{
		message.setListened(true);
		updateMessage(message);
	}

	void onLogout() {
		caches.clear();
	}

	synchronized void  loadAllConversationsFromDB() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~english
	 * Changes the data of one contact in the local database to those of the other contact.
	 * 
	 * **Note**
	 * - This method brings changes to data such as the message list, the conversation list, contact list, the block list in the database.
	 * - This method does not update the data stored in the memory.
	 *
	 * @param from		The user ID of the original contact.
	 * @param changeTo  The user ID of the new contact.
	 * @return 			The operation result.
	 *                  - `true`: Success.
	 *                  - `false`: Failure.
	 */
	public boolean updateParticipant(String from, String changeTo) {
		return emaObject.updateParticipant(from, changeTo);
	}

	/**
	 * \~english
	 * Uses the pagination to get read receipts for a group message from the server.
	 * 
	 * To send a read receipt for a group message, you can call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param msgId 				The message ID.
	 * @param pageSize 				The number of read receipts for the group message that you expect to get on each page. The value range is [1,50].
	 * @param startAckId 			The starting read receipt ID for query. After this parameter is set, the SDK retrieves read receipts, from the specified one, in the reverse chronological order of when the server receives them.
	 *                              If this parameter is set as `null` or an empty string, the SDK retrieves read receipts, from the latest one, in the reverse chronological order of when the server receives them.
	 * @return 						The list of retrieved read receipts (excluding the one with the starting ID) and the cursor for the next query.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public CursorResult<GroupReadAck> fetchGroupReadAcks(String msgId, int pageSize, String startAckId) throws ChatException {
		EMAError error = new EMAError();
        CursorResult<GroupReadAck> cusorResult = new CursorResult<>();

		ChatMessage msg = getMessage(msgId);
		String groupId = null;
		if (msg.getChatType() == ChatMessage.ChatType.GroupChat && msg.isNeedGroupAck()) {
			groupId = msg.conversationId();
		} else {
			EMLog.e(TAG, "not group msg or don't need ack");
			return cusorResult;
		}

		CursorResult<EMAGroupReadAck> _cusorResult = emaObject.fetchGroupReadAcks(msgId, groupId, error, pageSize, startAckId);
		handleError(error);
		cusorResult.setCursor(_cusorResult.getCursor());

		List<GroupReadAck> groupReadAcks = new ArrayList<>();

		for(EMAGroupReadAck _ack: _cusorResult.getData()) {
			groupReadAcks.add(new GroupReadAck(_ack));
		}
		cusorResult.setData(groupReadAcks);
		return cusorResult;
	}

	/**
	 * \~english
	 * Uses the pagination to get read receipts for the group message from the server.
	 * 
	 * To send a read receipt for a group message, you can call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param msgId 		The message ID.
	 * @param pageSize 		The number of read receipts for the group message that you expect to get on each page. The value range is [1,50].
	 * @param startAckId 	The starting read receipt ID for query. After this parameter is set, the SDK retrieves read receipts, from the specified one, in the reverse chronological order of when the server receives them.
	 *                      If this parameter is set as `null` or an empty string, the SDK retrieves read receipts, from the latest one, in the reverse chronological order of when the server receives them.
	 * @param callBack 		The result callback:
	 *                      - If the call succeeds, the callback {@link ValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved read receipts (excluding the one with the starting ID) and the cursor for the next query.
	 *                      - If the call fails, the callback {@link ValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchGroupReadAcks(final String msgId, final int pageSize,
										final String startAckId, final ValueCallBack<CursorResult<GroupReadAck>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupReadAcks(msgId, pageSize, startAckId));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Uses the pagination to get the messages of all types in the conversation from the server.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 *                               - One-to-one chat: The ID of the peer user;
	 *                               - Group chat: The group ID.
	 * @param type 					The conversation type. See {@link ConversationType}.
	 * @param pageSize 				The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 			The starting message ID for query. After this parameter is set, the SDK retrieves messages, from the specified one, in the reverse chronological order of when the server receives them.
	 *                              If this parameter is set as `null` or an empty string, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
	 * @return 					    The list of retrieved messages (excluding the one with the starting message ID) and the cursor for the next query.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public CursorResult<ChatMessage> fetchHistoryMessages(String conversationId, ConversationType type, int pageSize,
	                                 String startMsgId) throws ChatException {
		return fetchHistoryMessages(conversationId, type, pageSize, startMsgId, Conversation.SearchDirection.UP);
	}

	/**
	 * \~english
	 * Uses the pagination to get the messages of all types in the conversation from the server according to the message search direction.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 *                               - One-to-one chat: The ID of the peer user;
	 *                               - Group chat: The group ID.
	 * @param type 					The conversation type. See {@link ConversationType}.
	 * @param pageSize 				The number of messages you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 			The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                              If this parameter is set as "null" or an empty string, the SDK retrieves messages according to the message search direction while ignoring this parameter.
	 *                              - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
     *                              - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the chronological order of the time the server receives them.
	 * @param direction             The message search direction. See {@link Conversation.SearchDirection}.
	 *                              - (Default) `UP`: The SDK retrieves messages in the reverse chronological order of when the server receives them.
	 * 								- `DOWN`: The SDK retrieves messages in the chronological order of the time the server receives them.
	 *
	 * @return 					    The list of retrieved messages (excluding the one with the starting message ID) and the cursor for the next query.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public CursorResult<ChatMessage> fetchHistoryMessages(String conversationId,
														  ConversationType type,
														  int pageSize,
														  String startMsgId,
														  Conversation.SearchDirection direction) throws ChatException {
		EMAError error = new EMAError();
		EMAConversation.EMASearchDirection d = direction == Conversation.SearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;
		CursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId,
				type.ordinal(), pageSize, startMsgId, d, error);
		handleError(error);
		CursorResult<ChatMessage> cursorResult = new CursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<ChatMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new ChatMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 * \~english
	 * Uses the pagination to get the messages of all types in the conversation from the server.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation ID.
	 *                           - One-to-one chat: The ID of the peer user;
	 *                           - Group chat: The group ID.
	 * @param type 				The conversation type. See {@link ConversationType}.
	 * @param pageSize 			The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 		The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, in the reverse chronological order of when the server receives them.
	 *                          If this parameter is set as "null" or an empty string, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
	 * @param callBack 			The result callback:
	 *                          - If the call succeeds, the callback {@link ValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved messages (excluding the one with the starting message ID) and the cursor for the next query.
	 *                          - If the call fails, the callback {@link ValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchHistoryMessage(final String conversationId, final ConversationType type, final int pageSize,
	                                final String startMsgId, final ValueCallBack<CursorResult<ChatMessage>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Uses the pagination to get the messages of all types in the conversation from the server according to the message search direction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation ID.
	 *                           - One-to-one chat: The ID of the peer user;
	 *                           - Group chat: The group ID.
	 * @param type 				The conversation type. See {@link ConversationType}.
	 * @param pageSize 			The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId        The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                          If this parameter is set as "null" or an empty string, the SDK retrieves messages according to the message search direction while ignoring this parameter.
	  *                         - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
     *                          - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the chronological order of when the server receives them.
	 * @param direction         The message search direction. See {@link Conversation.SearchDirection}.
	 *                          - (Default) `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 							- `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @param callBack 			The result callback:
	 *                          - If the call succeeds, the callback {@link ValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved messages (excluding the one with the starting ID) and the cursor for the next query.
	 *                          - If the call fails, the callback {@link ValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchHistoryMessage(final String conversationId,
										 final ConversationType type,
										 final int pageSize,
										 final String startMsgId,
										 final Conversation.SearchDirection direction,
										 final ValueCallBack<CursorResult<ChatMessage>> callBack) {
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId, direction));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 *  \~english
	 *
	 * Uses the pagination to get the messages in the conversation from the server according to the message pulling parameter configuration class (`FetchMessageOption`).
	 *
	 * This is a synchronous method that blocks the current thread,sdk internal use.
	 *
     * @param conversationId The conversation ID.
	 *                        - One-to-one chat: The ID of the peer user;
	 *                        - Group chat: The group ID.
	 * @param type 	The conversation type. See {@link ConversationType}.
	 * @param pageSize 	The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param cursor The cursor position from which to start querying data.
	 * @param option The parameter configuration class for pulling historical messages from the server. See {@link FetchMessageOption}.
	 * @return The list of retrieved messages and the cursor for the next query.
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	private CursorResult<ChatMessage>  fetchHistoryMessages(String conversationId,
															ConversationType type,
															int pageSize, String cursor,
															FetchMessageOption option) throws ChatException {
		EMAError error=new EMAError();
		CursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId, type.ordinal(), pageSize, cursor, option==null?new EMAFetchMessageOption():option.emaObject, error);
		handleError(error);

		CursorResult<ChatMessage> cursorResult = new CursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<ChatMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new ChatMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 /**
	 *  \~english
	 *
	 * Uses the pagination to get the messages in the conversation from the server according to the message pulling parameter configuration class (`FetchMessageOption`).
	 *
	 * This is a asynchronous method.
	 *
	 
	 * @param conversationId The conversation ID.
	 *                        - One-to-one chat: The ID of the peer user;
	 *                        - Group chat: The group ID.
	 * @param type 	The conversation type. See {@link ConversationType}.
	 * @param pageSize 	The number of messages you expect to get on each page. The value range is [1,50].
	 * @param cursor The cursor position from which to start querying data.
	 * @param option The parameter configuration class for querying historical messages. See {@link FetchMessageOption}.
	 * @return The list of retrieved messages and the cursor for the next query.
	 * @param callBack 	The result callback:
	 *                  - If the call succeeds, the callback {@link ValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved messages and the cursor for the next query.
	 *                  - If the call fails, the callback {@link ValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchHistoryMessages(String conversationId,
										 ConversationType type,
										 int pageSize,String cursor,
										 FetchMessageOption option,
										 final ValueCallBack<CursorResult<ChatMessage>> callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, cursor,option));
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private void handleError(EMAError error) throws ChatException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new ChatException(error);
		}
	}

	private static final int LIST_SIZE = 512;

	/**
	 * \~english
	 * Retrieves messages of a certain type in the conversation from the local database.
	 *
	 * @param type       The message type. See {@link Type}.
	 * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
	 * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
	 * @param from       The user ID of the message sender in one-to-one chat or group chat.
	 *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
	 * @param direction  The message search direction. See {@link Conversation.SearchDirection}.
	 *                   - (Default) `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @return           The list of retrieved messages (excluding the one with the starting timestamp). If no message is obtained, an empty list is returned.
	 */
	public List<ChatMessage> searchMsgFromDB(Type type, long timeStamp, int maxCount, String from, Conversation.SearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == Conversation.SearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<ChatMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<ChatMessage>();
		} else {
			result = new ArrayList<ChatMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new ChatMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~english
	 * Retrieves messages with keywords in the conversation in the local database.
	 *
	 * @param keywords   The keywords for query.
	 * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
	 * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
	 * @param from       The user ID of the message sender in one-to-one chat or group chat.
	 *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
	 * @param direction  The message search direction. See {@link Conversation.SearchDirection}.
	 *                   - (Default) `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @return            The list of retrieved messages (excluding the one with the starting timestamp). If no message is obtained, an empty list is returned.
	 */
	public List<ChatMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, Conversation.SearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == Conversation.SearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<ChatMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<ChatMessage>();
		} else {
			result = new ArrayList<ChatMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new ChatMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~english
	 * Deletes local historical messages with a Unix timestamp before a specified one.
	 *
	 * @param timeStamp 			The starting Unix timestamp in the message for query. This unit is millisecond.
	 * @param callback 				The result callback. See {@link CallBack}.
	 */
	public void deleteMessagesBeforeTimestamp(long timeStamp, CallBack callback){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				boolean result = emaObject.removeMessagesBeforeTimestamp(timeStamp);
				if(result){
					for(Conversation.MessageCache cache : caches.values()){
						for(ChatMessage message : cache.getAllMessages()){
							if(message.getMsgTime() < timeStamp){
								cache.removeMessage(message.getMsgId());
							}
						}
					}
					callback.onSuccess();
				}else{
					callback.onError(Error.DATABASE_ERROR, "Database operation failed");
				}
			}
		});
	}

	/**
	 * \~english
	 * Reports an inappropriate message.
	 *
	 * This is an asynchronous method.
	 *
	 * @param msgId		The ID of the message to report.
	 * @param tag		The tag of the inappropriate message. You need to type a custom tag, like `porn` or `ad`.
	 * @param reportReason		The reporting reason. You need to type a specific reason.
	 * @param callBack 			The result callback. See {@link CallBack}.
	 *                          - If the call succeeds, {@link CallBack#onSuccess()} is triggered;
	 *                          - If the call fails, {@link CallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncReportMessage(String msgId, String tag, String reportReason,CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					reportMessage(msgId, tag, reportReason);
					callBack.onSuccess();
				} catch (ChatException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~english
	 * Reports an inappropriate message.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param msgId				The ID of the inappropriate message.
	 * @param tag        The tag of the inappropriate message. You need to type a custom tag, like `porn` or `ad`.
	 * @param reportReason		The reporting reason. You need to type a specific reason.
	 *
	 * @throws ChatException A description of the exception. See {@link Error}.
	 */
	public void reportMessage(String msgId, String tag, String reportReason ) throws ChatException {
		EMAError error = new EMAError();
		emaObject.reportMessage(msgId,tag,reportReason,error);
		handleError(error);
	}

	/**
	 * 获取翻译服务支持的语言。
	 *
	 * @param callBack 结果回调，详见 {@link CallBack}。
	 *                 - 调用成功，触发 {@link CallBack#onSuccess()} 返回获取的语言；
	 *                 - 调用失败，触发 {@link CallBack#onError(int, String)} 报错。
	 *
	 * \~english
	 * Gets all languages supported by the translation service.
	 *
	 * @param callBack The result callback. See {@link CallBack}.
	 *                 - If the call succeeds, {@link CallBack#onSuccess()} is triggered to return the obtained languages;
	 *                 - If the call fails, {@link CallBack#onError(int, String)} is triggered to report an error.
	 */
	public void fetchSupportLanguages(ValueCallBack<List<Language>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					List<Language> languages = new ArrayList<>();
					EMAError error = new EMAError();
					List<List<String>> result = emaObject.fetchSupportLanguages(error);
					handleError(error);
					for(List<String> list : result){
						languages.add(new Language(list.get(0), list.get(1), list.get(2)));
					}
					callBack.onSuccess(languages);
				} catch (ChatException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Translates a text message.
	 *
	 * @param message The message object for translation.
	 * @param languages The list of target language codes.
	 * @param callBack The result callback. See {@link CallBack}.
	 *                 - If the call succeeds, {@link CallBack#onSuccess()} is triggered to return the translation;
	 *                 - If the call fails, {@link CallBack#onError(int, String)} is triggered to report an error.
	 */
	public void translateMessage(ChatMessage message, List<String> languages, ValueCallBack<ChatMessage> callBack){
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try{
					EMAError error = new EMAError();
					EMAMessage emaMessage = emaObject.translateMessage(message.emaObject, languages, error);
					handleError(error);
					callBack.onSuccess(new ChatMessage(emaMessage));
				}catch(ChatException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~english
	 * Adds a Reaction.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction content.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void addReaction(final String messageId, final String reaction) throws ChatException {
		EMAError error = new EMAError();
		emaReactionObject.addReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Adds a Reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The message Reaction to add.
	 * @param callback  The result callback which contains the error information if the method fails. See {@link CallBack}.
	 */
	public void asyncAddReaction(final String messageId, final String reaction, final CallBack callback) {
		ChatClient.getInstance().execute(() -> {
			try {
				addReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (ChatException e) {
				EMLog.e(TAG, "asyncAddReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Deletes a Reaction.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction content to delete.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public void removeReaction(final String messageId, final String reaction) throws ChatException {
		EMAError error = new EMAError();
		emaReactionObject.removeReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~english
	 * Deletes a Reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction to delete.
	 * @param callback  The result callback which contains the error information if the method fails. See {@link CallBack}.
	 */
	public void asyncRemoveReaction(final String messageId, final String reaction, final CallBack callback) {
		ChatClient.getInstance().execute(() -> {
			try {
				removeReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (ChatException e) {
				EMLog.e(TAG, "asyncRemoveReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The conversation type. Only the one-to-one chats ({@link ChatMessage.ChatType#Chat} and group chats ({@link ChatMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is invalid only for group chats.
	 * @return map           The Reaction list under the specified message ID. The `UserList` of `MessageReaction` indicates the summary data, which only contains the information of the first three users.
	 * @throws ChatException	A description of the exception. See {@link Error}.
	 */
	public Map<String, List<MessageReaction>> getReactionList(final List<String> messageIdList, final ChatMessage.ChatType chatType, final String groupId) throws ChatException {
		EMAError error = new EMAError();
		String chatTypeStr = "chat";
		if (ChatMessage.ChatType.Chat == chatType) {
			chatTypeStr = "chat";
		} else if (ChatMessage.ChatType.GroupChat == chatType) {
			chatTypeStr = "groupchat";
		}

		Map<String, List<EMAMessageReaction>> emaReactionMap = emaReactionObject.getReactionList(messageIdList, chatTypeStr, groupId, error);
		handleError(error);

		Map<String, List<MessageReaction>> results = new HashMap<>(emaReactionMap.size());
		if (emaReactionMap.size() > 0) {
			List<MessageReaction> reactionList;
			for (Map.Entry<String, List<EMAMessageReaction>> entry : emaReactionMap.entrySet()) {
				reactionList = new ArrayList<>(entry.getValue().size());
				for (EMAMessageReaction emaMessageReaction : entry.getValue()) {
					if (null != emaMessageReaction) {
						reactionList.add(new MessageReaction(emaMessageReaction));
					}
				}
				results.put(entry.getKey(), reactionList);
			}
		}
		return results;
	}

	/**
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The conversation type. Only one-to-one chats ({@link ChatMessage.ChatType#Chat} and group chats ({@link ChatMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is valid only for group chats.
	 * @param callback       The Reaction list under the specified message ID. The `UserList` of `MessageReaction` indicates the summary data, which only contains the information of the first three users.
	 */
	public void asyncGetReactionList(final List<String> messageIdList, final ChatMessage.ChatType chatType, final String groupId, final ValueCallBack<Map<String, List<MessageReaction>>> callback) {
		ChatClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionList(messageIdList, chatType, groupId));
				}
			} catch (ChatException e) {
				EMLog.e(TAG, "asyncGetReactionList error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	/**
	 * \~english
	 * Gets the Reaction details.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The Reaction content.
	 * @param cursor       The cursor position from which to start querying data.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @return   The query result {@link CursorResult}, which contains the list of obtained Reactions and the cursor for the next query. If the cursor is `null`, all data is obtained.
	 * @throws ChatException	A description of the exception, see {@link Error}.
	 */
	public CursorResult<MessageReaction> getReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize) throws ChatException {
		EMAError error = new EMAError();
		CursorResult<EMAMessageReaction> result = emaReactionObject.getReactionDetail(messageId, reaction, cursor, pageSize, error);
		handleError(error);
		CursorResult<MessageReaction> cursorResult = new CursorResult<>();
		cursorResult.setCursor(result.getCursor());
		if (null != result.getData()) {
			List<MessageReaction> messageReactionList = new ArrayList<>(result.getData().size());
			for (EMAMessageReaction emaMessageReaction : result.getData()) {
				if (null != emaMessageReaction) {
					messageReactionList.add(new MessageReaction(emaMessageReaction));
				}
			}
			cursorResult.setData(messageReactionList);
		}
		return cursorResult;
	}

	/**
	 * \~english
	 * Gets the Reaction details.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The Reaction content.
	 * @param cursor       The cursor position from which to start querying data.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @param callback     The query result {@link CursorResult}, which contains the cursor for the next query and MessageReaction list (only the first data entry is used).
	 */
	public void asyncGetReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize, final ValueCallBack<CursorResult<MessageReaction>> callback) {
		ChatClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionDetail(messageId, reaction, cursor, pageSize));
				}
			} catch (ChatException e) {
				EMLog.e(TAG, "asyncGetReactionDetail error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	/**
	 * \~english
	 * Unidirectionally removes historical message by message ID from the server.
	 *
	 * @param conversationId		The conversation ID.
	 * @param type					The conversation type.
	 * @param msgIdList				The list of IDs of messages to be removed.
	 * @param callBack				The result callback which contains the error information if the method fails. See {@link CallBack}.
	 */
	protected void removeMessagesFromServer(String conversationId, ConversationType type, List<String> msgIdList, CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				EMAError error = emaObject.deleteRoamMsgFromServerById(conversationId,type,msgIdList);
				if(error.errCode()==EMAError.EM_NO_ERROR) {
					clearCaches(conversationId,msgIdList);
					if (callBack != null){
						callBack.onSuccess();
					}
				}else{
					if (callBack != null){
						callBack.onError(error.errCode(), error.errMsg());
					}
				}
			}
		});
	}

	/**
	 * \~english
	 * Unidirectionally removes historical message by timestamp from the server.
	 *
	 * @param conversationId		The conversation ID.
	 * @param type					The conversation type.
	 * @param beforeTimeStamp		The UNIX timestamp in millisecond. Messages with a timestamp smaller than the specified one will be removed.
	 * @param callBack				The result callback which contains the error information if the method fails. See {@link CallBack}.
	 */
	protected void removeMessagesFromServer(String conversationId, ConversationType type, long beforeTimeStamp, CallBack callBack){
		ChatClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				EMAError error =  emaObject.deleteRoamMsgFromServerByTime(conversationId,type,beforeTimeStamp);
				if(error.errCode()==EMAError.EM_NO_ERROR) {
					clearCaches(conversationId,beforeTimeStamp);
					if (callBack != null){
						callBack.onSuccess();
					}
				}else{
					if (callBack != null){
						callBack.onError(error.errCode(), error.errMsg());
					}
				}
			}
		});
	}

	void clearCaches(String conversationId,long beforeTimeStamp){
		if (caches.containsKey(conversationId) && caches.get(conversationId) != null){
			if (caches.get(conversationId).getAllMessages().size() > 0){
				for (ChatMessage message : caches.get(conversationId).getAllMessages()) {
					if(message.getMsgTime() <= beforeTimeStamp){
						caches.get(conversationId).removeMessage(message.getMsgId());
					}
				}
			}
		}
	}

	void clearCaches(String conversationId,List<String> msgIdList){
		if (msgIdList != null && msgIdList.size() > 0){
			for (String msgId : msgIdList) {
				if (caches.containsKey(conversationId) && caches.get(conversationId) != null){
					caches.get(conversationId).removeMessage(msgId);
				}
			}
		}
	}

}
