/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.service.ocsp;

import eu.europa.esig.dss.alert.ExceptionOnStatusAlert;
import eu.europa.esig.dss.alert.LogOnStatusAlert;
import eu.europa.esig.dss.alert.SilentOnStatusAlert;
import eu.europa.esig.dss.alert.StatusAlert;
import eu.europa.esig.dss.alert.status.MessageStatus;
import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.enumerations.RevocationOrigin;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.model.x509.revocation.ocsp.OCSP;
import eu.europa.esig.dss.service.NonceSource;
import eu.europa.esig.dss.service.http.commons.OCSPDataLoader;
import eu.europa.esig.dss.spi.CertificateExtensionsUtils;
import eu.europa.esig.dss.spi.DSSRevocationUtils;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.DataLoader;
import eu.europa.esig.dss.spi.exception.DSSExternalResourceException;
import eu.europa.esig.dss.spi.x509.revocation.OnlineRevocationSource;
import eu.europa.esig.dss.spi.x509.revocation.RevocationSourceAlternateUrlsSupport;
import eu.europa.esig.dss.spi.x509.revocation.RevocationToken;
import eu.europa.esig.dss.spi.x509.revocation.ocsp.OCSPRespStatus;
import eu.europa.esig.dss.spi.x509.revocation.ocsp.OCSPSource;
import eu.europa.esig.dss.spi.x509.revocation.ocsp.OCSPToken;
import eu.europa.esig.dss.utils.Utils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.cert.ocsp.OCSPException;
import org.bouncycastle.cert.ocsp.OCSPReq;
import org.bouncycastle.cert.ocsp.OCSPReqBuilder;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class OnlineOCSPSource
implements OCSPSource,
RevocationSourceAlternateUrlsSupport<OCSP> {
    private static final Logger LOG = LoggerFactory.getLogger(OnlineOCSPSource.class);
    private NonceSource nonceSource;
    private DataLoader dataLoader;
    private DigestAlgorithm certIDDigestAlgorithm = DigestAlgorithm.SHA1;
    private StatusAlert alertOnInvalidNonce = new ExceptionOnStatusAlert();
    private StatusAlert alertOnNonexistentNonce = new LogOnStatusAlert(Level.WARN);
    private StatusAlert alertOnInvalidUpdateTime = new SilentOnStatusAlert();
    private long nextUpdateTolerancePeriod = 0L;

    public OnlineOCSPSource() {
        this.dataLoader = new OCSPDataLoader();
        LOG.trace("+OnlineOCSPSource with the default data loader.");
    }

    public OnlineOCSPSource(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
        LOG.trace("+OnlineOCSPSource with the specific data loader.");
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setNonceSource(NonceSource nonceSource) {
        this.nonceSource = nonceSource;
    }

    public void setCertIDDigestAlgorithm(DigestAlgorithm certIDDigestAlgorithm) {
        Objects.requireNonNull(certIDDigestAlgorithm, "The certIDDigestAlgorithm must not be null!");
        this.certIDDigestAlgorithm = certIDDigestAlgorithm;
    }

    public void setAlertOnInvalidNonce(StatusAlert alertOnInvalidNonce) {
        this.alertOnInvalidNonce = alertOnInvalidNonce;
    }

    public void setAlertOnNonexistentNonce(StatusAlert alertOnNonexistentNonce) {
        this.alertOnNonexistentNonce = alertOnNonexistentNonce;
    }

    public void setAlertOnInvalidUpdateTime(StatusAlert alertOnInvalidUpdateTime) {
        this.alertOnInvalidUpdateTime = alertOnInvalidUpdateTime;
    }

    public void setNextUpdateTolerancePeriod(long nextUpdateTolerancePeriod) {
        this.nextUpdateTolerancePeriod = nextUpdateTolerancePeriod;
    }

    public OCSPToken getRevocationToken(CertificateToken certificateToken, CertificateToken issuerCertificateToken) {
        return this.getRevocationToken(certificateToken, issuerCertificateToken, Collections.emptyList());
    }

    public OCSPToken getRevocationToken(CertificateToken certificateToken, CertificateToken issuerCertificateToken, List<String> alternativeUrls) {
        Objects.requireNonNull(certificateToken, "CertificateToken cannot be null!");
        Objects.requireNonNull(issuerCertificateToken, "Issuer CertificateToken cannot be null!");
        Objects.requireNonNull(this.dataLoader, "DataLoader is not provided !");
        LOG.trace("--> OnlineOCSPSource queried for {}", (Object)certificateToken.getDSSIdAsString());
        List<String> ocspUrls = this.getOCSPAccessURLs(certificateToken, alternativeUrls);
        if (Utils.isCollectionEmpty(ocspUrls)) {
            throw new DSSExternalResourceException(String.format("No OCSP location found for certificate with Id '%s'", certificateToken.getDSSIdAsString()));
        }
        byte[] nonce = null;
        if (this.nonceSource != null) {
            nonce = this.nonceSource.getNonceValue();
        }
        byte[] content = this.buildOCSPRequest(certificateToken, issuerCertificateToken, nonce);
        int nbTries = ocspUrls.size();
        for (String ocspAccessLocation : ocspUrls) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Trying to retrieve an OCSP response from URL '{}'...", (Object)ocspAccessLocation);
            }
            --nbTries;
            try {
                BasicOCSPResp basicResponse = this.executeOCSPRequest(ocspAccessLocation, content);
                SingleResp latestSingleResponse = DSSRevocationUtils.getLatestSingleResponse((BasicOCSPResp)basicResponse, (CertificateToken)certificateToken, (CertificateToken)issuerCertificateToken);
                this.assertOCSPResponseValid(basicResponse, latestSingleResponse, nonce);
                OCSPToken ocspToken = new OCSPToken(basicResponse, latestSingleResponse, certificateToken, issuerCertificateToken);
                ocspToken.setSourceURL(ocspAccessLocation);
                ocspToken.setExternalOrigin(RevocationOrigin.EXTERNAL);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("OCSP Response '{}' has been retrieved from a source with URL '{}'.", (Object)ocspToken.getDSSIdAsString(), (Object)ocspAccessLocation);
                }
                return ocspToken;
            }
            catch (Exception e) {
                if (nbTries == 0) {
                    throw new DSSExternalResourceException(String.format("Unable to retrieve OCSP response for certificate with Id '%s' from URL '%s'. Reason : %s", certificateToken.getDSSIdAsString(), ocspAccessLocation, e.getMessage()), (Throwable)e);
                }
                LOG.warn("Unable to retrieve OCSP response with URL '{}' : {}", (Object)ocspAccessLocation, (Object)e.getMessage());
            }
        }
        throw new IllegalStateException(String.format("Invalid state within OnlineOCSPSource for a certificate call with id '%s'", certificateToken.getDSSIdAsString()));
    }

    protected List<String> getOCSPAccessURLs(CertificateToken certificateToken, List<String> alternativeUrls) {
        if (Utils.isCollectionNotEmpty(alternativeUrls)) {
            LOG.info("OCSP alternative urls : {}", alternativeUrls);
        }
        List ocspAccessUrls = CertificateExtensionsUtils.getOCSPAccessUrls((CertificateToken)certificateToken);
        ArrayList<String> ocspUrls = new ArrayList<String>();
        ocspUrls.addAll(ocspAccessUrls);
        ocspUrls.addAll(alternativeUrls);
        return ocspUrls;
    }

    @Deprecated
    public OnlineRevocationSource.RevocationTokenAndUrl<OCSP> getRevocationTokenAndUrl(CertificateToken certificateToken, CertificateToken issuerToken) {
        OCSPToken revocationToken = this.getRevocationToken(certificateToken, issuerToken);
        if (revocationToken != null) {
            return new OnlineRevocationSource.RevocationTokenAndUrl(revocationToken.getSourceURL(), (RevocationToken)revocationToken);
        }
        return null;
    }

    @Deprecated
    protected OnlineRevocationSource.RevocationTokenAndUrl<OCSP> getRevocationTokenAndUrl(CertificateToken certificateToken, CertificateToken issuerToken, List<String> ocspUrls) {
        OCSPToken revocationToken = this.getRevocationToken(certificateToken, issuerToken, ocspUrls);
        if (revocationToken != null) {
            return new OnlineRevocationSource.RevocationTokenAndUrl(revocationToken.getSourceURL(), (RevocationToken)revocationToken);
        }
        return null;
    }

    protected byte[] buildOCSPRequest(CertificateToken certificateToken, CertificateToken issuerToken, byte[] nonce) {
        try {
            OCSPReqBuilder ocspReqBuilder = new OCSPReqBuilder();
            CertificateID certId = DSSRevocationUtils.getOCSPCertificateID((CertificateToken)certificateToken, (CertificateToken)issuerToken, (DigestAlgorithm)this.certIDDigestAlgorithm);
            ocspReqBuilder.addRequest(certId);
            if (nonce != null) {
                DEROctetString encodedNonceValue = new DEROctetString(nonce);
                Extension extension = new Extension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, (ASN1OctetString)new DEROctetString((ASN1Encodable)encodedNonceValue));
                Extensions extensions = new Extensions(extension);
                ocspReqBuilder.setRequestExtensions(extensions);
            }
            OCSPReq ocspReq = ocspReqBuilder.build();
            return ocspReq.getEncoded();
        }
        catch (IOException | OCSPException e) {
            throw new DSSException("Cannot build OCSP Request", e);
        }
    }

    @Deprecated
    protected byte[] buildOCSPRequest(CertificateID certId, byte[] nonce) throws DSSException {
        try {
            OCSPReqBuilder ocspReqBuilder = new OCSPReqBuilder();
            ocspReqBuilder.addRequest(certId);
            if (nonce != null) {
                DEROctetString encodedNonceValue = new DEROctetString(nonce);
                Extension extension = new Extension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, (ASN1OctetString)new DEROctetString((ASN1Encodable)encodedNonceValue));
                Extensions extensions = new Extensions(extension);
                ocspReqBuilder.setRequestExtensions(extensions);
            }
            OCSPReq ocspReq = ocspReqBuilder.build();
            return ocspReq.getEncoded();
        }
        catch (IOException | OCSPException e) {
            throw new DSSException("Cannot build OCSP Request", e);
        }
    }

    protected BasicOCSPResp executeOCSPRequest(String ocspAccessLocation, byte[] request) throws IOException, OCSPException {
        byte[] ocspRespBytes = this.dataLoader.post(ocspAccessLocation, request);
        if (Utils.isArrayNotEmpty((byte[])ocspRespBytes)) {
            OCSPResp ocspResp;
            OCSPRespStatus status;
            if (LOG.isTraceEnabled()) {
                LOG.trace(String.format("Obtained OCSPResponse binaries from URL '%s' : %s", ocspAccessLocation, Utils.toBase64((byte[])ocspRespBytes)));
            }
            if (!OCSPRespStatus.SUCCESSFUL.equals((Object)(status = OCSPRespStatus.fromInt((int)(ocspResp = new OCSPResp(ocspRespBytes)).getStatus())))) {
                throw new DSSExternalResourceException(String.format("Ignored OCSP Response from URL '%s' : status -> %s", ocspAccessLocation, status));
            }
            Object responseObject = ocspResp.getResponseObject();
            if (!(responseObject instanceof BasicOCSPResp)) {
                throw new DSSExternalResourceException(String.format("OCSP Response Object shall be of type BasicOCSPResp! Obtained type : %s", responseObject.getClass().getSimpleName()));
            }
            return (BasicOCSPResp)responseObject;
        }
        throw new DSSExternalResourceException(String.format("OCSP DataLoader for certificate with url '%s' responded with an empty byte array!", ocspAccessLocation));
    }

    protected void assertOCSPResponseValid(BasicOCSPResp basicOCSPResp, SingleResp latestSingleResponse, byte[] expectedNonce) {
        if (expectedNonce != null) {
            byte[] receivedNonce = this.getEmbeddedNonceValue(basicOCSPResp);
            if (receivedNonce == null) {
                this.alertOnNonexistentNonce();
            } else {
                boolean nonceMatch = Arrays.equals(expectedNonce, receivedNonce);
                if (nonceMatch) {
                    return;
                }
                this.alertOnInvalidNonce(expectedNonce, receivedNonce);
            }
        }
        this.assertUpdateTimeValid(latestSingleResponse);
    }

    private byte[] getEmbeddedNonceValue(BasicOCSPResp basicOCSPResp) {
        try {
            Extension extension = basicOCSPResp.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
            if (extension != null) {
                ASN1OctetString extnValue = extension.getExtnValue();
                ASN1Primitive value = this.fromByteArray(extnValue);
                if (value instanceof DEROctetString) {
                    return ((DEROctetString)value).getOctets();
                }
                throw new OCSPException("Nonce extension value in OCSP response is not an OCTET STRING");
            }
            return null;
        }
        catch (Exception e) {
            throw new DSSExternalResourceException(String.format("Unable to extract the nonce from the OCSPResponse! Reason : [%s]", e.getMessage()), (Throwable)e);
        }
    }

    private ASN1Primitive fromByteArray(ASN1OctetString extnValue) throws OCSPException {
        try {
            return ASN1Primitive.fromByteArray((byte[])extnValue.getOctets());
        }
        catch (IOException ex) {
            throw new OCSPException("Invalid encoding of nonce extension value in OCSP response", (Throwable)ex);
        }
    }

    private void assertUpdateTimeValid(SingleResp singleResponse) {
        Date thisUpdate = singleResponse.getThisUpdate();
        if (thisUpdate == null) {
            this.alertOnInvalidUpdateTime("Obtained OCSP Response does not contain thisUpdate field!");
            return;
        }
        Date nextUpdate = singleResponse.getNextUpdate();
        if (nextUpdate == null) {
            this.alertOnInvalidUpdateTime("Obtained OCSP Response does not contain nextUpdate field!");
            return;
        }
        Date currentTime = new Date();
        long nextUpdateLimit = nextUpdate.getTime() + this.nextUpdateTolerancePeriod;
        if (currentTime.before(thisUpdate) || currentTime.after(new Date(nextUpdateLimit))) {
            this.alertOnInvalidUpdateTime(currentTime, thisUpdate, nextUpdate);
        }
    }

    private void alertOnNonexistentNonce() {
        MessageStatus status = new MessageStatus();
        status.setMessage("No nonce has been retrieved from OCSP response!");
        this.alertOnNonexistentNonce.alert((Object)status);
    }

    private void alertOnInvalidNonce(byte[] expectedNonce, byte[] receivedNonce) {
        MessageStatus status = new MessageStatus();
        status.setMessage(String.format("Nonce retrieved from OCSP response '#%s' does not match a dispatched nonce '#%s'.", Utils.toHex((byte[])receivedNonce), Utils.toHex((byte[])expectedNonce)));
        this.alertOnInvalidNonce.alert((Object)status);
    }

    private void alertOnInvalidUpdateTime(String message) {
        MessageStatus status = new MessageStatus();
        status.setMessage(message);
        this.alertOnInvalidUpdateTime.alert((Object)status);
    }

    private void alertOnInvalidUpdateTime(Date currentTime, Date thisUpdate, Date nextUpdate) {
        this.alertOnInvalidUpdateTime(String.format("The current time '%s' is out of thisUpdate '%s' - nextUpdate '%s' range!", DSSUtils.formatDateToRFC((Date)currentTime), DSSUtils.formatDateToRFC((Date)thisUpdate), DSSUtils.formatDateToRFC((Date)nextUpdate)));
    }
}

