package com.xebialabs.xltest.xunit;


import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.UUID;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import com.xebialabs.xltest.domain.*;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.xebialabs.overthere.OverthereFile;
import org.xml.sax.SAXException;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.xltest.importers.ImporterUtil.sendEvent;
import static java.lang.Math.max;
import static java.lang.String.format;

public class XUnitReportXmlImporter {

    public static final String TESTSUITE_TAG = "testsuite";
    public static final String TESTCASE_TAG = "testcase";

    private static final DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();

    private final DocumentBuilder documentBuilder;
    private final DateTimeFormatter isoDateTimeFormatter = ISODateTimeFormat.dateHourMinuteSecond();
    private final OverthereFile xunitReportXmlFile;
    private final String moduleName;
    private NodeList testSuites;

    public XUnitReportXmlImporter(OverthereFile xunitReportXmlFile) {
        this("", xunitReportXmlFile);
    }

    public XUnitReportXmlImporter(String moduleName, OverthereFile xunitReportXmlFile) {
        this.moduleName = moduleName;
        this.xunitReportXmlFile = xunitReportXmlFile;
        this.documentBuilder = newDocumentBuilder();
    }

    private DocumentBuilder newDocumentBuilder() {
        try {
            return documentBuilderFactory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new RuntimeException("Unable to create an XML document builder", e);
        }
    }

    public long getLastModified() throws ImportFailedException, NothingToImportException {
        long lastModified = 0;

        parseXml();

        for (int i = 0; i < testSuites.getLength(); i++) {
            Element testSuite = (Element) testSuites.item(i);
            lastModified = max(lastModified, getLastModifiedInMilliseconds(testSuite));
        }

        return lastModified;
    }

    public int doImport(UUID testRunId, EventHandler eventHandler) throws ImportException {
        int duration = 0;

        parseXml();

        String sourceFile = xunitReportXmlFile.getName();
        for (int i = 0; i < testSuites.getLength(); i++) {
            Element testSuite = (Element) testSuites.item(i);
            duration += getDurationInMilliseconds(testSuite);
            createEventsForTestResults(testSuite, testRunId, eventHandler, sourceFile);
        }
        return duration;
    }

    private void parseXml() throws ImportFailedException, NothingToImportException {
        if (testSuites != null) {
            return;
        }

        try (InputStream fis = xunitReportXmlFile.getInputStream()) {
            Document doc = documentBuilder.parse(fis);
            testSuites = doc.getElementsByTagName(TESTSUITE_TAG);
        } catch (IOException | SAXException e) {
            throw new ImportFailedException(format("Unable to parse file [%s]: %s", this.xunitReportXmlFile, e.getMessage()), e);
        }

        if (testSuites.getLength() == 0) {
            throw new NothingToImportException(format("Nothing to import for file [%s]: No test suites found.", xunitReportXmlFile));
        }
    }

    private void createEventsForTestResults(Element testSuite, UUID testRunId, EventHandler eventHandler, String testResultFilename) throws ImportException {
        String prefix = (isNullOrEmpty(moduleName) ? "" : (moduleName + Event.TESTNAME_DELIMITER)) + formatSuiteName(testSuite.getAttribute("name"));

        NodeList testCases = testSuite.getElementsByTagName(TESTCASE_TAG);

        final int length = testCases.getLength();
        for (int i = 0; i < length; i++) {
            Element testCase = (Element) testCases.item(i);
            TestCaseResult testCaseResult = testCaseResult(testCase);
            if (testCaseResult.passed) {
                sendEvent(eventHandler, testRunId, Event.FUNCTIONAL_RESULT, Event.props(
                        "result", "PASSED",
                        "duration", getDurationInMilliseconds(testCase),
                        "name", prefix + Event.TESTNAME_DELIMITER + testCase.getAttribute("name"),
                        "fileName", testResultFilename
                ));
            } else {
                sendEvent(eventHandler, testRunId, Event.FUNCTIONAL_RESULT, Event.props(
                        "result", "FAILED",
                        "duration", getDurationInMilliseconds(testCase),
                        "name", prefix + Event.TESTNAME_DELIMITER + testCase.getAttribute("name"),
                        "fileName", testResultFilename,
                        "firstError", testCaseResult.firstError
                ));
            }
        }
    }

    private TestCaseResult testCaseResult(Element testCase) {
        NodeList failureNodes = testCase.getElementsByTagName("failure");
        if (failureNodes == null || failureNodes.getLength() == 0) {
            failureNodes = testCase.getElementsByTagName("error");
        }
        if (failureNodes == null || failureNodes.getLength() == 0) {
            return TestCaseResult.PASSED();
        }
        Node firstError = failureNodes.item(0).getAttributes().getNamedItem("message");
        return TestCaseResult.FAILED(
                firstError != null ?
                    firstError.getNodeValue()
                :
                    failureNodes.item(0).getTextContent()
        );
    }

    private int getDurationInMilliseconds(Element testCase) {
        // time attribute is in seconds
        String duration = testCase.getAttribute("time");
        if (duration == null || "".equals(duration)) {
            return 0;
        }
        Float f = Float.parseFloat(duration);
        return (int) (f * 1000.0f);

    }

    private long getLastModifiedInMilliseconds(Element testCase) {
        // time attribute is in seconds
        String timestamp = testCase.getAttribute("timestamp");
        if (timestamp == null || "".equals(timestamp)) {
            return 0;
        }
        SimpleDateFormat format = new SimpleDateFormat("dd MMM yyyy hh:mm:ss Z");
        format.setTimeZone(TimeZone.getTimeZone("GMT"));

        try {
            return format.parse(timestamp).getTime();
        } catch (ParseException e) {
            //It seems that jUnit always writes utc, even if the timestamp has no timezone information (This is against the ISO8601 spec)
            return isoDateTimeFormatter.withZoneUTC().parseMillis(timestamp);
        }
    }

    /**
     * Try to split a suite name into a package name and a class name, as is done
     * in jUnit by default.
     *
     * @param name
     * @return a formatted name (with semicolon where appropriate)
     */
    static String formatSuiteName(String name) {
        int lastDot = name.lastIndexOf('.');
        if (lastDot < 0) {
            return name;
        }

        String packageName = name.substring(0, lastDot);
        String className = name.substring(lastDot + 1);

        if (isNullOrEmpty(packageName) || isNullOrEmpty(className)) {
            return name;
        }

        return packageName + Event.TESTNAME_DELIMITER + className;
    }

    private static class TestCaseResult {
        private final boolean passed;
        private final String firstError;

        private TestCaseResult(boolean passed, String firstError) {
            this.passed = passed;
            this.firstError = firstError;
        }

        private static TestCaseResult PASSED() {
            return new TestCaseResult(true, null);
        }

        private static TestCaseResult FAILED(String firstError) {
            return new TestCaseResult(false, firstError);
        }
    }
}
