package com.xebialabs.xltest.xunit;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

import static com.xebialabs.xltest.importers.ImporterUtil.sendEvent;
import static java.lang.String.format;

/**
 * The importer will import all files related to a (test) run. In order to determine the last-modified time
 * (used to prevent multiple imports), the files are sorted alphabetically and the time of the first file in the list
 * is used.
 */
public class XUnitImporter implements Importable {
    private static Logger LOG = LoggerFactory.getLogger(XUnitImporter.class);

    private final OverthereFile searchPath;
    private final String searchPattern;
    private final Collection<OverthereFile> testResultFiles;
    private final long lastModified;

    public XUnitImporter(OverthereFile searchPath, String searchPattern, Collection<OverthereFile> testResultFiles) throws ImportException {
        this.searchPath = searchPath;
        this.searchPattern = searchPattern;
        this.testResultFiles = sorted(testResultFiles);
        this.lastModified = validateTestResultFilesAndReturnLastModifiedTime();
    }

    private long validateTestResultFilesAndReturnLastModifiedTime() throws ImportException {
        // At least one file should be a valid JUnit XML file.
        long lastModified = -1L;

        for (OverthereFile testResult : testResultFiles) {
            XUnitReportXmlImporter xmlImporter = new XUnitReportXmlImporter(testResult);
            long ts = xmlImporter.getLastModified();
            if (ts == 0L) {
                LOG.debug("Timestamp is 0 for file; falling back to file last modified.");
                ts = testResult.lastModified();
            }
            if (ts > lastModified) {
                lastModified = ts;
            }
        }

        if (lastModified == -1L) {
            throw new NothingToImportException(format("No valid test results found in set %s", testResultFiles));
        }
        return lastModified;
    }

    @Override
    public void doImport(UUID testRunId, EventHandler eventHandler) throws ImportException {
        boolean emitImportStartedEvent = true;
        int totalDuration = 0;

        try {
            for (OverthereFile testResult : testResultFiles) {
                String moduleName = moduleName(searchPath, testResult);
                XUnitReportXmlImporter xmlImporter = new XUnitReportXmlImporter(moduleName, testResult);
                try {
                    if (emitImportStartedEvent) {
                        /* We set the searchPattern in the fileName. Once upon a time we generated an event per
                         * file. This was not practical. To prevent upgrades we left the property name the same
                         * but put in the searchPattern.
                         */
                        sendEvent(eventHandler, testRunId, Event.IMPORT_STARTED, Event.props(
                                "lastModified", lastModified,
                                "fileName", searchPattern));
                        emitImportStartedEvent = false;
                    }
                    totalDuration += xmlImporter.doImport(testRunId, eventHandler);
                } catch (NothingToImportException e) {
                    LOG.info("Nothing to import for file {}", testResult);
                }
            }
         } catch (AlreadyImportedException e) {
            LOG.info("Skipping import for {}, since we already found a similar import in the repository: {}", lastModified, e.getMessage());
            throw e;
        } finally {
            sendEvent(eventHandler, testRunId, Event.IMPORT_FINISHED, Event.props(
                    "duration", totalDuration));
        }
    }

    @Override
    public long getTimestamp() {
        return lastModified;
    }

    private Collection<OverthereFile> sorted(Collection<OverthereFile> testResultFiles) {
        ArrayList<OverthereFile> files = new ArrayList<OverthereFile>(testResultFiles);
        Collections.sort(files, new Comparator<OverthereFile>() {
            @Override
            public int compare(OverthereFile o1, OverthereFile o2) {
                return o1.getPath().compareTo(o2.getPath());
            }
        });
        return files;
    }

    public static String moduleName(OverthereFile searchPath, OverthereFile resultsPath) {
        if (resultsPath == null) {
            return null;
        }
        if (searchPath.equals(resultsPath.getParentFile())) {
            return resultsPath.getName();
        }
        return moduleName(searchPath, resultsPath.getParentFile());
    }

    public static boolean haveValidFiles(Collection<OverthereFile> testResultFiles) {
        return false;
    }
}
