package com.xebialabs.xltest.xunit;


import java.io.InputStream;
import java.util.UUID;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.domain.Importable;

import static com.google.common.base.Strings.isNullOrEmpty;

public class XUnitReportXmlImporter implements Importable {

    public static final char SEPARATOR = ';';

    private final OverthereFile xunitReportXmlFile;
    private final String moduleName;

    public XUnitReportXmlImporter(String moduleName, OverthereFile xunitReportXmlFile) {
        this.moduleName = moduleName;
        this.xunitReportXmlFile = xunitReportXmlFile;
    }

    @Override
    public void doImport(UUID testRunId, EventHandler eventHandler) throws Exception {

        eventHandler.onReceive(new Event(testRunId, Event.IMPORT_STARTED, Event.props(
                "fileName", xunitReportXmlFile.getPath(),
                "lastModified", xunitReportXmlFile.lastModified())));

        NodeList testSuites;
        try (InputStream fis = xunitReportXmlFile.getInputStream()) {
            testSuites = parse(fis);
            if (testSuites == null) {
                return;
            }
        }

        long totalDuration = 0;
        String sourceFile = xunitReportXmlFile.getName();
        for (int i = 0; i < testSuites.getLength(); i++) {
            Element testSuite = (Element) testSuites.item(i);
            long testSuiteDuration = getDuration(testSuite);
            totalDuration = totalDuration + testSuiteDuration;
            tellAboutTestSuite(testSuite, testRunId, eventHandler, sourceFile);
        }

        eventHandler.onReceive(new Event(testRunId, Event.IMPORT_FINISHED, Event.props(
                "duration", totalDuration
        )));
    }

    private long getDuration(Element testSuite) {
        String timeInSecondsAsString = testSuite.getAttribute("time");
        if (timeInSecondsAsString == null || timeInSecondsAsString.equals("")) {
            return 0l;
        }
        float timeInSecondsAsFloat = Float.parseFloat(timeInSecondsAsString);
        return (long) (timeInSecondsAsFloat * 1000.0);
    }

    private NodeList parse(InputStream fis) {
        try {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document doc = db.parse(fis);
            return doc.getElementsByTagName("testsuite");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    private void tellAboutTestSuite(Element testSuite, UUID testRunId, EventHandler eventHandler, String testResultFilename) throws Exception {
        String prefix = (isNullOrEmpty(moduleName) ? "" : (moduleName + SEPARATOR)) + formatSuiteName(readAttribute(testSuite, "name"));

        NodeList testCases = testSuite.getElementsByTagName("testcase");

        final int length = testCases.getLength();
        for (int i = 0; i < length; i++) {
            Element testCase = (Element) testCases.item(i);
            eventHandler.onReceive(new Event(testRunId, Event.FUNCTIONAL_RESULT, Event.props(
                    "result", testCaseResult(testCase),
                    "duration", testCaseDurationInMilliSeconds(testCase),
                    "name", prefix + SEPARATOR + readAttribute(testCase, "name"),
                    "fileName", testResultFilename
            )));
        }
    }

    private String readAttribute(Element testSuite, String attribute) {
        return testSuite.getAttribute(attribute);
    }

    private String testCaseResult(Element testCase) {
        NodeList failureNodes = testCase.getElementsByTagName("failure");
        if (failureNodes == null || failureNodes.getLength() == 0) {
            return "PASSED";
        }
        return "FAILED";
    }

    private long testCaseDurationInMilliSeconds(Element testCase) {
        // time attribute is in seconds
        String duration = readAttribute(testCase, "time");
        if (duration == null || "".equals(duration)) {
            return 0;
        }
        Float f = Float.parseFloat(duration);
        return (long) (f * 1000.0f);

    }

    /**
     * Try to split a suite name into a package name and a class name, as is done
     * in jUnit by default.
     *
     * @param name
     * @return a formatted name (with semicolon where appropriate)
     */
    static String formatSuiteName(String name) {
        int lastDot = name.lastIndexOf('.');
        if (lastDot < 0) {
            return name;
        }

        String packageName = name.substring(0, lastDot);
        String className = name.substring(lastDot + 1);

        if (isNullOrEmpty(packageName) || isNullOrEmpty(className)) {
            return name;
        }

        return packageName + SEPARATOR + className;
    }

}
