/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.EngineConfig;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
extends AbstractLifecycleComponent<IndexingMemoryController> {
    public static final String INDEX_BUFFER_SIZE_SETTING = "indices.memory.index_buffer_size";
    public static final String MIN_INDEX_BUFFER_SIZE_SETTING = "indices.memory.min_index_buffer_size";
    public static final String MAX_INDEX_BUFFER_SIZE_SETTING = "indices.memory.max_index_buffer_size";
    public static final String MIN_SHARD_INDEX_BUFFER_SIZE_SETTING = "indices.memory.min_shard_index_buffer_size";
    public static final String MAX_SHARD_INDEX_BUFFER_SIZE_SETTING = "indices.memory.max_shard_index_buffer_size";
    public static final String TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.translog_buffer_size";
    public static final String MIN_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.min_translog_buffer_size";
    public static final String MAX_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.max_translog_buffer_size";
    public static final String MIN_SHARD_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.min_shard_translog_buffer_size";
    public static final String MAX_SHARD_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.max_shard_translog_buffer_size";
    public static final String SHARD_INACTIVE_TIME_SETTING = "indices.memory.shard_inactive_time";
    public static final String SHARD_INACTIVE_INTERVAL_TIME_SETTING = "indices.memory.interval";
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final ByteSizeValue translogBuffer;
    private final ByteSizeValue minShardTranslogBufferSize;
    private final ByteSizeValue maxShardTranslogBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private volatile ScheduledFuture scheduler;
    private static final EnumSet<IndexShardState> CAN_UPDATE_INDEX_BUFFER_STATES = EnumSet.of(IndexShardState.RECOVERING, IndexShardState.POST_RECOVERY, IndexShardState.STARTED, IndexShardState.RELOCATED);
    private final ShardsIndicesStatusChecker statusChecker;

    @Inject
    public IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        this(settings, threadPool, indicesService, JvmInfo.jvmInfo().getMem().getHeapMax().bytes());
    }

    protected IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService, long jvmMemoryInBytes) {
        super(settings);
        ByteSizeValue translogBuffer;
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = settings.get(INDEX_BUFFER_SIZE_SETTING, "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)jvmMemoryInBytes * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = settings.getAsBytesSize(MIN_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = settings.getAsBytesSize(MAX_INDEX_BUFFER_SIZE_SETTING, null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = settings.getAsBytesSize(MIN_SHARD_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = settings.getAsBytesSize(MAX_SHARD_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(512L, ByteSizeUnit.MB));
        String translogBufferSetting = settings.get(TRANSLOG_BUFFER_SIZE_SETTING, "1%");
        if (translogBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(translogBufferSetting.substring(0, translogBufferSetting.length() - 1));
            translogBuffer = new ByteSizeValue((long)((double)jvmMemoryInBytes * (percent / 100.0)));
            ByteSizeValue minTranslogBuffer = settings.getAsBytesSize(MIN_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(256L, ByteSizeUnit.KB));
            ByteSizeValue maxTranslogBuffer = settings.getAsBytesSize(MAX_TRANSLOG_BUFFER_SIZE_SETTING, null);
            if (translogBuffer.bytes() < minTranslogBuffer.bytes()) {
                translogBuffer = minTranslogBuffer;
            }
            if (maxTranslogBuffer != null && translogBuffer.bytes() > maxTranslogBuffer.bytes()) {
                translogBuffer = maxTranslogBuffer;
            }
        } else {
            translogBuffer = ByteSizeValue.parseBytesSizeValue(translogBufferSetting, null);
        }
        this.translogBuffer = translogBuffer;
        this.minShardTranslogBufferSize = settings.getAsBytesSize(MIN_SHARD_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(2L, ByteSizeUnit.KB));
        this.maxShardTranslogBufferSize = settings.getAsBytesSize(MAX_SHARD_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(64L, ByteSizeUnit.KB));
        this.inactiveTime = settings.getAsTime(SHARD_INACTIVE_TIME_SETTING, TimeValue.timeValueMinutes(5L));
        this.interval = settings.getAsTime(SHARD_INACTIVE_INTERVAL_TIME_SETTING, TimeValue.timeValueSeconds(30L));
        this.statusChecker = new ShardsIndicesStatusChecker();
        this.logger.debug("using indexing buffer size [{}], with {} [{}], {} [{}], {} [{}], {} [{}]", this.indexingBuffer, MIN_SHARD_INDEX_BUFFER_SIZE_SETTING, this.minShardIndexBufferSize, MAX_SHARD_INDEX_BUFFER_SIZE_SETTING, this.maxShardIndexBufferSize, SHARD_INACTIVE_TIME_SETTING, this.inactiveTime, SHARD_INACTIVE_INTERVAL_TIME_SETTING, this.interval);
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        this.scheduler = this.threadPool.scheduleWithFixedDelay(this.statusChecker, this.interval);
    }

    @Override
    protected void doStop() throws ElasticsearchException {
        FutureUtils.cancel(this.scheduler);
        this.scheduler = null;
    }

    @Override
    protected void doClose() throws ElasticsearchException {
    }

    public ByteSizeValue indexingBufferSize() {
        return this.indexingBuffer;
    }

    public ByteSizeValue translogBufferSize() {
        return this.translogBuffer;
    }

    protected List<ShardId> availableShards() {
        ArrayList<ShardId> list = new ArrayList<ShardId>();
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                if (!this.shardAvailable(indexShard)) continue;
                list.add(indexShard.shardId());
            }
        }
        return list;
    }

    protected boolean shardAvailable(ShardId shardId) {
        return this.shardAvailable(this.getShard(shardId));
    }

    protected boolean shardAvailable(@Nullable IndexShard shard) {
        return shard != null && CAN_UPDATE_INDEX_BUFFER_STATES.contains((Object)shard.state());
    }

    protected IndexShard getShard(ShardId shardId) {
        IndexService indexService = this.indicesService.indexService(shardId.index().name());
        if (indexService != null) {
            IndexShard indexShard = indexService.shard(shardId.id());
            return indexShard;
        }
        return null;
    }

    protected void updateShardBuffers(ShardId shardId, ByteSizeValue shardIndexingBufferSize, ByteSizeValue shardTranslogBufferSize) {
        IndexShard shard = this.getShard(shardId);
        if (shard != null) {
            try {
                shard.updateBufferSize(shardIndexingBufferSize, shardTranslogBufferSize);
            }
            catch (EngineClosedException e) {
            }
            catch (FlushNotAllowedEngineException e) {
            }
            catch (Exception e) {
                this.logger.warn("failed to set shard {} index buffer to [{}]", shardId, shardIndexingBufferSize);
            }
        }
    }

    protected ShardIndexingStatus getTranslogStatus(ShardId shardId) {
        Translog translog;
        IndexShard indexShard = this.getShard(shardId);
        if (indexShard == null) {
            return null;
        }
        try {
            translog = indexShard.translog();
        }
        catch (EngineClosedException e) {
            return null;
        }
        ShardIndexingStatus status = new ShardIndexingStatus();
        status.translogId = translog.currentId();
        status.translogNumberOfOperations = translog.estimatedNumberOfOperations();
        return status;
    }

    void forceCheck() {
        this.statusChecker.run();
    }

    protected long currentTimeInNanos() {
        return System.nanoTime();
    }

    protected void markShardAsInactive(ShardId shardId) {
        String ignoreReason = null;
        IndexShard shard = this.getShard(shardId);
        if (shard != null) {
            try {
                shard.markAsInactive();
            }
            catch (EngineClosedException e) {
                ignoreReason = "EngineClosedException";
            }
            catch (FlushNotAllowedEngineException e) {
                ignoreReason = "FlushNotAllowedEngineException";
            }
        } else {
            ignoreReason = "shard not found";
        }
        if (ignoreReason != null) {
            this.logger.trace("ignore [{}] while marking shard {} as inactive", ignoreReason, shardId);
        }
    }

    static class ShardIndexingStatus {
        long translogId = -1L;
        int translogNumberOfOperations = -1;
        boolean activeIndexing = true;
        long idleSinceNanoTime = -1L;

        ShardIndexingStatus() {
        }

        public void updateWith(long currentNanoTime, ShardIndexingStatus current, long inactiveNanoInterval) {
            boolean idle;
            boolean bl = idle = this.translogId == current.translogId && this.translogNumberOfOperations == current.translogNumberOfOperations;
            if (this.activeIndexing && idle) {
                if (this.idleSinceNanoTime < 0L) {
                    this.idleSinceNanoTime = currentNanoTime;
                } else if (currentNanoTime - this.idleSinceNanoTime > inactiveNanoInterval) {
                    this.activeIndexing = false;
                }
            } else if (!this.activeIndexing && !idle && current.translogNumberOfOperations > 0) {
                this.activeIndexing = true;
                this.idleSinceNanoTime = -1L;
            }
            this.translogId = current.translogId;
            this.translogNumberOfOperations = current.translogNumberOfOperations;
        }
    }

    private static enum ShardStatusChangeType {
        ADDED,
        DELETED,
        BECAME_ACTIVE,
        BECAME_INACTIVE;

    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        private final Map<ShardId, ShardIndexingStatus> shardsIndicesStatus = new HashMap<ShardId, ShardIndexingStatus>();

        ShardsIndicesStatusChecker() {
        }

        @Override
        public void run() {
            EnumSet<ShardStatusChangeType> changes = EnumSet.noneOf(ShardStatusChangeType.class);
            changes.addAll(this.purgeDeletedAndClosedShards());
            ArrayList<ShardId> activeToInactiveIndexingShards = new ArrayList<ShardId>();
            int activeShards = this.updateShardStatuses(changes, activeToInactiveIndexingShards);
            for (ShardId indexShard : activeToInactiveIndexingShards) {
                IndexingMemoryController.this.markShardAsInactive(indexShard);
            }
            if (!changes.isEmpty()) {
                this.calcAndSetShardBuffers(activeShards, "[" + changes + "]");
            }
        }

        private int updateShardStatuses(EnumSet<ShardStatusChangeType> changes, List<ShardId> activeToInactiveIndexingShards) {
            int activeShards = 0;
            for (ShardId shardId : IndexingMemoryController.this.availableShards()) {
                ShardIndexingStatus currentStatus = IndexingMemoryController.this.getTranslogStatus(shardId);
                if (currentStatus == null) continue;
                ShardIndexingStatus status = this.shardsIndicesStatus.get(shardId);
                if (status == null) {
                    status = currentStatus;
                    this.shardsIndicesStatus.put(shardId, status);
                    changes.add(ShardStatusChangeType.ADDED);
                } else {
                    boolean lastActiveIndexing = status.activeIndexing;
                    status.updateWith(IndexingMemoryController.this.currentTimeInNanos(), currentStatus, IndexingMemoryController.this.inactiveTime.nanos());
                    if (lastActiveIndexing && !status.activeIndexing) {
                        activeToInactiveIndexingShards.add(shardId);
                        changes.add(ShardStatusChangeType.BECAME_INACTIVE);
                        IndexingMemoryController.this.logger.debug("marking shard {} as inactive (inactive_time[{}]) indexing wise, setting size to [{}]", shardId, IndexingMemoryController.this.inactiveTime, EngineConfig.INACTIVE_SHARD_INDEXING_BUFFER);
                    } else if (!lastActiveIndexing && status.activeIndexing) {
                        changes.add(ShardStatusChangeType.BECAME_ACTIVE);
                        IndexingMemoryController.this.logger.debug("marking shard {} as active indexing wise", shardId);
                    }
                }
                if (!status.activeIndexing) continue;
                ++activeShards;
            }
            return activeShards;
        }

        private EnumSet<ShardStatusChangeType> purgeDeletedAndClosedShards() {
            EnumSet<ShardStatusChangeType> changes = EnumSet.noneOf(ShardStatusChangeType.class);
            Iterator<ShardId> statusShardIdIterator = this.shardsIndicesStatus.keySet().iterator();
            while (statusShardIdIterator.hasNext()) {
                ShardId shardId = statusShardIdIterator.next();
                if (IndexingMemoryController.this.shardAvailable(shardId)) continue;
                changes.add(ShardStatusChangeType.DELETED);
                statusShardIdIterator.remove();
            }
            return changes;
        }

        private void calcAndSetShardBuffers(int activeShards, String reason) {
            ByteSizeValue shardTranslogBufferSize;
            if (activeShards == 0) {
                return;
            }
            ByteSizeValue shardIndexingBufferSize = new ByteSizeValue(IndexingMemoryController.this.indexingBuffer.bytes() / (long)activeShards);
            if (shardIndexingBufferSize.bytes() < IndexingMemoryController.this.minShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.minShardIndexBufferSize;
            }
            if (shardIndexingBufferSize.bytes() > IndexingMemoryController.this.maxShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.maxShardIndexBufferSize;
            }
            if ((shardTranslogBufferSize = new ByteSizeValue(IndexingMemoryController.this.translogBuffer.bytes() / (long)activeShards)).bytes() < IndexingMemoryController.this.minShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.minShardTranslogBufferSize;
            }
            if (shardTranslogBufferSize.bytes() > IndexingMemoryController.this.maxShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.maxShardTranslogBufferSize;
            }
            IndexingMemoryController.this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to indexing=[{}], translog=[{}]", reason, IndexingMemoryController.this.indexingBuffer, activeShards, shardIndexingBufferSize, shardTranslogBufferSize);
            for (ShardId shardId : IndexingMemoryController.this.availableShards()) {
                ShardIndexingStatus status = this.shardsIndicesStatus.get(shardId);
                if (status != null && !status.activeIndexing) continue;
                IndexingMemoryController.this.updateShardBuffers(shardId, shardIndexingBufferSize, shardTranslogBufferSize);
            }
        }
    }
}

