# NOTE: Please do not edit this file directly
# Create your custom modules in the 'ext' folder

from math import floor
from itertools import groupby
from operator import itemgetter

from modules.hierarchy import *

class Flakiness(object):
    def __init__(self, testRun, testRuns, query_parameters, start_date, end_date, tags):
        self._testRun = testRun
        self._testRuns = testRuns
        self._queryParameters = query_parameters
        self._prefixArray = get_prefix_array(self._queryParameters)
        self._startDate = start_date
        self._endDate = end_date
        self._tags = tags

    def _get_result(self, r):
        return r == 'OK' or r == 'PASSED'

    def _edges(self, results):
        e = 0
        v = results[0]
        for r in results:
            if r != v:
                e += 1
                v = r
        return e

    def getFlakinessResult(self):
        numberOfTestsToShow = int(self._queryParameters.get('testsToShow') or 10)
        maxResults = int(self._queryParameters.get('maxResults') or 30)
        prefix = self._queryParameters.get('prefix') or ''

        by_last_modified = itemgetter(0)

        found_import_events = self._testRuns.getEventsBetween(self._startDate.getTime(), self._endDate.getTime(), {'@testSpecification': self._testRun.getTestSpecificationName(), '@type': 'importStarted'})
        testrun_vs_lastmodified = [(event.get('@testedAt'), event.get('@runId')) for event in found_import_events ]
        testrun_vs_lastmodified.sort()
        testrun_vs_lastmodified = testrun_vs_lastmodified[-maxResults:]

        by_name = itemgetter(0)
        by_run_id = itemgetter(1)
        by_result = itemgetter(2)
        by_event = itemgetter(3)

        found_events = self._testRuns.getEventsBetween(self._startDate.getTime(), self._endDate.getTime(), {'@testSpecification': self._testRun.getTestSpecificationName(), '@type': 'functionalResult', '@tags': self._tags})

        quadruplets = [(';'.join(event.get('@hierarchy')), event.get('@runId'), self._get_result(event.get('@result')), event) for event in found_events if prefix_within_hierarchy(event, self._prefixArray)]
        quadruplets.sort()
        tests = []
        for name, group in groupby(quadruplets, by_name):
            # build array: passed (True), failed (False), nothing (None)
            run_id_to_result = { by_run_id(e) : {'result': by_result(e),
                                                 'duration': by_event(e).get('@duration'),
                                                 'tags': by_event(e).get('@tags'),
                                                 'eventId': by_event(e).getId(),
                                                 'firstError': by_event(e).firstError if by_event(e).hasProperty('@firstError') else ''} for e in group }
            # order the results on the time axis (most recent first, since testrun_vs_lastmodified is ordered as such)
            # results in a list of True, False, None
            results = [run_id_to_result.get(run_id) for _, run_id in testrun_vs_lastmodified]
            flakiness_results = [ r['result'] for r in results if r ]
            flakiness = 0
            nr_of_edges = (len(flakiness_results) - 1)
            if nr_of_edges > 0:
                flakiness = floor(100.0 * self._edges(flakiness_results) / nr_of_edges)

            names = name.split(';')
            nr_of_trues = flakiness_results.count(True)
            tests.append({'name': name, 'suiteName': '.'.join(names[:-1]), 'label': names[-1], 'flakiness': flakiness, 'results': results, 'nr_of_trues': nr_of_trues})

        tests.sort(key=itemgetter('nr_of_trues')) # Sort first on nr of true values. This puts the most failed test at the top if flakiness = equal
        tests.sort(key=itemgetter('flakiness'), reverse=True)

        dates = map(by_last_modified, testrun_vs_lastmodified)
        return {
            'title': 'Flakiness overview',
            'description': 'This report identifies the test with the most inconsistent results over time.\nPoint to a block to see detailed information about the test it represents.',
            'testSpecification': self._testRun.getTestSpecificationName(),
            'tests': tests[:numberOfTestsToShow],
            'dates': dates,
            'startDate': dates[0] if dates else 0,
            'endDate': dates[-1] if dates else 0,
            'resultsLength': len(testrun_vs_lastmodified)
        }
