/*
 * Decompiled with CFR 0.152.
 */
package com.xebialabs.xltest.service;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.xebialabs.xltest.domain.BaseTestSpecification;
import com.xebialabs.xltest.domain.ExecutionResult;
import com.xebialabs.xltest.domain.NoTestRunFoundException;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestSpecificationSet;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.repository.TestSpecificationRepository;
import com.xebialabs.xltest.service.ExecutionProgressEvent;
import com.xebialabs.xltest.utils.ExecutingTestRun;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class TestExecutionStateHolder {
    private static final Logger LOG = LoggerFactory.getLogger(TestExecutionStateHolder.class);
    private static final int DEFAULT_CACHE_EXPIRE_TIME = 15000;
    private final ConcurrentHashMap<String, ExecutingTestRun> runningTasks = new ConcurrentHashMap();
    private final Cache<String, ExecutingTestRun> finishedTasks;
    private final TestRunsRepository testRunsRepository;
    private final TestSpecificationRepository testSpecificationRepository;
    private final long expireAfterMillisOfLastWrite;

    @Autowired
    public TestExecutionStateHolder(TestRunsRepository testRunsRepository, TestSpecificationRepository testSpecificationRepository) {
        this(testRunsRepository, testSpecificationRepository, 15000L);
    }

    public TestExecutionStateHolder(TestRunsRepository testRunsRepository, TestSpecificationRepository testSpecificationRepository, long expireAfterMillisOfLastWrite) {
        this.testRunsRepository = testRunsRepository;
        this.testSpecificationRepository = testSpecificationRepository;
        this.expireAfterMillisOfLastWrite = expireAfterMillisOfLastWrite;
        this.finishedTasks = this.createRunsCache();
    }

    public void newTestRun(String testSpecification, UUID taskId) {
        this.runningTasks.putIfAbsent(testSpecification, new ExecutingTestRun(testSpecification, taskId, null));
    }

    public void moveTaskToExpiredCache(String testSpecification) {
        ExecutingTestRun executingTestRun = this.runningTasks.get(testSpecification);
        executingTestRun.setFinishedTime(new Date());
        this.finishedTasks.put((Object)testSpecification, (Object)executingTestRun);
        this.runningTasks.remove(testSpecification);
    }

    public ExecutingTestRun getFromRunningCache(String testSpecification) {
        ExecutingTestRun testRun = this.runningTasks.get(testSpecification);
        this.syncWithRepoIfNeeded(testRun);
        return testRun;
    }

    public void syncIfNeeded(String ... testSpecifications) {
        for (String testSpecification : testSpecifications) {
            ExecutingTestRun testRun = this.getCachedTestRun(testSpecification);
            this.syncWithRepoIfNeeded(testRun);
        }
    }

    public void syncWithRepoIfNeeded(ExecutingTestRun testRun) {
        if (testRun != null && testRun.getStartTime() == null) {
            BaseTestSpecification ci;
            TestRun testRunFromRepo = null;
            try {
                testRunFromRepo = this.testRunsRepository.getTestRun(testRun.getTestRunId());
            }
            catch (NoTestRunFoundException e) {
                // empty catch block
            }
            if (testRunFromRepo != null) {
                testRun.setStartTime(testRunFromRepo.getStartTime());
                this.update(testRun);
            }
            if ((ci = this.testSpecificationRepository.get(testRun.getTestSpecificationName())) != null) {
                BaseTestSpecification baseTestSpecification = ci;
                if (baseTestSpecification instanceof TestSpecificationSet) {
                    long longestDurationOfChildren = this.getLongestDurationOfChildren((TestSpecificationSet)baseTestSpecification);
                    testRun.setAveragePreviousDuration(longestDurationOfChildren);
                    LOG.debug("testRun details synced for TestSpecificationSet" + testRun.getTestSpecificationName());
                } else if (testRunFromRepo != null) {
                    long averagePreviousDuration = this.calculateAveragePreviousDuration(testRunFromRepo);
                    testRun.setAveragePreviousDuration(averagePreviousDuration);
                    LOG.debug("testRun details synced for test specification " + testRun.getTestSpecificationName());
                }
                this.update(testRun);
            }
        }
    }

    public void updateTask(String testSpecificationName, Future<String> task) {
        ExecutingTestRun cachedTestRun = this.getFromRunningCache(testSpecificationName);
        if (cachedTestRun != null) {
            cachedTestRun.setTask(task);
            this.update(cachedTestRun);
            LOG.debug("Executing Test run start date updated:" + cachedTestRun.getStartTime());
        }
    }

    public void updateStartTime(String testSpecificationName) {
        ExecutingTestRun cachedTestRun = this.getFromRunningCache(testSpecificationName);
        if (cachedTestRun != null) {
            cachedTestRun.setStartTime(new Date());
            this.update(cachedTestRun);
            LOG.debug("Executing Test run start date updated:" + cachedTestRun.getStartTime());
        }
    }

    public void updateQualification(String testSpec, boolean qualificationResult, String failureReason) {
        ExecutingTestRun cachedTestRun = this.getFromRunningCache(testSpec);
        if (cachedTestRun != null) {
            cachedTestRun.setQualificationResult(qualificationResult);
            cachedTestRun.setFailureReason(failureReason);
            this.update(cachedTestRun);
        }
    }

    public List<ExecutionProgressEvent> getProgress(List<BaseTestSpecification> testSpecifications) {
        LOG.trace("finding status of test specifications");
        ArrayList<ExecutionProgressEvent> events = new ArrayList<ExecutionProgressEvent>();
        for (BaseTestSpecification testSpecification : testSpecifications) {
            ExecutionProgressEvent progressEvent = this.getProgressEvent(testSpecification);
            if (progressEvent == null) continue;
            events.add(progressEvent);
        }
        return events;
    }

    public ExecutionProgressEvent getProgressEvent(BaseTestSpecification testSpecification) {
        ExecutingTestRun testRun = this.getCachedTestRun(testSpecification.getName());
        ExecutionProgressEvent progressEvent = null;
        if (testRun != null && testRun.getStartTime() != null) {
            long averagePreviousDuration = testRun.getAveragePreviousDuration();
            progressEvent = testRun.getProgressEvent(averagePreviousDuration);
            if (this.getFromRunningCache(testSpecification.getName()) == null && testRun.isTestSpecificationSet()) {
                ExecutionResult executionResult = this.deriveQualificationFromChildren(testSpecification);
                progressEvent.setExecutionResult(executionResult);
            }
        }
        return progressEvent;
    }

    protected ExecutingTestRun getFromExpiredCache(String testSpecification) {
        ExecutingTestRun testRun = (ExecutingTestRun)this.finishedTasks.getIfPresent((Object)testSpecification);
        this.syncWithRepoIfNeeded(testRun);
        return testRun;
    }

    private void update(ExecutingTestRun cachedTestRun) {
        if (cachedTestRun != null) {
            cachedTestRun.setLastUpdateTime(new Date());
            this.runningTasks.replace(cachedTestRun.getTestSpecificationName(), cachedTestRun);
        }
    }

    private ExecutionResult deriveQualificationFromChildren(BaseTestSpecification testSpecification) {
        if (testSpecification instanceof TestSpecificationSet) {
            Set<BaseTestSpecification> children = this.getAllChildren((TestSpecificationSet)testSpecification);
            for (BaseTestSpecification child : children) {
                ExecutingTestRun testRun = this.getFromExpiredCache(child.getName());
                if (testRun == null || testRun.getExecutionResult().getStatus().equals((Object)ExecutionResult.ExecutionStatus.FAILED)) continue;
                return ExecutionResult.failed();
            }
        }
        return ExecutionResult.finished();
    }

    private ExecutingTestRun getCachedTestRun(String testSpecification) {
        ExecutingTestRun testRun = this.runningTasks.get(testSpecification);
        testRun = testRun != null ? testRun : (ExecutingTestRun)this.finishedTasks.getIfPresent((Object)testSpecification);
        this.syncWithRepoIfNeeded(testRun);
        return testRun;
    }

    private long calculateAveragePreviousDuration(TestRun testRun) {
        List previousRuns = this.testRunsRepository.getPreviousRuns(testRun.getTestSpecificationName(), 10);
        return this.calculateAverageDuration(previousRuns);
    }

    private long calculateAverageDuration(List<TestRun> previousRuns) {
        Long totalDuration = 0L;
        for (TestRun eachPrevious : previousRuns) {
            if (eachPrevious.getFinishedTime() == null) continue;
            totalDuration = totalDuration + (eachPrevious.getFinishedTime().getTime() - eachPrevious.getStartTime().getTime());
        }
        return !previousRuns.isEmpty() ? totalDuration / (long)previousRuns.size() : -1L;
    }

    private long getLongestDurationOfChildren(TestSpecificationSet testSpecification) {
        long duration = 0L;
        Set<BaseTestSpecification> children = this.getAllChildren(testSpecification);
        for (BaseTestSpecification eachChild : children) {
            ExecutingTestRun cachedTestRun = this.getCachedTestRun(eachChild.getName());
            if (cachedTestRun == null) continue;
            duration = Math.max(cachedTestRun.getAveragePreviousDuration(), duration);
        }
        return duration;
    }

    private Set<BaseTestSpecification> getAllChildren(TestSpecificationSet specificationSet) {
        HashSet allChildren = Sets.newHashSet();
        for (BaseTestSpecification each : specificationSet.getTestSpecifications()) {
            if (each instanceof TestSpecificationSet) {
                allChildren.addAll(this.getAllChildren((TestSpecificationSet)each));
                continue;
            }
            allChildren.add(each);
        }
        return allChildren;
    }

    private Cache<String, ExecutingTestRun> createRunsCache() {
        LoadingCache runningTestRuns = CacheBuilder.newBuilder().concurrencyLevel(10).expireAfterWrite(this.expireAfterMillisOfLastWrite, TimeUnit.MILLISECONDS).build((CacheLoader)new TestRunLoader());
        return runningTestRuns;
    }

    public long getExpireAfterMillisOfLastWrite() {
        return this.expireAfterMillisOfLastWrite;
    }

    private class TestRunLoader
    extends CacheLoader<String, ExecutingTestRun> {
        private TestRunLoader() {
        }

        public ExecutingTestRun load(String key) throws Exception {
            Long oneDayAgo = new DateTime().minusDays(1).getMillis();
            TestRun run = TestExecutionStateHolder.this.testRunsRepository.getLatestTestRun(key, oneDayAgo.longValue());
            ExecutingTestRun executingRun = new ExecutingTestRun(run.getTestSpecificationName(), run.getTestRunId(), null);
            executingRun.setStartTime(run.getStartTime());
            return executingRun;
        }
    }
}

