package com.xebialabs.xltest.resources;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.TestSpecification;
import com.xebialabs.xltest.domain.TestTool;
import com.xebialabs.xltest.repository.TestSpecificationRepository;
import com.xebialabs.xltest.repository.TestTools;
import com.xebialabs.xltest.service.TestResultLocator;

import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;

import static java.lang.String.format;

@Controller
@Path("/testtools")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class TestToolsController {
    private static final Logger LOG = LoggerFactory.getLogger(TestToolsController.class);

    private final TestTools testTools;
    private final TestSpecificationRepository testSpecifications;

    @Autowired
    public TestToolsController(TestTools testTools, TestSpecificationRepository testSpecifications) {
        this.testTools = testTools;
        this.testSpecifications = testSpecifications;
    }

    @GET
    public Response getTestTools() {
        JSONArray json = new JSONArray();
        for (TestTool testTool : testTools.getTestTools()) {
            JSONObject tool = new JSONObject();
            tool.put("name", testTool.getName());
            tool.put("pattern", testTool.getSearchPattern());
            json.add(tool);
        }
        return Response.ok(json.toString()).build();
    }

    @GET
    @Path("{testTool}/paths")
    public Response validateHostConfiguration(@PathParam("testTool") String testTool,
                                              @QueryParam("type") String type,
                                              @QueryParam("address") String address,
                                              @QueryParam("username") String username,
                                              @QueryParam("password") String password,
                                              @QueryParam("path") String path,
                                              @QueryParam("pattern") String pattern,
                                              @QueryParam("jobName") String jobName) {

        List<TestSpecification> testSpecificationsWithHost = testSpecifications.findWithHost(testTool, pattern, address, path, -1, type);

        Collection<OverthereFile> files;

        if(type.equals("overthere.JenkinsHost")) {
            Response urlVerify = verifyJenkinsUrl(address);
            if (urlVerify != null) {
                return urlVerify;
            }
        }

        try {
            files = new TestResultLocator(testTools)
                    .forHost(type, address, username, password, jobName)
                    .forTestTool(testTool)
                    .locate(path, pattern);
        } catch (Exception e) {
            LOG.info("Unable to validate host configuration. Type: {}, address: {}, username: {}, path: {}, testtool: {}", type, address, username, path, testTool, e);
            JSONObject json = new JSONObject();
            json.put("error", e.getMessage());
            return Response.status(Response.Status.BAD_REQUEST).entity(json.toString()).build();
        }

        if (files != null && !files.isEmpty()) {
            JSONObject json = new JSONObject();
            JSONArray paths = new JSONArray();
            json.put("paths", paths);
            for (OverthereFile file : files) {
                paths.add(file.getPath());
            }
            JSONArray testSpecificationsJson = new JSONArray();
            json.put("testSpecifications", testSpecificationsJson);
            for (TestSpecification testSpecification : testSpecificationsWithHost) {
                testSpecificationsJson.add(testSpecification.getName());
            }
            return Response.ok(json.toString()).build();
        } else {
            JSONObject json = new JSONObject();
            json.put("error", format("No test results found in path %s", path));
            return Response.status(Response.Status.NOT_FOUND).entity(json.toString()).build();
        }
    }

    /**
     * Verify Jenkins URL to catch parse errors early
     *
     * @param url Jenkins URL to parse
     * @return null if url is OK, Response object with message if there is an error.
     */
    private Response verifyJenkinsUrl(String url) {
        try {
            URL addressUrl = new URL(url);
        } catch (MalformedURLException e) {
            JSONObject json = new JSONObject();
            json.put("error", "Could not parse Jenkins URL: " + e.getMessage());
            return Response.status(Response.Status.BAD_REQUEST).entity(json.toString()).build();
        }
        return null;
    }
}
