package com.xebialabs.xltest.resources;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.*;

import com.xebialabs.deployit.plugin.overthere.Host;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.jayway.jsonpath.spi.JsonProviderFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestSpecification;
import com.xebialabs.xltest.service.EventRepository;

import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;


@Controller
@Path("/migrate")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class MigrationController {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationController.class);

    private static final String[] propsToCopy = {"branch", "browser", "environment", "repositoryUrl", "suiteFilter", "suiteName", "webdriver"};


    private final EventRepository eventRepository;
    private final RepositoryService repository;

    private static boolean alreadyMigrated = false;

    private long totalInsertedEvents = 0;

    @Autowired
    public MigrationController(EventRepository eventRepository, RepositoryService repository) {
        this.eventRepository = eventRepository;
        this.repository = repository;
    }

    @GET
    @Path("/")
    public Response migrate(@Context UriInfo uriInfo) throws IOException {
        String feedback = "";
        MultivaluedMap<String, String> queryParameters = uriInfo.getQueryParameters();
        String host = null;
        if (queryParameters.get("host") != null) {
            host = queryParameters.get("host").get(0);
        }
        String port = null;
        if (queryParameters.get("port") != null) {
            port = queryParameters.get("port").get(0);
        }
        if (host != null && port != null && !alreadyMigrated) {
            URL oldBaseUrl = new URL("http://" + host + ":" + port); // eg. http://localhost:9400
            if (checkMigratable(oldBaseUrl)) {
                LOG.info("migration started");
                alreadyMigrated = true;
                migrateJCRReports();
                feedback = migrateES(oldBaseUrl);
            } else {
                feedback = "Sorry, can't migrate as I can't reach your old ElasticSearch on host " + host + " and port " + port;
            }
        } else {
            if (alreadyMigrated) {
                feedback = "Sorry, can't migrate. It seems you already did. Restart server when in doubt";
            } else {
                feedback = "Sorry, can't migrate. I expected two query parameters: host=<hostname> and port=<portnumber>";
            }
        }
        return Response.ok(feedback).build();
    }

    private void migrateJCRReports() {
        Report compositePerTeam = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        compositePerTeam.setId("Configuration/Reports/CompositePerTeam");
        compositePerTeam.setProperty("title", "Test Results per team");
        compositePerTeam.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(compositePerTeam);
        Report bar = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        bar.setId("Configuration/Reports/barchart");
        bar.setProperty("scriptLocation", "reports/GraphReport.py");
        bar.setProperty("drilldownsequence", "team, application, applicationPackage, usecase, testcase");
        bar.setProperty("drilldown", compositePerTeam);
        bar.setProperty("useTeamInDrillDown", true);
        repository.createOrUpdate(bar);
        Report pie = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.PieReport")).newInstance("");
        pie.setId("Configuration/Reports/pie");
        pie.setProperty("scriptLocation", "reports/PieReport.py");
        repository.createOrUpdate(pie);
        Report details = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        details.setId("Configuration/Reports/drilldown");
        details.setProperty("scriptLocation", "reports/drilldowndetails.ftl");
        details.setProperty("reportType", "html");
        details.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(details);
        compositePerTeam.setProperty("topLeft", bar);
        compositePerTeam.setProperty("topRight", pie);
        compositePerTeam.setProperty("bottom", details);
        repository.createOrUpdate(compositePerTeam);

        Report compositePerApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        compositePerApp.setId("Configuration/Reports/CompositePerApp");
        compositePerApp.setProperty("title", "Test Results");
        compositePerApp.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(compositePerApp);
        Report barchartPerApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        barchartPerApp.setId("Configuration/Reports/barchartPerApp");
        barchartPerApp.setProperty("scriptLocation", "reports/GraphReport.py");
        barchartPerApp.setProperty("drilldownsequence", "application, applicationPackage, usecase, testcase");
        barchartPerApp.setProperty("drilldown", compositePerApp);
        repository.createOrUpdate(barchartPerApp);
        compositePerApp.setProperty("topLeft", barchartPerApp);
        compositePerApp.setProperty("topRight", pie);
        compositePerApp.setProperty("bottom", details);
        repository.createOrUpdate(compositePerApp);

        Report monitor = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        monitor.setId("Configuration/Reports/monitor");
        monitor.setProperty("title", "Test Automation Results");
        monitor.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(monitor);

        Report monitorApp = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.CompositeReport")).newInstance("");
        monitorApp.setId("Configuration/Reports/monitorApp");
        monitorApp.setProperty("title", "Test Automation Results Per App");
        monitorApp.setProperty("scriptLocation", "reports/CompositeReport.py");
        repository.createOrUpdate(monitorApp);

        Report monitorChart = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorChart.setId("Configuration/Reports/Monitor chart");
        monitorChart.setProperty("scriptLocation", "reports/DashboardReport.py");
        monitorChart.setProperty("drilldownsequence", "team, application, applicationPackage, usecase");
        monitorChart.setProperty("drilldown", monitor);
        monitorChart.setProperty("useTeamInDrillDown", true);
        repository.createOrUpdate(monitorChart);

        Report monitorAppChart = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorAppChart.setId("Configuration/Reports/Monitor APP chart");
        monitorAppChart.setProperty("scriptLocation", "reports/DashboardReport.py");
        monitorAppChart.setProperty("drilldownsequence", "team, application, usecase, testcase");
        monitorAppChart.setProperty("drilldown", monitorApp);
        repository.createOrUpdate(monitorAppChart);

        Report monitorPie = DescriptorRegistry.getDescriptor(Type.valueOf("xltest.GraphReport")).newInstance("");
        monitorPie.setId("Configuration/Reports/Monitor pie chart");
        monitorPie.setProperty("scriptLocation", "reports/DashboardPieReport.py");
        monitorPie.setProperty("drilldownsequence", "team, application, applicationPackage, usecase, testcase");
        monitorPie.setProperty("drilldown", monitor);
        repository.createOrUpdate(monitorPie);

        Report monitorNavigation = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        monitorNavigation.setId("Configuration/Reports/Monitor navigation chart");
        monitorNavigation.setProperty("scriptLocation", "reports/monitorBottom.ftl");
        monitorNavigation.setProperty("reportType", "html");
        monitorNavigation.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(monitorNavigation);

        Report monitorAppNavigation = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        monitorAppNavigation.setId("Configuration/Reports/Monitor APP navigation chart");
        monitorAppNavigation.setProperty("scriptLocation", "reports/monitorBottomApp.ftl");
        monitorAppNavigation.setProperty("reportType", "html");
        monitorAppNavigation.setProperty("maxRunsInHistory", 10);
        repository.createOrUpdate(monitorAppNavigation);

        monitor.setProperty("topLeft", monitorChart);
        monitor.setProperty("topRight", monitorPie);
        monitor.setProperty("bottom", monitorNavigation);
        repository.createOrUpdate(monitor);

        monitorApp.setProperty("topLeft", monitorAppChart);
        monitorApp.setProperty("topRight", monitorPie);
        monitorApp.setProperty("bottom", monitorAppNavigation);
        repository.createOrUpdate(monitorApp);

        Report jobsPerSlave = DescriptorRegistry.getDescriptor(Type.valueOf("freemarker.Report")).newInstance("");
        jobsPerSlave.setId("Configuration/Reports/jobsPerSlave");
        jobsPerSlave.setProperty("scriptLocation", "reports/jobsPerSlave.ftl");
        jobsPerSlave.setProperty("reportType", "html");
        jobsPerSlave.setProperty("iconName", "none");
        jobsPerSlave.setProperty("userFriendlyDescription", "Jobs per slave report");
        repository.createOrUpdate(jobsPerSlave);

        Report freemarkerDuration = DescriptorRegistry.getDescriptor(Type.valueOf("bol.FreemarkerReportWithHistory")).newInstance("");
        freemarkerDuration.setId("Configuration/Reports/BolDuration");
        freemarkerDuration.setProperty("scriptLocation", "reports/durationdetails.ftl");
        freemarkerDuration.setProperty("reportType", "html");
        freemarkerDuration.setProperty("iconName", "none");
        freemarkerDuration.setProperty("userFriendlyDescription", "Tabular duration report");
        freemarkerDuration.setProperty("maxRunsInHistory", 20);
        repository.createOrUpdate(freemarkerDuration);
    }

    private boolean checkMigratable(URL oldBaseUrl) {
        HttpURLConnection connToOldES = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_mapping");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();

            LOG.info("Connected to old ESs. Resp. code on old: " + responseCodeOnOldMappingUrl);

            if (responseCodeOnOldMappingUrl == 200) {
                // we assume we're all set and off we go
            } else {
                LOG.error("Could not connect to one or both ES instances, see above.....");
                return false;
            }

        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
        }
        LOG.info("Found old ES so we have a migratable situation");
        return true;
    }

    private String migrateES(URL oldBaseUrl) throws IOException {
        long now = System.currentTimeMillis();

        migrateRuns(oldBaseUrl);

        long durationInMillis = System.currentTimeMillis() - now;
        long durationInSec = durationInMillis / 1000;
        long durationInMin = durationInSec / 60;
        long durationInHrs = durationInMin / 60;
        durationInMin = durationInMin - (durationInHrs * 60);
        durationInSec = (durationInSec - (durationInHrs * 60 * 60)) - (durationInMin * 60);
        String feedback = "Migration done in " + durationInHrs + " hrs " + durationInMin + " min " + durationInSec + " sec, and there were " + totalInsertedEvents + " events found.";
        LOG.info(feedback);
        return feedback;
    }

    private void migrateRuns(URL oldBaseUrl) throws IOException {

        List<Event> startTestRunEvents = getStartOrFinishTestRunEvents(oldBaseUrl, "startTestRun");
        LOG.info("I found " + startTestRunEvents.size() + " start events");

        for (Event startTestRunEvent : startTestRunEvents) {
            long oldInserted = totalInsertedEvents;
            LOG.info("Searching events for test run {}", startTestRunEvent.getTestRunId());
            String runId = startTestRunEvent.get("runId");
            List<Event> jobStatusEvents = getEventsPerRunId(oldBaseUrl, "jobStatus", runId);
            Event toInsertStartEvent = startTestRunEvent;

            if (!jobStatusEvents.isEmpty()) {
                Map<String, Object> otherProperties = new HashMap<>();
                otherProperties.putAll(jobStatusEvents.get(0).getProperties());
                otherProperties.putAll(startTestRunEvent.getProperties());
                toInsertStartEvent = new Event("executionStarted", otherProperties);
            }
            insertEvent(toInsertStartEvent);
            insertEvents(jobStatusEvents);
            migrateResults(oldBaseUrl, toInsertStartEvent.getTestRunId(), toInsertStartEvent.<String>get("testSpecification"));
            insertCorrespondingImportEvent(toInsertStartEvent);

            List<Event> finishTestRunEvents = getEventsPerRunId(oldBaseUrl, "finishTestRun", runId);
            Event toInsertFinishEvent = null;
            if (finishTestRunEvents.isEmpty()) {
                LOG.info("No finish event. Will generate one.");
                toInsertFinishEvent = compensateMissingFinishedEvent(runId, jobStatusEvents);
            } else if (!jobStatusEvents.isEmpty()) {
                Map<String, Object> otherProperties = new HashMap<>();
                otherProperties.putAll(jobStatusEvents.get(0).getProperties());
                otherProperties.putAll(finishTestRunEvents.get(0).getProperties());
                toInsertFinishEvent = new Event("executionFinished", otherProperties);
            }

            if (toInsertFinishEvent != null) {
                insertEvent(toInsertFinishEvent);
            } else {
                LOG.warn("Was not able to produce a finishedExecution event for {}", startTestRunEvent.getTestRunId());
            }
            LOG.info("Inserted {} events for run {}", totalInsertedEvents - oldInserted, startTestRunEvent.getTestRunId());
        }
    }

    private void insertCorrespondingImportEvent(Event startEvent) {
        Map<String, Object> otherProperties = new HashMap<>();
        otherProperties.put("runId", startEvent.get("runId"));
        otherProperties.put("timestamp", startEvent.get("timestamp"));
        otherProperties.put("lastModified", startEvent.get("timestamp"));
        otherProperties.put("testSpecification", startEvent.get("testSpecification"));
        insertEvent(new Event("importStarted", otherProperties));
        insertEvent(new Event("importFinished", otherProperties));
    }

    private void insertEvents(List<Event> events) {
        for (Event event : events) {
            insertEvent(event);
        }
    }

    private void insertEvent(Event event) {
        LOG.debug("Insert new event for {} {}", event.getType(), event.getTestRunId());
        eventRepository.insert(event);
        totalInsertedEvents++;
    }

    private Event compensateMissingFinishedEvent(String runId, List<Event> jobStatusEvents) {
        if (!jobStatusEvents.isEmpty()) {
            Event lastFinishedJobStatus = getLastFinishTimeEvent(jobStatusEvents);
            Map<String, Object> otherProperties = new HashMap<>();
            addParameters(otherProperties, jobStatusEvents.get(0));
            otherProperties.put(Event.TEST_SPECIFICATION, convertToSimpleName((String) lastFinishedJobStatus.get("testSpecification")));
            otherProperties.put(Event.RUN_ID, runId);
            otherProperties.put(Event.TIMESTAMP, lastFinishedJobStatus.get("timestamp"));
            // get parameters from any job status event and put them in finished event
            Event compensatedFinishEvent = new Event("executionFinished", otherProperties);
            return compensatedFinishEvent;
        }
        return null;
    }


    private String convertToSimpleName(String testSpecification) {
        if (testSpecification != null && testSpecification.startsWith("Configuration/TestSetDefinitions/")) {
            return testSpecification.substring(testSpecification.lastIndexOf("/") + 1);
        }
        return testSpecification;
    }

    private Event getLastFinishTimeEvent(List<Event> list) {
        Event lastFinishTimeEvent = null;
        for (Event jobStatus : list) {
            if (lastFinishTimeEvent == null) {
                lastFinishTimeEvent = jobStatus;
            } else {
                String lastFinishTimeEventFinishTime = lastFinishTimeEvent.get("finishedTime");
                String jobStatusFinishTime = jobStatus.get("finishedTime");
                if (lastFinishTimeEventFinishTime != null && jobStatusFinishTime != null && Long.parseLong(lastFinishTimeEventFinishTime) < Long.parseLong(jobStatusFinishTime)) {
                    lastFinishTimeEvent = jobStatus;
                }
            }
        }
        return lastFinishTimeEvent;
    }

    private List<Event> getStartOrFinishTestRunEvents(URL oldBaseUrl, String startOrFinish) throws IOException {
        List<Event> startTestRunEvents = new ArrayList<>();
        HttpURLConnection connToOldES = null;
        InputStream inputStream = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:" + startOrFinish + "&size=1000000");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
            if (responseCodeOnOldMappingUrl == 200) {
                inputStream = connToOldES.getInputStream();
                JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                JSONObject hitsAsObject = (JSONObject) result.get("hits");
                JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                for (int i = 0; i < realHits.size(); i++) {
                    JSONObject startOrFinishTestRun = (JSONObject) realHits.get(i);
                    JSONObject source = (JSONObject) startOrFinishTestRun.get("_source");
                    Map<String, Object> otherProperties = new HashMap<>();
                    for (String key : source.keySet()) {
                        if (key.equals("testSetId")) {
                            String convertToSimpleName = convertToSimpleName((String) source.get(key));
                            otherProperties.put(Event.TEST_SPECIFICATION, convertToSimpleName);
                        }
                        if (key.equals("run_id")) {
                            otherProperties.put(Event.RUN_ID, source.get(key));
                        }
                        if (key.equals("_ts")) {
                            otherProperties.put(Event.TIMESTAMP, source.get(key));
                        }
                    }
                    // get parameters from any job status event and put them in finished event
                    String modernType = "executionStarted";
                    if ("finishTestRun".equals(startOrFinish)) {
                        modernType = "executionFinished";
                    }
                    Event modernEvent = new Event(modernType, otherProperties);
                    startTestRunEvents.add(modernEvent);
                }
            } else {
                LOG.error("Could not retrieve instances of the " + startOrFinish + " type");
            }
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
            if (connToOldES != null) connToOldES.disconnect();
        }
        return startTestRunEvents;
    }

    private void addParameters(Map<String, Object> properties, Event event) {
        // event contains parameters. add these to the properties
        for (int i = 0; i < propsToCopy.length; i++) {
            if (event.hasProperty(propsToCopy[i])) {
                properties.put(propsToCopy[i], event.get(propsToCopy[i]));
            }
        }
    }

    private List<Event> getEventsPerRunId(URL oldBaseUrl, String eventType, String testRunId) throws IOException {
        List<Event> jobStatusEvents = new ArrayList<>();
        Set<String> alreadyCreatedSpecifications = new HashSet<>();
        HttpURLConnection connToOldES = null;
        InputStream inputStream = null;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:" + eventType + "%20AND%20run_id:\"" + testRunId + "\"&size=1000000");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
            if (responseCodeOnOldMappingUrl == 200) {
                inputStream = connToOldES.getInputStream();
                JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                JSONObject hitsAsObject = (JSONObject) result.get("hits");
                JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                LOG.info("Have to process " + realHits.size() + " " + eventType + " Events");
                for (int i = 0; i < realHits.size(); i++) {
                    JSONObject startOrFinishTestRun = (JSONObject) realHits.get(i);
                    JSONObject source = (JSONObject) startOrFinishTestRun.get("_source");
                    Map<String, Object> otherProperties = new HashMap<>(source);

                    if (otherProperties.containsKey("testSetDefinition")) {
                        otherProperties.put("testSpecification", convertToSimpleName((String) (source.get("testSetDefinition"))));
                    } else if (otherProperties.containsKey("testSetId")) {
                        otherProperties.put("testSpecification", convertToSimpleName((String) (source.get("testSetId"))));
                    }
                    if (otherProperties.containsKey("run_id")) {
                        otherProperties.put(Event.RUN_ID, source.get("run_id"));
                    }
                    otherProperties.put("timestamp", source.get("_ts"));
                    if (otherProperties.containsKey("status") && otherProperties.get("status").equals("finished")) {
                        Number started = (Number) source.get("started");
                        long duration = 0;
                        if (started != null && source.get("_ts") != null) {
                            duration = ((Number) source.get("_ts")).longValue() - started.longValue();
                        }
                        otherProperties.put("started", started);
                        otherProperties.put("duration", duration);
                        otherProperties.put(Event.IMPORT_FINISHED_RESULT, source.get("reason"));
                    }
                    if (otherProperties.containsKey("testset")) {
                        otherProperties.put("name", semicolonSeparate((String) source.get("testset")));
                    }

                    Event jobStatusEvent = new Event(eventType, otherProperties);

                    if ("jobStatus".equals(eventType) && jobStatusEvent.hasProperty("testSpecification")) {
                        String testSpecification = jobStatusEvent.get("testSpecification");
                        if (!alreadyCreatedSpecifications.contains(testSpecification)) {
                            Host host = null;
                            if (jobStatusEvent.hasProperty("jenkinsUri") && jobStatusEvent.hasProperty("jobName")) {
                                host = createJenkinsHostInJcr((String) jobStatusEvent.get("jenkinsUri"), testSpecification, (String) jobStatusEvent.get("jobName"));
                            }
                            createTestSpecificationInJcr(testSpecification, host);
                            alreadyCreatedSpecifications.add(testSpecification);
                        }
                    }
                    jobStatusEvents.add(jobStatusEvent);
                }
            } else {
                LOG.error("Could not retrieve instances of the jobStatus type");
            }
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
        }
        return jobStatusEvents;
    }

    private Host createJenkinsHostInJcr(String jenkinsUri, String testSpecification, String jobName) {
        Host host = DescriptorRegistry.getDescriptor(Type.valueOf("overthere.JenkinsHost")).newInstance("");
        host.setId("Infrastructure/" + testSpecification + " Host");
        // set minimal amount of properties: just those needed for reporting
        host.setProperty("address", jenkinsUri);
        host.setProperty("jobName", jobName);
        repository.createOrUpdate(host);
        LOG.info("New jenkins host created in JCR " + host.getId());
        return host;
    }

    private void createTestSpecificationInJcr(String testSpecificationName, Host host) {
        TestSpecification ts = DescriptorRegistry.getDescriptor(Type.valueOf("bol.SlicedFitNesse")).newInstance("");
        ts.setId("Configuration/TestSpecifications/" + testSpecificationName);
        ts.setTestToolName("FitNesse");
        ts.setSearchPattern("**");
        ts.setWorkingDirectory(".");
        ts.setProperty("importTestResults", false);
        ts.setProperty("commandLine", "-");
        ts.setProperty("timeout", 0);

        if (host != null) {
            ts.setHost(host);
        }
        repository.createOrUpdate(ts);
        LOG.info("New test specification created in JCR " + ts.getId());
    }

    private int migrateResults(URL oldBaseUrl, UUID testRunId, String testSpecification) {
        HttpURLConnection connToOldES = null;
        InputStream inputStream = null;
        int cnt = 0;
        try {
            URL oldMappingUrl = new URL(oldBaseUrl, "xltest/_search?q=type:result%20AND%20run_id:\"" + testRunId.toString() + "\"&size=1000000");
            connToOldES = (HttpURLConnection) oldMappingUrl.openConnection();
            int responseCodeOnOldMappingUrl = connToOldES.getResponseCode();
            if (responseCodeOnOldMappingUrl == 200) {
                inputStream = connToOldES.getInputStream();
                JSONObject result = (JSONObject) JsonProviderFactory.createProvider().parse(inputStream);

                JSONObject hitsAsObject = (JSONObject) result.get("hits");
                JSONArray realHits = (JSONArray) hitsAsObject.get("hits");
                LOG.info("Have to process " + realHits.size() + " Result Events");
                // the old result events do not have "testSpecification", so we stick it in below.
                // we derive it from any jobStatus event
                for (int i = 0; i < realHits.size(); i++) {
                    JSONObject resultEvent = (JSONObject) realHits.get(i);
                    JSONObject source = (JSONObject) resultEvent.get("_source");
                    Map<String, Object> otherProperties = new HashMap<>(source);
                    // 13:36:56: functionalResult ->
                    // {"duration":38,"name":"DemoSuite;Backend;UseCase001;TestCase002","result":"PASSED","right":3,
                    // "runId":"70a03821-85e9-11e4-a225-c8e0eb15ab7f","testSpecification":"demoFitNesse",
                    //"timestamp":1418647016000,"type":"functionalResult","wrong":0}

                    if (otherProperties.containsKey("run_id")) {
                        otherProperties.put(Event.RUN_ID, source.get("run_id"));
                        otherProperties.remove("run_id");
                    }
                    if (otherProperties.containsKey("testset")) {
                        otherProperties.put("name", semicolonSeparate((String) source.get("testset")));
                        otherProperties.remove("testset");
                    }
                    if (otherProperties.containsKey("time")) {
                        otherProperties.put("duration", (long) (((Number) source.get("time")).doubleValue() * 1000));
                        otherProperties.remove("time");
                    }
                    if (!otherProperties.containsKey("duration")) {
                        otherProperties.put("duration", 0);
                    }
                    // next, there is no "testSpecification" in the old results, needed by reports, so we slip it in (derived from the run)
                    otherProperties.put("testSpecification", testSpecification);
                    // properties 'right' and 'wrong' have been removed in the listener, so can't set those two!
                    cnt++;

                    Event functionalResult = new Event("functionalResult", otherProperties);
                    insertEvent(functionalResult);
                }
            } else {
                LOG.error("Could not retrieve instances of the jobStatus type");
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (connToOldES != null) connToOldES.disconnect();
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // just ignore
                }
            }
        }
        return cnt;
    }

    private String semicolonSeparate(String fitNesseTestCaseName) {
        return fitNesseTestCaseName.replaceAll("\\.", ";");
    }


}

