package com.xebialabs.xltest.repository;


import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.ExecutableTestSpecification;
import com.xebialabs.xltest.domain.TestSpecification;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

@Component
public class TestSpecificationRepository {

    private final RepositoryService repository;
    private final HostRepository hosts;

    @Autowired
    public TestSpecificationRepository(RepositoryService repositoryService, HostRepository hosts) {
        this.repository = repositoryService;
        this.hosts = hosts;
    }

    public TestSpecification get(String testSpecificationName) {
        return repository.read("Configuration/TestSpecifications/" + testSpecificationName);
    }

    public List<TestSpecification> findBy(String testToolName, String searchPattern, String workingDirectory) {
        SearchParameters criteria = new SearchParameters().setType(valueOf(TestSpecification.class));
        criteria.addProperty("testToolName", testToolName);
        criteria.addProperty("searchPattern", searchPattern);
        criteria.addProperty("workingDirectory", workingDirectory);
        List<TestSpecification> result = repository.listEntities(criteria);
        if (result == null) return Collections.emptyList();
        return result;
    }

    public List<TestSpecification> findWithHost(String testToolName, String searchPattern, String address, String workingDirectory, int port, String hostType) {
        List<TestSpecification> testSpecifications = findBy(testToolName, searchPattern, workingDirectory);
        List<TestSpecification> results = new ArrayList<>();
        if (testSpecifications.size() > 1) {
            Host host = hosts.findHost(address == null ? "" : address, port, hostType);
            if (host != null) {
                // filter on references to this host
                for (TestSpecification ts : testSpecifications) {
                    if (ts.getHost().getId().equals(host.getId())) {
                        results.add(ts);
                    }
                }
            }
        } else if (testSpecifications.size() == 1) {
            return testSpecifications;
        }
        return testSpecifications;
    }


    public TestSpecification findFirstWithHost(String testToolName, String searchPattern, String address, String workingDirectory, int port, String hostType) {
        List<TestSpecification> testSpecifications = findWithHost(testToolName, searchPattern, address, workingDirectory, port, hostType);
        if (testSpecifications.size() == 1) return testSpecifications.get(0);
        return null; // when < 1 nothing found, when > 1 we don't know what to choose so we return null
    }

    public TestSpecification findOrCreateTestSpecificationWithJenkinsHost(String jobName, String testToolName, String searchPattern, String jenkinsUrl) {
        TestSpecification testSpecification = findFirstWithHost(
                testToolName,
                searchPattern,
                jenkinsUrl,
                ".",
                0,
                HostRepository.HOST_TYPE_OVERTHERE_JENKINS
        );

        if (testSpecification != null) {
            return testSpecification;
        }

        testSpecification = new ExecutableTestSpecification();
        testSpecification.setId("Configuration/TestSpecifications/" + jobName);
        testSpecification.setWorkingDirectory(".");
        testSpecification.setTestToolName(testToolName);
        testSpecification.setSearchPattern(searchPattern);
        ((ExecutableTestSpecification) testSpecification).setCommandLine("echo");
        ((ExecutableTestSpecification) testSpecification).setImportTestResults(false);

        Host host = hosts.findOrCreateJenkinsHost(jenkinsUrl, jobName);
        testSpecification.setHost(host);
        repository.create(testSpecification);
        return testSpecification;
    }
}
