package com.xebialabs.xltest.serialize.metadata;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Map;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.xltype.serialization.json.JsonWriter;

@Component
@Provider
@Produces({MediaType.APPLICATION_JSON})
public class PropertyDescriptorJsonWriter implements MessageBodyWriter<PropertyDescriptor> {

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return true;
    }

    @Override
    public long getSize(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders, OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(propertyDescriptor).getBytes());
    }

    @VisibleForTesting
    String toJson(PropertyDescriptor propertyDescriptor) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writePropertyDescriptor(writer, propertyDescriptor);
        return stringWriter.toString();
    }

    static void writePropertyDescriptor(JsonWriter writer, PropertyDescriptor propertyDescriptor) {
        writer.object();
        writer.key("name").value(propertyDescriptor.getName());
        writer.key("fqn").value(propertyDescriptor.getFqn());
        writer.key("label").value(propertyDescriptor.getLabel());
        writer.key("kind").value(propertyDescriptor.getKind().name());
        writer.key("description").value(propertyDescriptor.getDescription());
        writer.key("category").value(propertyDescriptor.getCategory());
        writer.key("asContainment").value(propertyDescriptor.isAsContainment());
        writer.key("inspection").value(propertyDescriptor.isInspectionProperty());
        writer.key("required").value(propertyDescriptor.isRequired());
        writer.key("requiredInspection").value(propertyDescriptor.isRequiredForInspection());
        writer.key("password").value(propertyDescriptor.isPassword());
        writer.key("transient").value(propertyDescriptor.isTransient());
        writer.key("size").value(propertyDescriptor.getSize().name());
        if (propertyDescriptor.isHidden()) {
            writer.key("hidden").value(true);
        }
        if (propertyDescriptor.getDefaultValue() != null) {
            Object defaultValue = propertyDescriptor.getDefaultValue();
            if (defaultValue instanceof Collection) {
                valuesAsStrings(writer.key("defaultValue"), (Collection<?>) defaultValue);
            } else if (defaultValue instanceof Map) {
                mapAsStrings(writer.key("defaultValue"), (Map<?, ?>) defaultValue);
            } else {
                writer.key("defaultValue").value(defaultValue);
            }
        }
        if (propertyDescriptor.getKind() == PropertyKind.CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI || propertyDescriptor.getKind() == PropertyKind.SET_OF_CI) {
            writer.key("referencedType").value(propertyDescriptor.getReferencedType().toString());
        }
        if (propertyDescriptor.getKind() == PropertyKind.ENUM) {
            valuesAsStrings(writer.key("enumValues"), propertyDescriptor.getEnumValues());
        }
        writer.endObject();
    }

    private static void valuesAsStrings(JsonWriter writer, Collection<?> values) {
        writer.array();
        for (Object item : values) {
            writer.value(item);
        }
        writer.endArray();
    }

    private static void mapAsStrings(JsonWriter writer, Map<?, ?> map) {
        writer.object();
        for (Object key : map.keySet()) {
            writer.key(String.valueOf(key));
            writer.value(map.get(key));
        }
        writer.endObject();
    }

}
