package com.xebialabs.xltest.jmeter;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.PerformanceResultsSummary;
import com.xebialabs.xltest.domain.SamplingPerformanceResultsSummary;
import com.xebialabs.xltest.importers.AbstractPerformanceSummaryResultsImporter;

public class JMeterXmlImporter extends AbstractPerformanceSummaryResultsImporter {
    private static final Logger LOG = LoggerFactory.getLogger(JMeterXmlImporter.class);

    private final OverthereFile jmeterXmlFile;

    protected static final String EL_TEST_RESULTS = "testResults";
    protected static final String EL_HTTP_SAMPLE = "httpSample";

    protected static final QName ATTR_VERSION = QName.valueOf("version");
    protected static final QName ATTR_S = QName.valueOf("s");
    protected static final QName ATTR_TS = QName.valueOf("ts");
    protected static final QName ATTR_T = QName.valueOf("t");

    public JMeterXmlImporter(OverthereFile jmeterXmlFile) {
        this.jmeterXmlFile = jmeterXmlFile;
    }

    @Override
    public long getTimestamp() {
        return jmeterXmlFile.lastModified();
    }

    @Override
    public String getImportablePath() {
        return jmeterXmlFile.getPath();
    }

    @Override
    public PerformanceResultsSummary constructPerformanceResultsSummary() throws JMeterImportFailedException {
        try {
            XMLInputFactory factory = XMLInputFactory.newInstance();
            XMLEventReader reader = factory.createXMLEventReader(jmeterXmlFile.getInputStream(), "UTF-8");

            SamplingPerformanceResultsSummary performanceResultsSummary = new SamplingPerformanceResultsSummary();

            while (reader.hasNext()) {
                XMLEvent e = reader.nextEvent();
                if (e.isStartElement()) {
                    StartElement element = e.asStartElement();

                    if (EL_TEST_RESULTS.equals(element.getName().getLocalPart())) {
                        String version = element.getAttributeByName(ATTR_VERSION).getValue();
                        LOG.info("Importing JMeter XML format {}", version);
                    } else if (EL_HTTP_SAMPLE.equals(element.getName().getLocalPart())) {
                        String sampleResult = getAttribute(element, ATTR_S);

                        long timestamp = Long.parseLong(getAttribute(element, ATTR_TS));
                        long elapsed = Long.parseLong(getAttribute(element, ATTR_T));

                        performanceResultsSummary.updateWithSample(timestamp, elapsed, sampleResult);
                    } else {
                        if (LOG.isTraceEnabled()) {
                            LOG.trace("Ignoring type={} element={}", e.getEventType(), e);
                        }
                    }
                }
            }
            return performanceResultsSummary;
        } catch (XMLStreamException e) {
            throw new JMeterImportFailedException(String.format("Could not read JMeter XML data from: '%s'", jmeterXmlFile.toString()), e);
        }
    }

    private String getAttribute(StartElement item, QName attributeName) {
        Attribute attr = item.getAttributeByName(attributeName);
        return attr.getValue();
    }

    @Override
    public String toString() {
        return String.format("JMeterXmlImporter(%s)", this.jmeterXmlFile);
    }

    /**
     *

     Taken from JMeter documentation: http://jmeter.apache.org/usermanual/listeners.html (14.7)

     The sample attributes have the following meaning:

     Attribute	Content
     by	Bytes
     de	Data encoding
     dt	Data type
     ec	Error count (0 or 1, unless multiple samples are aggregated)
     hn	Hostname where the sample was generated
     it	Idle Time = time not spent sampling (milliseconds) (generally 0)
     lb	Label
     lt	Latency = time to initial response (milliseconds) - not all samplers support this
     na	Number of active threads for all thread groups
     ng	Number of active threads in this group
     rc	Response Code (e.g. 200)
     rm	Response Message (e.g. OK)
     s	Success flag (true/false)
     sc	Sample count (1, unless multiple samples are aggregated)
     t	Elapsed time (milliseconds)
     tn	Thread Name
     ts	timeStamp (milliseconds since midnight Jan 1, 1970 UTC)
     varname	Value of the named variable (versions of JMeter after 2.3.1)

     */
}
