package com.xebialabs.xltest.jenkins;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.jboss.resteasy.util.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Strings;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;

import com.xebialabs.overthere.*;
import com.xebialabs.overthere.spi.AddressPortMapper;
import com.xebialabs.overthere.spi.BaseOverthereConnection;
import com.xebialabs.overthere.spi.OverthereConnectionBuilder;
import com.xebialabs.overthere.spi.Protocol;

import static com.xebialabs.overthere.ConnectionOptions.ADDRESS;
import static com.xebialabs.overthere.ConnectionOptions.PASSWORD;
import static com.xebialabs.overthere.ConnectionOptions.USERNAME;
import static java.lang.String.format;

@Protocol(name = "jenkins")
public class JenkinsConnection extends BaseOverthereConnection implements OverthereConnectionBuilder {

    private static final Logger LOG = LoggerFactory.getLogger(JenkinsConnection.class);

    public static final String JENKINS_PROTOCOL = "jenkins";

    /**
     * The name of the job.
     */
    public static final String JOB_NAME = "jobName";

    /**
     * This parameter is set so we can track the job, based on the parameter provided.
     */
    public static final String UNIQUE_ID_PARAMETER = "XL_TEST_RUN_ID";
    /**
     * Additional parameters needed for teh Job, such as browser, environment, tags, ....
     */
    public static final String JOB_PARAMETERS = "jobParameters";

    public static final String TIMEOUT = "timeout";

    private final URL address;

    private final String username;

    private final String password;

    private final String jobName;

    private final Map<String, String> jobParameters;

    private final int timeout;

    private File localWorkspaceDirectory;

    public JenkinsConnection(String protocol, ConnectionOptions options, AddressPortMapper mapper) {
        super(protocol, options, mapper, true);
        String addr = options.get(ADDRESS);
        try {
            address = new URL(addr);
        } catch (MalformedURLException mue) {
            //Should never happen, url has been parsed before.
            throw new RuntimeIOException(mue);
        }
        username = options.getOptional(USERNAME);
        password = options.getOptional(PASSWORD);
        jobName = options.get(JOB_NAME);
        jobParameters = options.getOptional(JOB_PARAMETERS);
        timeout = options.get(TIMEOUT, JenkinsQueueInspector.ONE_HOUR);
    }


    @Override
    public OverthereFile getFile(String hostPath) {
        // TODO: get path from Jenkins workspace. Watch out for sync from slave to master.
        LOG.info("Looking for path: {}", hostPath);
        return new JenkinsFile(this, hostPath);
    }

    @Override
    public OverthereFile getFile(OverthereFile parent, String child) {
        LOG.info("Looking for path: {}/{}", parent, child);
        return new JenkinsFile(this, (JenkinsFile) parent, child);
    }

    public void execute(Map<String, Object> parameters) {
        final OverthereProcess process = startProcess(parameters);
        try {
            process.waitFor();
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();

            LOG.info("Execution interrupted, destroying the process.");
            process.destroy();

            throw new RuntimeIOException("Execution interrupted", exc);
        }
    }

    /**
     * Starts a command with its argument and returns control to the caller.
     *
     * @param commandLine the command line to execute.
     * @return an object representing the executing command or <tt>null</tt> if this is not supported by the host
     * connection.
     */
    @Override
    public OverthereProcess startProcess(CmdLine commandLine) {
        Map<String, Object> parameters = new HashMap<>();
        if (jobParameters != null) {
            parameters.putAll(jobParameters);
        }
        parameters.put("commandLine", commandLine.toCommandLine(getHostOperatingSystem(), false));
        return startProcess(parameters);
    }

    public OverthereProcess startProcess(Map<String, Object> jobParameters) {
        final String uniqueId = jobParameters.containsKey(UNIQUE_ID_PARAMETER) ? jobParameters.get(UNIQUE_ID_PARAMETER).toString() : UUID.randomUUID().toString();
        Map<String, Object> parameters = new HashMap<>(jobParameters);
        parameters.put(UNIQUE_ID_PARAMETER, uniqueId);

        final JenkinsQueueInspector jenkinsQueueInspector = new JenkinsQueueInspector(address, jobName, username, password, UNIQUE_ID_PARAMETER, uniqueId);

        int responseCode = buildJob(parameters);

        return new JenkinsOverthereProcess(jenkinsQueueInspector);
    }


    @Override
    public OverthereConnection connect() {
        return this;
    }

    @Override
    protected void doClose() {
        // no-op
    }

    @Override
    public OverthereFile getFileForTempFile(OverthereFile parent, String name) {
        throw new UnsupportedOperationException("Cannot deal with temp files on " + this);
    }


    @Override
    public String toString() {
        return JENKINS_PROTOCOL + ":" + username + "@" + address + "job/" + jobName;

    }

    protected File downloadJenkinsWorkspace(String hostPath) {
        if (localWorkspaceDirectory != null) {
            return localWorkspaceDirectory;
        }
        HttpURLConnection conn = null;
        try {
            File saveDir = newLocalWorkspaceDirectory(hostPath);
            final String fileName = "xltest-download.zip";
            URL zipUrl = new URL(address, format("job/%s/ws/%s/*zip*/%s", jobName, hostPath, fileName));
            conn = (HttpURLConnection) zipUrl.openConnection();
            addHeaders(conn);
            int responseCode = conn.getResponseCode();

            // always check HTTP response code first
            switch (responseCode) {
                case 200:
                    try (InputStream inputStream = conn.getInputStream()) {
                        extract(saveDir, inputStream);
                    }
                    LOG.debug("Done, unzipped in dir: " + localWorkspaceDirectory.getPath());
                    break;
                case 401:
                    if (username != null) {
                        throw new RuntimeIOException(format("Can not log on to Jenkins with user %s", username));
                    } else {
                        throw new RuntimeIOException(format("Authorization is required for Jenkins at %s", address));
                    }
                case 403:
                    throw new RuntimeIOException(format("Not allowed to access workspace folder '%s'", hostPath));
                case 404:
                    throw new RuntimeIOException(format("The folder '%s' could not be found in the workspace for job %s", hostPath, jobName));
                default:
                    throw new RuntimeIOException(format("Unable to download workspace folder '%s'(status code = %d)", hostPath, responseCode));
            }

            return localWorkspaceDirectory;
        } catch (IOException e) {
            localWorkspaceDirectory = null;
            throw new RuntimeIOException(format("Can not connect to Jenkins on URL %s (invalid url?)", address), e);
        } finally {
            if (conn != null) conn.disconnect();
        }
    }

    private File newLocalWorkspaceDirectory(String hostPath) {
        localWorkspaceDirectory = Files.createTempDir();
        File saveDir;
        String parent = new File(hostPath).getParent();
        if (parent != null) {
            saveDir = new File(localWorkspaceDirectory, parent);
        } else {
            saveDir = localWorkspaceDirectory;
            if ("".equals(hostPath) || "/".equals(hostPath) || ".".equals(hostPath)) {
                // Download top-level, Jenkins will package everything in a workspace folder.
                localWorkspaceDirectory = new File(localWorkspaceDirectory, "workspace");
            }
        }
        return saveDir;
    }

    private HttpURLConnection addHeaders(HttpURLConnection conn) {
        conn.setInstanceFollowRedirects(true);
        if (!Strings.isNullOrEmpty(username)) {
            String userPassword = username + ":" + (password != null ? password : "");
            String encoding = new String(Base64.encodeBytes(userPassword.getBytes()));
            conn.setRequestProperty("Authorization", "Basic " + encoding);
        }
        return conn;
    }

    protected void extract(File baseDir, InputStream is) throws IOException {
        final ZipInputStream zis = new ZipInputStream(is);
        ZipEntry entry;
        while ((entry = zis.getNextEntry()) != null) {
            if (!entry.isDirectory()) {
                final File file = new File(baseDir, entry.getName());
                LOG.debug("Copying {} into file {}", entry.getName(), file);
                Files.createParentDirs(file);
                Files.write(ByteStreams.toByteArray(zis), file);
            }
        }
    }

    public int buildJob(Map<String, Object> parameters) {
        HttpURLConnection conn = null;
        try {
            URL buildURL = new URL(address, format("job/%s/buildWithParameters?%s", jobName, makeContent(parameters)));
            LOG.info("Kicking off a job on Jenkins using this URL: " + buildURL.toString());
            conn = (HttpURLConnection) buildURL.openConnection();
            conn.setInstanceFollowRedirects(false);
            addHeaders(conn);
            // Need this to trigger the sending of the request
            int responseCode = conn.getResponseCode();
            LOG.info("Kicking off a job on Jenkins resulted in http responseCode: " + responseCode);
            return responseCode;
        } catch (MalformedURLException e) {
            throw new RuntimeIOException("Invalid URL", e);
        } catch (IOException e) {
            throw new RuntimeIOException("Unable to start job", e);
        } finally {
            if (conn != null) conn.disconnect();
        }
    }

    private String makeContent(Map<String, Object> parameters) {
        StringBuilder sb = new StringBuilder();
        if (parameters != null) {
            for (String key : parameters.keySet()) {
                sb.append(key);
                sb.append('=');
                sb.append(parameters.get(key));
                sb.append('&');
            }
        }
        return sb.toString().substring(0, sb.toString().length() - 1);
    }

    public File getAbsoluteFileInLocalWorkspaceDirectory(String path) {
        return new File(downloadJenkinsWorkspace(path), path);
    }

    public class JenkinsOverthereProcess implements OverthereProcess {
        private final JenkinsQueueInspector jenkinsQueueInspector;
        private boolean hasExited;
        private int exitCode;

        private JenkinsOverthereProcess(JenkinsQueueInspector jenkinsQueueInspector) {
            this.jenkinsQueueInspector = jenkinsQueueInspector;
            hasExited = false;
            exitCode = -1;
        }

        @Override
        public OutputStream getStdin() {
            return new ByteArrayOutputStream();
        }

        @Override
        public InputStream getStdout() {
            return new ByteArrayInputStream(new byte[] {});
        }

        @Override
        public InputStream getStderr() {
            return new ByteArrayInputStream(new byte[] {});
        }

        @Override
        public int waitFor() throws InterruptedException {
            if (!hasExited) {
                try {
                    exitCode = jenkinsQueueInspector.waitForJobToFinish(timeout);
                } catch (Exception e) {
                    throw new RuntimeIOException("Error while waiting for job " + jobName + " to finish", e);
                } finally {
                    hasExited = true;
                }
            }
            return exitCode;
        }

        @Override
        public void destroy() {
            if (!hasExited) {
                try {
                    jenkinsQueueInspector.stopBuild();
                } catch (Exception e) {
                    throw new RuntimeIOException("Unable to force-stop job " + jobName, e);
                } finally {
                    hasExited = true;
                }
            }
        }

        @Override
        public int exitValue() throws IllegalThreadStateException {
            if (!hasExited) {
                try {
                    int status = jenkinsQueueInspector.getJobStatus(timeout);
                    if (status != JenkinsQueueInspector.BUILD_STILL_RUNNING) {
                        exitCode = status;
                        hasExited = true;
                    }
                } catch (IOException e) {
                    throw new RuntimeException("Unable to determine build status", e);
                }
            }
            return exitCode;
        }

        public String getSlaveName() {
            return jenkinsQueueInspector.getSlaveName();
        }

        public Number getBuildNumber() {
            return jenkinsQueueInspector.getBuildNumber();
        }
    }
}
