package com.xebialabs.xltest.jenkins;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.CharStreams;
import com.google.common.net.UrlEscapers;
import com.jayway.jsonpath.JsonPath;

import static com.google.common.base.Strings.isNullOrEmpty;

/**
 * Sends events to XL Test with information found in the wiki pages.
 */
public class JenkinsQueueInspector {
    private static final Logger LOG = LoggerFactory.getLogger(JenkinsQueueInspector.class.getName());

    public static final String SUCCESS = "SUCCESS";
    public static final String UNSTABLE = "UNSTABLE";

    public static final int MINUTE = 60 * 1000;
    public static final int ONE_HOUR = 60 * MINUTE;
    public static final String XL_TEST_RUN_ID = "XL_TEST_RUN_ID";

    /* Exit codes */
    public static final int BUILD_SUCCESS = 0;
    public static final int BUILD_UNSTABLE = 1;
    public static final int BUILD_FAILURE = 2;
    public static final int BUILD_TIMEOUT = 3;
    public static final int BUILD_JENKINS_ERROR = 4;
    public static final int BUILD_STILL_RUNNING = -1;

    static final JsonPath BUILD_XLTEST_RESULT = JsonPath.compile("$.result");
    private final String uniqueIdParameterName;
    private final String uniqueId;
    private final String jobName;

    // the jenkins base url is something like: "http://localhost:8080"
    private final URL jenkinsBaseUrl;
    private final JenkinsRequest jenkinsRequest;
    private Number buildNumber;
    private String slaveName;
    private int nrOfJobsInQueueRetries = 120;
    private long timeoutTime;

    public JenkinsQueueInspector(URL url, String jobName, String username, String password, String uniqueIdParameterName, String uniqueId) {
        this.jenkinsBaseUrl = url;
        this.jobName = jobName;
        this.uniqueIdParameterName = uniqueIdParameterName;
        this.uniqueId = uniqueId;
        this.jenkinsRequest = isNullOrEmpty(username) ? new JenkinsRequest() : new JenkinsRequest(username, password);
    }

    public JenkinsQueueInspector(URL url, String jobName, JenkinsRequest jenkinsRequest, String uniqueIdParameterName, String uniqueId) {
        this.jenkinsBaseUrl = url;
        this.jobName = jobName;
        this.uniqueIdParameterName = uniqueIdParameterName;
        this.uniqueId = uniqueId;
        this.jenkinsRequest = jenkinsRequest;
    }

    public int inspectNumberOfJobsInBuildQueue() throws Exception {
        synchronized (this.getClass()) {
            StatusCodeAndContent statusCodeAndContent = null;
            int retries = nrOfJobsInQueueRetries; // keep retrying for a while
            while (retries > 0) {
                statusCodeAndContent = jenkinsRequest.httpGetRequest(new URL(jenkinsBaseUrl, "queue/api/json"));
                retries--;
                if (statusCodeAndContent.statusCode == 200) {
                    int numberOfItemsInQueue = statusCodeAndContent.pathCount("items");
                    LOG.debug("found {} jobs waiting in the queue", numberOfItemsInQueue);
                    return numberOfItemsInQueue;
                }
                Thread.sleep(500);
            }
            if (statusCodeAndContent != null) {
                throw new JenkinsConnectionException("Calling Jenkins failed with statuscode " + statusCodeAndContent.statusCode);

            } else {
                throw new JenkinsConnectionException("Calling Jenkins failed with unknown statuscode");
            }
        }
    }

    public String getSlaveName() {
        return slaveName;
    }
    public Number getBuildNumber() {
        return buildNumber;
    }

    public Number getQueueIdForBuildWithXlTestUrl() throws IOException {
        // http://jenkins.dev.bol.com/queue/api/json?tree=items[id,actions[parameters[*]]]&pretty=true
        URL url = new URL(jenkinsBaseUrl, "queue/api/json?tree=items[id,actions[parameters[*]]]");
        StatusCodeAndContent statusCodeAndContent = jenkinsRequest.httpGetRequest(url);
        if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling {} did return status code {}: {}", url, statusCodeAndContent.statusCode, statusCodeAndContent.json);
            return null;
        }
        return extractQueueIdFromBuildQueue((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
    }


    public boolean cancelQueuedBuild(Number id) throws Exception {
        // POST! http://localhost:8282/queue/cancelItem?id=2
        StatusCodeAndContent statusCodeAndContent = jenkinsRequest.httpPostRequest(new URL(jenkinsBaseUrl, "queue/cancelItem=" + id));
        return statusCodeAndContent.statusCode < 400;
    }

    public NumberSlavePair getNumberForBuildWithXlTestUrl() throws IOException {
        URL url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/api/json?tree=builds[number,result,url,builtOn,actions[parameters[*]]]");
        StatusCodeAndContent statusCodeAndContent = jenkinsRequest.httpGetRequest(url);
        if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling {} did return status code {}: {}", url, statusCodeAndContent.statusCode, statusCodeAndContent.json);
            return null;
        }
        Number buildNumber = extractBuildNumberFromJob((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
        String slave = extractSlaveFromJob((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
        return new NumberSlavePair(buildNumber, slave);
    }

    private String escape(String s) throws UnsupportedEncodingException {
        return UrlEscapers.urlFragmentEscaper().escape(s);
    }

    public NumberSlavePair waitForBuildNumber() throws Exception {

        int retries = 360;
        while (retries > 0) {
            retries--;
            NumberSlavePair numberSlavePair = getBuildNumberAndSlave();
            if (numberSlavePair != null) {
                return numberSlavePair;
            }
            Thread.sleep(5000);
        }
        LOG.error("Finding Jenkins Job for URL {} failed dramatically. We retried 360 times and now give up", uniqueId);
        return null;
    }

    public NumberSlavePair getBuildNumberAndSlave() throws IOException {
        NumberSlavePair numberSlavePair = getNumberForBuildWithXlTestUrl();
        if (numberSlavePair == null || numberSlavePair.getNumber() == null) {
            // it is not building, it might be in the queue
            Number queueIdForBuildWithXlTestUrl = getQueueIdForBuildWithXlTestUrl();
            if (queueIdForBuildWithXlTestUrl != null) {
                // job is queued. Just wait a bit
                LOG.debug("Found no Job with XL_TEST_RUN_ID={}. Will retry as we found it in the Queue with id: {}", uniqueId, queueIdForBuildWithXlTestUrl);
            } else {
                // job seems not building and it is also not queued. We blame jenkins and wait a bit
                LOG.debug("Found no Job with XL_TEST_RUN_ID={}. It is also not in the Queue. We blame Jenkins. Will retry anyway.", uniqueId);
            }
            // Note we waited 5000 in any case
            return null;
        } else {
            LOG.debug("Found Job with XL_TEST_RUN_ID={}. It is: {}" , uniqueId, numberSlavePair.getNumber());
            return numberSlavePair;
        }
    }

    public String buildResult() throws IOException {
        URL url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/" + buildNumber + "/api/json?tree=number,result");
        StatusCodeAndContent statusCodeAndContent = jenkinsRequest.httpGetRequest(url);

        if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling {} did return status code {}: {}", url, statusCodeAndContent.statusCode, statusCodeAndContent.json);
            return null;
        } else {
            return (String) BUILD_XLTEST_RESULT.read(statusCodeAndContent.json);
        }
    }

    public int waitForJobToFinish(int timeout) throws Exception {

        LOG.info("Job {} for run {} is waiting in the build queue", jobName, uniqueId);

        try {
            // Check if build is in build queue, wait until it's
            NumberSlavePair buildNumberSlavePair = waitForBuildNumber();
            if (buildNumberSlavePair == null) {
                // queue inspector gave up
                LOG.error("Jenkins Queue Inspector gave up. No clue what to do here");
                return BUILD_JENKINS_ERROR;

            }
            buildNumber = buildNumberSlavePair.getNumber();
            slaveName = buildNumberSlavePair.getSlave();

            // We add 50% extra time here, since we expect Jenkins to handle the time out initially.
            timeoutTime = System.currentTimeMillis() + ((timeout > 0 ? timeout * MINUTE : ONE_HOUR));

            if (LOG.isInfoEnabled()) {
                LOG.info("Job {} for run {} with time out {} should finish before {}", jobName, uniqueId, timeout, new Date(timeoutTime));
            }

            String buildResult;
            while ((buildResult = buildResult()) == null && !timedOut()) {
                sleepOnIt();
            }

            if (timedOut()) {
                LOG.info("Job {} for run {} with time out {} just timed out", jobName, uniqueId, timeout);
                stopBuild();
                return BUILD_TIMEOUT;
            } else {
                if (!SUCCESS.equals(buildResult)) {
//                    sendEvent(makeJobEvent(subRunId, "finished", buildResult == null ? "NULL" : buildResult.toLowerCase(), testSpecification, startMoment, slave, buildNumber));
                    LOG.info("Job {} for run {} did not finish successfully", jobName, uniqueId);
                    return BUILD_FAILURE;
                } else {
//                    sendEvent(makeJobEvent(subRunId, "finished", "success", testSpecification, startMoment, slave, buildNumber));
                    LOG.info("Job {} for run {} did finish successfully", jobName, uniqueId);
                    return BUILD_SUCCESS;
                }
            }
        } catch (InterruptedException ie) {
            cancelJobOnJenkins();
            throw ie;
        }
    }

    public int getJobStatus(int timeout) throws IOException {
        // Check if build is in build queue, wait until it's
        if (buildNumber == null) {
            NumberSlavePair buildNumberSlavePair = getBuildNumberAndSlave();
            if (buildNumberSlavePair == null) {
                return BUILD_STILL_RUNNING;
            }
            buildNumber = buildNumberSlavePair.getNumber();
            slaveName = buildNumberSlavePair.getSlave();
            // Set timeout as soon as we're out of the build queue
            timeoutTime = System.currentTimeMillis() + ((timeout > 0 ? timeout * MINUTE : ONE_HOUR));
        }

        if (LOG.isInfoEnabled()) {
            LOG.info("Job {} for run {} with time out {} should finish before {}", jobName, uniqueId, timeout, new Date(timeoutTime));
        }

        String buildResult = buildResult();
        boolean isTimedOut = timedOut();
        if (buildResult == null && isTimedOut) {
            return BUILD_STILL_RUNNING;
        }

        if (isTimedOut) {
            LOG.info("Job {} for run {} with time out {} just timed out", jobName, uniqueId, timeout);
            stopBuild();
            return BUILD_TIMEOUT;
        } else if (SUCCESS.equals(buildResult)) {
            LOG.info("Job {} for run {} did finish successfully", jobName, uniqueId);
            return BUILD_SUCCESS;
        } else if (UNSTABLE.equals(buildResult)) {
            LOG.info("Job {} for run {} did finish but is unstable", jobName, uniqueId);
            return BUILD_UNSTABLE;
        } else {
            LOG.info("Job {} for run {} did not finish successfully", jobName, uniqueId);
            return BUILD_FAILURE;
        }
    }

    private boolean timedOut() {
        return System.currentTimeMillis() > timeoutTime;
    }

    private void sleepOnIt() throws InterruptedException {
        // Should release thread and come back at some later time to check the status again.
        Thread.sleep(5000);
    }

    public void stopBuild() throws IOException {
        URL url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/" + buildNumber + "/stop");
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod("POST");
        try {
            if (conn.getResponseCode() != 200) {
                LOG.warn("Calling {} did return status code {}: {}", url, conn.getResponseCode(), CharStreams.toString(new InputStreamReader(conn.getInputStream())));
            }
        } finally {
            conn.disconnect();
        }
    }

    // This function does a best effort attempt to kill jobs running on Jenkins.
    protected void cancelJobOnJenkins() {
        Number queueId = null;
        try {
            queueId = getQueueIdForBuildWithXlTestUrl();
            if (queueId != null && cancelQueuedBuild(queueId)) {
                return;
            }
        } catch (Exception e) {
            LOG.error("Unable to remove build {} from queue", queueId, e);
        }

        try {
            if (this.buildNumber == null) {
                this.buildNumber = waitForBuildNumber().getNumber();
            }
            if (this.buildNumber != null) {
                stopBuild();
            }
        } catch (Exception e) {
            LOG.error("Unable to stop build {}/{}", jobName, this.buildNumber, e);
        }
    }

    Number extractQueueIdFromBuildQueue(Map<String, List<Map<String, Object>>> json) {
        return (Number) extractIdFromJenkinsResponse(json, "items", "id");
    }

    Number extractBuildNumberFromJob(Map<String, List<Map<String, Object>>> json) {
        return (Number) extractIdFromJenkinsResponse(json, "builds", "number");
    }

    String extractSlaveFromJob(Map<String, List<Map<String, Object>>> json) {
        return (String) extractIdFromJenkinsResponse(json, "builds", "builtOn");
    }

    private Object extractIdFromJenkinsResponse(Map<String, List<Map<String, Object>>> builds, String groupName, String idName) {
        // { "items": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "id": 123 }, ... ]}
        // { "builds": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "number": 123 }, ... ]}
        for (Map<String, Object> build : builds.get(groupName)) {
            List<Map<String, List<Map<String, String>>>> actions = ((List<Map<String, List<Map<String, String>>>>) build.get("actions"));
            if (actions == null || actions.size() == 0) {
                continue;
            }
            List<Map<String, String>> parameters = actions.get(0).get("parameters");
            if (parameters == null || parameters.size() == 0) {
                continue;
            }
            for (Map<String, String> parameter : parameters) {
                if (uniqueIdParameterName.equals(parameter.get("name")) && uniqueId.equals(parameter.get("value"))) {
                    return build.get(idName);
                }
            }
        }
        return null;
    }

    public void setNrOfJobsInQueueRetries(final int nrOfJobsInQueueRetries) {
        this.nrOfJobsInQueueRetries = nrOfJobsInQueueRetries;
    }
}
