package com.xebialabs.xltest.gatling;

import com.google.common.io.*;

import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.ImportFailedException;
import com.xebialabs.xltest.domain.PerformanceResultsSummary;
import com.xebialabs.xltest.importers.AbstractPerformanceSummaryResultsImporter;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class GatlingSimulationLogImporter extends AbstractPerformanceSummaryResultsImporter {
    private OverthereFile gatlingReportDir;
    private OverthereFile simulationLogFile;

    private String lastLine;
    private String firstLine;

    private long startTime;
    private long endTime;
    private String simulationName;

    public GatlingSimulationLogImporter(OverthereFile gatlingReportDir) throws ImportFailedException {
        this.gatlingReportDir = gatlingReportDir;
        simulationLogFile = gatlingReportDir.getFile("simulation.log");
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(simulationLogFile.getInputStream(), StandardCharsets.UTF_8))) {
            firstLine = bufferedReader.readLine();
            startTime = Line.parseLine(firstLine).getStartTime();
            simulationName = Line.parseLine(firstLine).getSimulationName();

            String line;
            while ((line = bufferedReader.readLine()) != null) {
                lastLine = line;
            }
            endTime = Line.parseLine(lastLine).getEndTime();
        } catch (IOException e) {
            throw new ImportFailedException(e);
        }
    }

    @Override
    public Event createImportStartedEvent(UUID testRunId) {
        return super.createImportStartedEvent(testRunId).setSimulationName(simulationName);
    }

    @Override
    public long getTimestamp() {
        return startTime;
    }

    OverthereFile getGatlingReportDir() {
        return gatlingReportDir;
    }

    @Override
    public String getImportablePath() {
        return simulationLogFile.getPath();
    }

    @Override
    public PerformanceResultsSummary constructPerformanceResultsSummary() throws ImportFailedException {
        try {
            File jsonReport = new File(gatlingReportDir.getPath() + File.separator + "js" + File.separator + "global_stats.json");
            if (!jsonReport.exists()) {
                throw new FileNotFoundException(jsonReport.getAbsolutePath());
            }

            JsonParser parser = new JsonParser();
            parser.parse(Files.toString(jsonReport, Charset.defaultCharset()));

            PerformanceResultsSummary performanceResultsSummary = new PerformanceResultsSummary();
            performanceResultsSummary.setEarliestTimestamp(startTime);
            performanceResultsSummary.setLatestTimestamp(endTime);

            performanceResultsSummary.setTotalNumberOfRequests((Integer) parser.getValue("numberOfRequests.total"));
            performanceResultsSummary.setTotalNumberOfOkRequests((Integer) parser.getValue("numberOfRequests.ok"));
            performanceResultsSummary.setTotalNumberOfNotOkRequests((Integer) parser.getValue("numberOfRequests.ko"));

            performanceResultsSummary.setMinResponseTime((Integer) parser.getValue("minResponseTime.total"));
            performanceResultsSummary.setMinResponseTimeOk((Integer) parser.getValue("minResponseTime.ok"));
            performanceResultsSummary.setMinResponseTimeNotOk((Integer) parser.getValue("minResponseTime.ko"));

            performanceResultsSummary.setMaxResponseTime((Integer) parser.getValue("maxResponseTime.total"));
            performanceResultsSummary.setMaxResponseTimeOk((Integer) parser.getValue("maxResponseTime.ok"));
            performanceResultsSummary.setMaxResponseTimeNotOk((Integer) parser.getValue("maxResponseTime.ko"));

            performanceResultsSummary.setMeanResponseTime((Integer) parser.getValue("meanResponseTime.total"));
            performanceResultsSummary.setMeanResponseTimeOk((Integer) parser.getValue("meanResponseTime.ok"));
            performanceResultsSummary.setMeanResponseTimeNotOk((Integer) parser.getValue("meanResponseTime.ko"));

            performanceResultsSummary.setStandardDeviation((Integer) parser.getValue("standardDeviation.total"));
            performanceResultsSummary.setStandardDeviationOk((Integer) parser.getValue("standardDeviation.ok"));
            performanceResultsSummary.setStandardDeviationNotOk((Integer) parser.getValue("standardDeviation.ko"));

            Map<String, Object> otherProperties = new HashMap<>();
            parsePercentilesAndPutOnMap(parser, otherProperties, "percentiles1");
            parsePercentilesAndPutOnMap(parser, otherProperties, "percentiles2");
            parseGroupAndPutOnMap(parser, otherProperties, "group1");
            parseGroupAndPutOnMap(parser, otherProperties, "group2");
            parseGroupAndPutOnMap(parser, otherProperties, "group3");
            parseGroupAndPutOnMap(parser, otherProperties, "group4");

            performanceResultsSummary.setOtherProperties(otherProperties);

            return performanceResultsSummary;
        } catch (RuntimeException | IOException e) {
            throw new ImportFailedException(e);
        }
    }

    private void parsePercentilesAndPutOnMap(JsonParser parser, Map<String, Object> otherProperties, String key) {
        otherProperties.put(key + ".total", (Integer) parser.getValue(key + ".total"));
        otherProperties.put(key + ".ok", (Integer) parser.getValue(key + ".ok"));
        otherProperties.put(key + ".ko", (Integer) parser.getValue(key + ".ko"));
    }

    private void parseGroupAndPutOnMap(JsonParser parser, Map<String, Object> otherProperties, String key) {
        otherProperties.put(key + ".name", (String) parser.getValue(key + ".name"));
        otherProperties.put(key + ".count", (Integer) parser.getValue(key + ".count"));
        otherProperties.put(key + ".percentage", (Integer) parser.getValue(key + ".percentage"));
    }

}
