import { interpolateLinks, interpolateUsers, wrapMentionsText } from './index';

describe('markdown helper', () => {
    describe('interpolateLinks', () => {
        it('should return original html if no links', () => {
            const html = '<div>i am not a link</div>';
            const result = interpolateLinks(html);
            expect(result).toBe('<div>i am not a link</div>');
        });

        it('should replace internal relative links correctly', () => {
            const html = '<a href="#/internal">Internal Link</a>';
            const result = interpolateLinks(html);
            expect(result).toBe('<a href="#/internal">Internal Link</a>');
        });

        it('should replace internal absolute links correctly', () => {
            const html = `<a href="${document.location.origin}/internal">Internal Link</a>`;
            const result = interpolateLinks(html);
            expect(result).toBe(`<a href="${document.location.origin}/internal">Internal Link</a>`);
        });

        it('should replace external links to open in new tab', () => {
            const html = '<a href="http://external.com">External Link</a>';
            const result = interpolateLinks(html);
            expect(result).toBe('<a href="http://external.com" rel="noreferrer noopener" target="_blank">External Link</a>');
        });
    });

    describe('interpolateUsers', () => {
        const users = [
            { username: 'john_doe', fullName: 'John Doe' },
            { username: 'jane_doe', fullName: 'Jane Doe' },
        ];

        it('should return original html if no mentions', () => {
            const html = '<strong>im just bold</strong>';
            const result = interpolateLinks(html);
            expect(result).toBe('<strong>im just bold</strong>');
        });

        it('should replace user mentions with full name', () => {
            const html = '<strong>@john_doe</strong>';
            const result = interpolateUsers(html, users);
            expect(result).toBe('<span class="mention" title="john_doe">@John Doe</span>');
        });

        it('should replace user mentions with username if full name is not available', () => {
            const html = '<strong>@jane_doe</strong>';
            const result = interpolateUsers(html, users);
            expect(result).toBe('<span class="mention" title="jane_doe">@Jane Doe</span>');
        });

        it('should not replace mentions if user is not found', () => {
            const html = '<strong>@unknown_user</strong>';
            const result = interpolateUsers(html, users);
            expect(result).toBe('<strong>@unknown_user</strong>');
        });
    });

    describe('wrapMentionsText', () => {
        it('should wrap mentions with double asterisks', () => {
            const text = 'Hello @john_doe';
            const result = wrapMentionsText(text);
            expect(result).toBe('Hello **@john_doe**');
        });

        it('should handle multiple mentions in text', () => {
            const text = 'Hello @john_doe and @jane_doe';
            const result = wrapMentionsText(text);
            expect(result).toBe('Hello **@john_doe** and **@jane_doe**');
        });

        it('should not wrap text without mentions', () => {
            const text = 'Hello world';
            const result = wrapMentionsText(text);
            expect(result).toBe('Hello world');
        });
    });
});
