import React from 'react';
import { Popover } from 'antd';
import PropTypes from 'prop-types';
import cs from 'classnames';
import { XlReactWidgetTable } from 'xl-react-components';
import { XlrMembersTags } from '..';
import throttle from 'lodash/throttle';
import { DotIconWrapper } from '../../../features/common/components/dot-icon-wrapper/dot-icon-wrapper';
import './teams-permissions-table.less';
import './permissions-table.less';
const initialTableState = {
    layout: [
        { id: 0, dataKey: 'name', title: 'Action' },
        { id: 1, dataKey: 'teams', title: 'Teams' },
    ],
};

export class PermissionsTable extends React.Component {
    static propTypes = {
        data: PropTypes.arrayOf(
            PropTypes.shape({
                id: PropTypes.string,
                name: PropTypes.string.isRequired,
                description: PropTypes.string,
                clazz: PropTypes.object,
                icon: PropTypes.string,
                teams: PropTypes.arrayOf(PropTypes.object),
                readOnly: PropTypes.bool.isRequired,
            }),
        ),
        loading: PropTypes.bool.isRequired,
        onAddTeam: PropTypes.func.isRequired,
        onRemoveTeam: PropTypes.func.isRequired,
        readOnly: PropTypes.bool,
        readOnlyUser: PropTypes.bool,
        teams: PropTypes.arrayOf(PropTypes.object),
    };

    static defaultProps = {
        readOnly: false,
    };

    constructor(props) {
        super(props);
        this.state = {
            filter: {
                name: '',
                teams: '',
            },
        };
    }

    createColumnDefinition = (headerDef) => ({
        dataField: headerDef.dataKey,
        sort: false,
        text: headerDef.title,
        formatter: this.formatCell(headerDef.dataKey),
    });

    formatCell = (dataKey) => (cell, row) => this._formatCell(dataKey, cell, row);

    _formatCell = (dataKey, cell, row) => {
        switch (dataKey) {
            case 'name':
                if (!row.id) {
                    return (
                        <React.Fragment>
                            <DotIconWrapper iconId={row.icon} />
                            <strong>{cell}</strong>
                        </React.Fragment>
                    );
                } else {
                    return (
                        <div className={cs('text-column', row.clazz)}>
                            {cell}
                            {row.description && (
                                <Popover
                                    content={<div>{row.description}</div>}
                                    mouseEnterDelay={0.1}
                                    mouseLeaveDelay={0}
                                    overlayClassName="permissions-table-v3-popover"
                                    placement="bottom"
                                >
                                    <i className="xl-icon help-icon"></i>
                                </Popover>
                            )}
                        </div>
                    );
                }
            case 'teams':
                if (!row.id) {
                    return <React.Fragment />;
                } else {
                    return this.formatTeamCell(cell, row);
                }
            default:
                return cell;
        }
    };

    formatTeamCell = (cell, row) => (
        <XlrMembersTags
            allowAddOnlyFromSuggestion={true}
            classNames={{ tagInput: 'tag-new role', selected: 'tag role' }}
            handleAddition={this.addTeamToPermission(row)}
            handleDelete={this.removeTeamFromPermission(row)}
            label="id"
            readOnly={this.props.readOnly || row.readOnly || this.props.readOnlyUser}
            suggestions={this.mapTeams(this.props.teams).filter((t) => !t.readOnly)}
            tags={this.mapTeams(cell).map((t) => ({ ...t, readOnly: t.readOnly || this.props.readOnly || this.props.readOnlyUser }))}
        />
    );

    mapTeams = (teams) => teams.map((t) => ({ ...t, id: t.teamName }));

    addTeamToPermission = (permission) => (team) => this.props.onAddTeam(permission, team);

    removeTeamFromPermission = (permission) => (index) => this.props.onRemoveTeam(permission, index);

    handleColumnFilterChange = throttle((filter) => {
        this.setState({ filter });
    }, throttleFilterChangeTimeout);

    getData = () => {
        const { filter } = this.state;
        const { data } = this.props;
        const name = filter.name.toLowerCase();
        const teams = filter.teams.toLowerCase();
        const hasFilters = !!(filter.name || filter.teams);

        if (hasFilters) {
            return data.filter((entry) => {
                return (
                    (filter.name && entry.name.toLowerCase().includes(name)) ||
                    (filter.teams && entry.teams.find((team) => team.teamName.toLowerCase().includes(teams)))
                );
            });
        }

        return data;
    };

    render() {
        const { loading } = this.props;
        const { filter } = this.state;
        return (
            <div className="permissions-table-v3 teams-permissions-table-react">
                <XlReactWidgetTable
                    className={'transparent'}
                    createColumnDefinition={this.createColumnDefinition}
                    data={this.getData()}
                    defaultErrorMessage={'Something wrong happened'}
                    defaultNoDataText={'No permissions found'}
                    displayOnlyLabelIfNoData={false}
                    filter={filter}
                    filterType="column"
                    keyField="name"
                    layout={initialTableState.layout}
                    loading={loading}
                    loadingMessage={'loading...'}
                    onFilterChange={this.handleColumnFilterChange}
                    // tableId={tableId}
                />
            </div>
        );
    }
}

const throttleFilterChangeTimeout = 500;
