import React from 'react';
import { mount } from 'enzyme/build';
import {
    calculatePercentageForReleaseOrWorkflow,
    calculatePercentageForStepper,
    calculatePercentage,
    ReleaseProgressComponent,
} from './release-progress.component';
import * as taskHelper from '../../../../features/tasks/helper/task.helper';

describe('release-progress', () => {
    describe('calculatePercentageForReleaseOrWorkflow', () => {
        jest.mock('../../../../features/tasks/helper/task.helper');
        const release = {
            extensions: {
                progress: { totalTasks: 10, totalRemainingTasks: 2 }, // fake numbers
            },
            tasks: [
                { status: 'COMPLETED' },
                { status: 'COMPLETED_IN_ADVANCE' },
                {
                    type: 'xlrelease.SequentialGroup',
                    status: 'IN_PROGRESS',
                    tasks: [{ status: 'COMPLETED' }, { status: 'IN_PROGRESS' }],
                },
            ],
        };

        it('should calculate workflow percentage', () => {
            jest.spyOn(taskHelper, 'getAllStepperTasks').mockReturnValue(release.tasks);
            expect(calculatePercentageForReleaseOrWorkflow(release, true)).toBe(60);
        });
        it('should calculate release percentage', () => {
            expect(calculatePercentageForReleaseOrWorkflow(release, false)).toBe(80);
        });
    });

    describe('calculatePercentageForStepper', () => {
        jest.mock('../../../../features/tasks/helper/task.helper');

        it('should return 0 when there are no tasks', () => {
            const release = { tasks: [] };
            jest.spyOn(taskHelper, 'getAllStepperTasks').mockReturnValue(release.tasks);
            expect(calculatePercentageForStepper(release)).toBe(0);
        });

        it('should return 100 when all tasks are completed', () => {
            const release = { tasks: [{ status: 'COMPLETED' }, { status: 'COMPLETED_IN_ADVANCE' }] };
            jest.spyOn(taskHelper, 'getAllStepperTasks').mockReturnValue(release.tasks);
            expect(calculatePercentageForStepper(release)).toBe(100);
        });

        it('should return correct percentage when some tasks are completed', () => {
            const release = { tasks: [{ status: 'COMPLETED' }, { status: 'COMPLETED_IN_ADVANCE' }, { status: 'FAILED' }] };
            jest.spyOn(taskHelper, 'getAllStepperTasks').mockReturnValue(release.tasks);
            expect(calculatePercentageForStepper(release)).toBe(67);
        });

        it('should return correct percentage when having tasks groups', () => {
            const release = {
                tasks: [
                    { status: 'COMPLETED' },
                    { status: 'COMPLETED_IN_ADVANCE' },
                    {
                        type: 'xlrelease.SequentialGroup',
                        status: 'IN_PROGRESS',
                        tasks: [{ status: 'COMPLETED' }, { status: 'IN_PROGRESS' }],
                    },
                ],
            };
            jest.spyOn(taskHelper, 'getAllStepperTasks').mockReturnValue(release.tasks);
            expect(calculatePercentageForStepper(release)).toBe(60);
        });
    });

    describe('calculatePercentage', () => {
        it('should return 0 when progress is undefined', () => {
            expect(calculatePercentage(undefined)).toBe(0);
        });

        it('should return 0 when there are no tasks', () => {
            const progress = { totalTasks: 0, totalRemainingTasks: 0 };
            expect(calculatePercentage(progress)).toBe(0);
        });

        it('should return 100 when all tasks are completed', () => {
            const progress = { totalTasks: 2, totalRemainingTasks: 0 };
            expect(calculatePercentage(progress)).toBe(100);
        });

        it('should return correct percentage when some tasks are completed', () => {
            const progress = { totalTasks: 3, totalRemainingTasks: 1 };
            expect(calculatePercentage(progress)).toBe(67);
        });
    });

    describe('ReleaseProgressComponent', () => {
        it('should render release progress', () => {
            const completedPercentage = 54.32;
            const releaseProgressComponent = mount(<ReleaseProgressComponent completedPercentage={completedPercentage} />);
            expect(releaseProgressComponent.find('.release-progress-bar-view').text()).toBe('54.32% Completed');
            expect(releaseProgressComponent.find('.release-progress-bar-view > .progress-bar-container > .progress-bar-line').prop('style')).toHaveProperty(
                'width',
                '54.32%',
            );
        });

        it('should return 0 if progress is undefined', () => {
            const progress = undefined;
            const completedPercentage = undefined;

            const releaseProgressComponent = mount(<ReleaseProgressComponent completedPercentage={completedPercentage} progress={progress} />);
            expect(releaseProgressComponent.find('.release-progress-bar-view').text()).toBe('0% Completed');
        });

        it('should return 0 when calculating the percetange', () => {
            expect(calculatePercentage(undefined)).toBe(0);
        });
    });
});
