import { MutableRefObject, useLayoutEffect, useState } from 'react';
import debounce from 'lodash/debounce';

export interface UseChipGroupWidths {
    chipCounterWidth: number;
    chipWidths: number[];
    containerWidth: number;
}

export const useChipGroup = (
    exceedingChipsCountRef: MutableRefObject<HTMLSpanElement | null>,
    elementsRef: MutableRefObject<HTMLSpanElement[]>,
    containerRef: MutableRefObject<HTMLDivElement | null>,
): UseChipGroupWidths => {
    const [chipWidths, setChipWidths] = useState<number[]>([]);
    const [containerWidth, setContainerWidth] = useState<number>(0);

    const chipCounterWidth = exceedingChipsCountRef.current?.getBoundingClientRect().width || 0;

    /* Get width for each chip so that it can be used to calculate how much chips can fit into the container  */
    useLayoutEffect(() => {
        const width = elementsRef.current.map((el) => el.getBoundingClientRect().width);
        setChipWidths(width);
    }, [elementsRef]);

    useLayoutEffect(() => {
        if (!containerRef.current) return;
        const handleResize = (width: number) => setContainerWidth(width);
        const handleDebounce = debounce(handleResize, 100);
        const observer = new ResizeObserver((entries) => {
            handleDebounce(entries[0].target.getBoundingClientRect().width);
        });
        observer.observe(containerRef.current);
        return () => {
            observer.disconnect();
            handleDebounce.cancel();
        };
    }, [containerRef]);

    return {
        chipWidths,
        containerWidth,
        chipCounterWidth,
    };
};
