import React, { MouseEvent, useState, useEffect } from 'react';
import ReactMarkdown from 'react-markdown';
import classNames from 'classnames';
import { DotButton, DotIconButton, DotInputText, DotTypography, DotTooltip } from '@digital-ai/dot-components';
import { HelpButton } from '../../../features/main-navigation/help-button/help-button';
import { MentionsInput, trigger as mentionPrefix } from '../../../features/common/components/mentions/mentions-input';
import { ReleaseVariable, User } from '../../../types';
import getAngularService from '../../../features/common/services/angular-accessor';
import { VariablesInterpolatorFactory } from '../../../features/tasks/types/angular';
import { mentionsRegex } from '../../../js/util/filters';
import './textarea-markdown.component.less';

export interface TextareaMarkdownProps {
    enableMentions?: boolean;
    inputId: string;
    isReadOnly?: boolean;
    isValueRequired?: boolean;
    maxRows?: number;
    name: string;
    onDelete?: (event: MouseEvent<HTMLButtonElement>) => void;
    onTextChange: (text: string) => void;
    placeholder?: string;
    rows: number;
    text: string;
    users?: User[];
    variables?: ReleaseVariable;
}

export const TextAreaMarkdown = ({
    enableMentions = false,
    inputId,
    isReadOnly = false,
    isValueRequired = true,
    maxRows,
    name,
    onDelete,
    onTextChange,
    placeholder,
    rows,
    text,
    users = [],
    variables,
}: TextareaMarkdownProps) => {
    const [editing, setEditing] = useState<boolean>(false);
    const [originalText, setOriginalText] = useState<string>(text);
    const [currentText, setCurrentText] = useState<string>(text);
    const variablesInterpolator = getAngularService('VariablesInterpolator') as never as VariablesInterpolatorFactory;

    useEffect(() => {
        setOriginalText(text);
        setCurrentText(text);
    }, [text]);

    const saveChanges = () => {
        const trimmedText = currentText.trim();
        if (isValueRequired && !trimmedText) {
            revertChanges();
            return;
        }
        onTextChange(trimmedText);
        setOriginalText(trimmedText);
        setEditing(false);
    };

    const revertChanges = () => {
        setCurrentText(originalText);
        setEditing(false);
    };

    const handleDelete = (event: MouseEvent<HTMLButtonElement>) => {
        onDelete && onDelete(event);
        setEditing(false);
    };

    const renderTextEditor = () => (
        <div>
            {enableMentions ? (
                <MentionsInput
                    id={inputId}
                    maxRows={maxRows}
                    multiline={true}
                    name={name}
                    onChange={(inputValue) => setCurrentText(inputValue)}
                    rows={rows}
                    users={users}
                    value={currentText}
                />
            ) : (
                <DotInputText
                    autoFocus
                    id={inputId}
                    multiline={true}
                    name={name}
                    onChange={(e) => setCurrentText(e.target.value)}
                    rows={rows}
                    rowsMax={maxRows}
                    value={currentText}
                />
            )}
            <div className="actions-wrapper">
                <DotButton data-testid="cancel-button" onClick={revertChanges} type="text">
                    Cancel
                </DotButton>
                <DotButton data-testid="save-button" onClick={saveChanges} type="primary">
                    Save
                </DotButton>
                <HelpButton helpLink="how-to/using-markdown-in-xl-release.html" />
            </div>
        </div>
    );

    const wrapMentionsText = (textWithMention: string) => {
        return textWithMention.replace(mentionsRegex, (match, username) => {
            const prefix = match.substr(0, match.indexOf(mentionPrefix));
            return `${prefix}**${mentionPrefix}${username}**`;
        });
    };

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const renderMarkDownStrongElement = ({ children, ...props }) => {
        if (enableMentions && children && children.length === 1 && children[0] && children[0].startsWith(mentionPrefix)) {
            const user = users.find((currentUser) => currentUser.username === children[0].substr(mentionPrefix.length));
            if (user) {
                return (
                    <span className="mention" title={user.username}>
                        {user.fullName || user.username}
                    </span>
                );
            }
        }
        return <strong {...props}>{children}</strong>;
    };

    const replaceVariables = (value: string) => {
        if (variables && value) {
            return variablesInterpolator.interpolateInText(variables, value);
        } else {
            return value;
        }
    };

    const renderMarkDownPreview = () => (
        <div
            className={classNames('markdown-viewer', { 'read-only': isReadOnly })}
            onDoubleClick={() => !isReadOnly && setEditing(true)}
            role="button"
            tabIndex={isReadOnly ? undefined : 0}
        >
            {!isReadOnly ? (
                <div className="markdown-viewer-actions">
                    {onDelete && originalText && <DotIconButton data-testid="delete-button" iconId="delete" onClick={handleDelete} size="small" />}
                    <DotIconButton data-testid="edit-button" iconId="edit" onClick={() => setEditing(true)} size="small" />
                </div>
            ) : null}
            {currentText ? (
                <DotTooltip title={!isReadOnly ? 'Double-click to edit' : undefined}>
                    <ReactMarkdown
                        className="markdown-wrapper"
                        components={{
                            strong: renderMarkDownStrongElement,
                            // eslint-disable-next-line react/display-name,react/prop-types
                            a: ({ href, ...props }) => {
                                if (href) {
                                    const internalRelativePath = href.startsWith('#/');
                                    const internalAbsolutePath = href.startsWith(document.location.origin);
                                    const openInNewTab = !internalAbsolutePath && !internalRelativePath;
                                    if (openInNewTab) {
                                        return <a {...props} href={href} rel="noreferrer noopener" target="_blank" />;
                                    } else {
                                        return <a {...props} href={href} />;
                                    }
                                }
                                return <a {...props} />;
                            },
                        }}
                    >
                        {enableMentions ? replaceVariables(wrapMentionsText(currentText)) : replaceVariables(currentText)}
                    </ReactMarkdown>
                </DotTooltip>
            ) : (
                !isReadOnly && (
                    <DotTooltip title="Double-click to edit">
                        <div className="placeholder-wrapper" onDoubleClick={() => setEditing(true)}>
                            <DotTypography className="placeholder" variant="body1">
                                {placeholder}
                            </DotTypography>
                        </div>
                    </DotTooltip>
                )
            )}
        </div>
    );

    return <div className="markdown-switcher">{editing ? renderTextEditor() : renderMarkDownPreview()}</div>;
};
