import React from 'react';
import { ReactWrapper } from 'enzyme';
import { DotThemeProvider } from '@digital-ai/dot-components';
import { TextAreaMarkdown, TextareaMarkdownProps } from './textarea-markdown.component';
import { mountComponentWithStore } from '../../../../tests/unit/testing-utils';

type ActionTypeId = 'save-button' | 'cancel-button' | 'edit-button' | 'delete-button';

describe('TextAreaMarkdown Component', () => {
    const onTextChange = jest.fn();

    const defaultProps: TextareaMarkdownProps = {
        inputId: 'markdown-input',
        name: 'markdown',
        onTextChange,
        rows: 5,
        text: 'my **markdown** text',
    };

    const mount = (props: TextareaMarkdownProps = defaultProps) => {
        return mountComponentWithStore(
            <DotThemeProvider>
                <TextAreaMarkdown {...props} />
            </DotThemeProvider>,
        );
    };

    const markdownSwitcherSelector = '.markdown-switcher';
    const markdownViewerSelector = `${markdownSwitcherSelector} .markdown-viewer`;
    const markdownWrapperSelector = `${markdownViewerSelector} .markdown-wrapper p`;
    const selectAction = (testId: ActionTypeId) => `${markdownSwitcherSelector} button[data-testid="${testId}"]`;
    const textareaSelector = `${markdownSwitcherSelector} textarea#markdown-input`;

    const checkInitialState = (wrapper: ReactWrapper) => {
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my markdown text');
        expect(wrapper.find(`${markdownWrapperSelector} strong`).text()).toBe('markdown');
    };

    const switchToEditMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('edit-button')).simulate('click');
        expect(wrapper.find(textareaSelector).exists()).toBeTruthy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeFalsy();
    };

    const switchToViewMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('cancel-button')).simulate('click');
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeTruthy();
    };

    const selectMarkup = (markupSelector: string) => `${markdownWrapperSelector} ${markupSelector}`;

    beforeEach(() => {
        jest.resetAllMocks();
    });

    test('should render markdown by default', () => {
        const wrapper = mount();
        checkInitialState(wrapper);
    });

    test('should switch to edit mode', () => {
        const wrapper = mount();
        // initial markdown mode
        checkInitialState(wrapper);
        // edit mode
        switchToEditMode(wrapper);
        // markdown mode
        switchToViewMode(wrapper);
        expect(onTextChange).not.toHaveBeenCalled();
    });

    test('should save and cancel changes', () => {
        const wrapper = mount();
        // original text
        checkInitialState(wrapper);

        // do change
        switchToEditMode(wrapper);
        wrapper.find(textareaSelector).simulate('change', { target: { value: 'my ```fancy code```' } });
        // cancel change
        switchToViewMode(wrapper);
        // original state
        checkInitialState(wrapper);
        expect(wrapper.find(selectMarkup('code')).exists()).toBeFalsy();
        // do change
        switchToEditMode(wrapper);
        wrapper.find(textareaSelector).simulate('change', { target: { value: 'my ```fancy code```' } });
        // save change
        wrapper.find(selectAction('save-button')).simulate('click');
        expect(onTextChange).toHaveBeenCalledWith('my ```fancy code```');
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my fancy code');
        expect(wrapper.find(selectMarkup('code')).text()).toBe('fancy code');
    });
});
