package com.xebialabs.xlrelease.storage.s3

import com.xebialabs.xlrelease.storage.Storage
import com.xebialabs.xlrelease.storage.s3.S3Storage.{StringExtension, UriExtension, DELIMITER}
import grizzled.slf4j.Logging
import software.amazon.awssdk.core.ResponseInputStream
import software.amazon.awssdk.core.exception.{SdkClientException, SdkException}
import software.amazon.awssdk.regions.Region
import software.amazon.awssdk.services.s3.S3Client
import software.amazon.awssdk.services.s3.model._
import software.amazon.awssdk.core.sync.RequestBody
import software.amazon.awssdk.utils.StringUtils.isNotBlank

import java.net.URI
import java.io.{FileNotFoundException, IOException, InputStream}
import scala.jdk.CollectionConverters._

class S3Storage(private val config: S3StorageConfig) extends Storage with Logging {

  private[s3] val s3Client = {
    val builder = S3Client.builder()
      .region(Region.of(config.region))
    if (isNotBlank(config.endpointUri)) {
      builder.endpointOverride(URI.create(config.endpointUri))
    }
    builder.build()
  }

  private implicit def implicitConfig: S3StorageConfig = config

  override def put(uri: URI, data: InputStream): URI = {
    val key = uri.toFileKey
    val putObjectRequest = PutObjectRequest.builder()
      .bucket(config.bucketName)
      .key(key)
      .build()
    val requestBody = RequestBody.fromInputStream(data, data.available())
    val putResponse = s3Client.putObject(putObjectRequest, requestBody)
    if (logger.isTraceEnabled) {
      logger.trace(s"Put object $key with request ID ${putResponse.responseMetadata().requestId()}")
    }
    key.toKeyUri
  }

  override def get(uri: URI): InputStream = {
    val key = uri.toFileKey
    try {
      val getObjectRequest = GetObjectRequest.builder()
        .bucket(config.bucketName)
        .key(key)
        .build()
      logger.trace(s"Get object $key")
      val s3object: ResponseInputStream[GetObjectResponse] = s3Client.getObject(getObjectRequest)
      s3object
    } catch {
      case _: NoSuchKeyException =>
        throw new FileNotFoundException(s"Unable to find file at '$uri'")
      case e: SdkClientException =>
        throw new IOException(s"Unable to find file at '$uri'", e)
    }
  }

  override def exists(uri: URI): Boolean = {
    val key = uri.toFileKey
    try {
      val headObjectRequest = HeadObjectRequest.builder()
        .bucket(config.bucketName)
        .key(key)
        .build()
      s3Client.headObject(headObjectRequest)
      true
    } catch {
      case _: NoSuchKeyException =>
        false
      case e: SdkClientException =>
        throw new IOException(s"Unable to find file at '$uri'", e)
    }
  }

  override def size(uri: URI): Long = {
    val key = uri.toFileKey
    try {
      val headObjectRequest = HeadObjectRequest.builder()
        .bucket(config.bucketName)
        .key(key)
        .build()
      val response = s3Client.headObject(headObjectRequest)
      if (logger.isTraceEnabled) {
        logger.trace(s"Get size $key with version ${response.versionId()}")
      }
      response.contentLength()
    } catch {
      case _: NoSuchKeyException =>
        throw new FileNotFoundException(s"Unable to find file at '$uri'")
      case e: SdkClientException =>
        throw new IOException(s"Unable to find file size at '$uri'", e)
    }
  }

  override def delete(uri: URI): Boolean = {
    val key = uri.toFileKey
    try {
      logger.trace(s"Delete object $key")
      s3Client.deleteObject(
        DeleteObjectRequest.builder()
          .bucket(config.bucketName)
          .key(key)
          .build()
      )
      true
    } catch {
      case _: NoSuchKeyException =>
        logger.warn(s"No such key: $key")
        false
      case e: SdkException =>
        throw new IOException(s"Unable to delete file at '$uri'", e)
    }
  }

  override def deleteIfEmpty(uri: URI): Boolean = {
    val key = uri.toKey
    try {
      val listObjectsRequest = ListObjectsV2Request.builder()
        .bucket(config.bucketName)
        .prefix(key)
        .build()
      val objectListing = s3Client.listObjectsV2(listObjectsRequest)
      if (objectListing.contents().isEmpty) {
        val deleteObjectRequest = DeleteObjectRequest.builder()
          .bucket(config.bucketName)
          .key(key)
          .build()
        s3Client.deleteObject(deleteObjectRequest)
        true
      } else {
        false
      }
    } catch {
      case _: NoSuchKeyException =>
        true
      case e: SdkClientException =>
        throw new IOException(s"Unable to delete if empty at '$uri'", e)
    }
  }

  override def listDirectories(uri: URI): List[URI] = {
    val key = uri.toKey
    try {
      logger.trace(s"List directories for $key")
      val objectListing = s3Client.listObjectsV2Paginator(
        ListObjectsV2Request.builder()
          .bucket(config.bucketName)
          .prefix(key)
          .delimiter(DELIMITER)
          .build()
      )

      objectListing.iterator().asScala
        .flatMap(list =>
          list.commonPrefixes().asScala
            .sortBy { s =>
              (s.prefix().length, s.prefix())
            }
            .map { s =>
              s.prefix().toKeyUri
            }
        )
        .toList
    } catch {
      case _: NoSuchKeyException =>
        throw new FileNotFoundException(s"Unable to find directory key at '$uri'")
      case e: SdkClientException =>
        throw new IOException(s"Unable to list directories at '$uri'", e)
    }
  }

  override def listFiles(uri: URI): List[URI] = {
    val key = uri.toKey
    try {
      logger.trace(s"List files for $key")

      val objectListing = s3Client.listObjectsV2Paginator(
        ListObjectsV2Request.builder()
          .bucket(config.bucketName)
          .prefix(key)
          .delimiter(DELIMITER)
          .build()
      )

      objectListing.iterator().asScala
        .flatMap(list =>
          list.contents().asScala
            .sortBy { obj =>
              (obj.key().length, obj.key())
            }
            .map(obj => obj.key().toKeyUri)
        )
        .toList
    } catch {
      case _: NoSuchKeyException =>
        throw new FileNotFoundException(s"Unable to find directory key at '$uri'")
      case e: SdkClientException =>
        throw new IOException(s"Unable to list files at '$uri'", e)
    }
  }

  override def cleanup(uri: URI): Boolean = {
    val key = uri.toKey
    try {
      logger.trace(s"Clean directories under $key")

      val listingToDelete = s3Client.listObjectsV2Paginator(
        ListObjectsV2Request.builder()
          .bucket(config.bucketName)
          .prefix(key)
          .delimiter(DELIMITER)
          .build()
      )

      val deleteIterator = listingToDelete.iterator().asScala

      while (deleteIterator.hasNext) {
        // returns max keys per request - 1000
        val keysToDelete = deleteIterator.next().contents().asScala
          .map(obj =>
            ObjectIdentifier.builder()
              .key(obj.key())
              .build()
          )
        if (keysToDelete.nonEmpty) {
          val delete = Delete.builder()
            .objects(keysToDelete.asJava)
            .build()
          s3Client.deleteObjects(
            DeleteObjectsRequest.builder()
              .bucket(config.bucketName)
              .delete(delete)
              .build()
          )
        }
      }
      true
    } catch {
      case _: NoSuchKeyException =>
        throw new FileNotFoundException(s"Unable to find directory key at '$uri'")
      case e: Exception =>
        logger.warn(s"Exception while deleting under key $key: ${e.getMessage}")
        false
    }
  }

  override def uriScheme: String = config.uriScheme
}

object S3Storage {

  private val DELIMITER = "/"

  implicit class StringExtension(keyPath: String) {

    def toKeyUri(implicit config: S3StorageConfig): URI = new URI(config.uriScheme, null, DELIMITER + prepareKeyPath(config), null)

    def toKey(implicit config: S3StorageConfig): String = {
      val key = prepareKeyPath(config)
      if (key.endsWith(DELIMITER)) key else key + DELIMITER // ensure key ends with delimiter
    }

    def toFileKey(implicit config: S3StorageConfig): String = prepareKeyPath(config).stripSuffix(DELIMITER) // ensure key does not end with delimiter

    private def prepareKeyPath(config: S3StorageConfig): String = {
      val path = keyPath.replace("\\", DELIMITER).stripPrefix(DELIMITER)
      if (path.startsWith(config.baseKey)) path else config.baseKey + DELIMITER + path
    }
  }

  implicit class UriExtension(uri: URI) {
    def toKey(implicit config: S3StorageConfig): String = {
      uri.getPath.toKey(config)
    }

    def toFileKey(implicit config: S3StorageConfig): String = {
      uri.getPath.toFileKey(config)
    }
  }
}