package com.xebialabs.xlrelease.storage.domain

import com.fasterxml.jackson.annotation.JsonSubTypes.Type
import com.fasterxml.jackson.annotation.{JsonSubTypes, JsonTypeInfo}
import com.xebialabs.xlrelease.support.serialization.SerializableMsg
import org.apache.commons.codec.digest.DigestUtils

import java.io.{ByteArrayInputStream, InputStream}
import java.net.URI

sealed trait StorageScheme {
  def uriScheme: String
}

@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "_type", visible = true)
@JsonSubTypes(Array(
  new Type(value = classOf[LogEntry], name = "LogEntry")
))
sealed trait StorageEntry extends StorageScheme with SerializableMsg {
  def uriPath: String

  def content: InputStream
}

case class LogEntry(taskId: String,
                    executionId: String,
                    jobId: Long,
                    chunk: Long,
                    lastEntryTimestamp: String,
                    payload: Array[Byte],
                    uriScheme: String
                   ) extends StorageEntry {
  override def toString: String = s"jobId=$jobId, executionId=$executionId, taskId=$taskId, chunk=$chunk, lastEntryTimestamp=$lastEntryTimestamp"

  override def uriPath: String = {
    val taskIdHash = DigestUtils.sha256Hex(IdUtils.getFolderlessId(taskId))
    s"/jobs/$taskIdHash/$executionId/$jobId/$chunk"
  }

  override def content: InputStream = new ByteArrayInputStream(payload)
}

@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "_type", visible = true)
@JsonSubTypes(Array(
  new Type(value = classOf[LogEntryRef], name = "LogEntryRef"),
  new Type(value = classOf[JobEntryRef], name = "JobEntryRef"),
))
sealed trait StorageEntryRef extends StorageScheme with SerializableMsg {
  def uri: URI

  override def uriScheme: String = uri.getScheme
}

case class LogEntryRef(taskId: String, executionId: String, jobId: Long, chunk: Long, lastEntryTimestamp: String, uri: URI) extends StorageEntryRef {
  override def toString: String = s"taskId=$taskId, jobId=$jobId, chunk=$chunk, uri='$uri'"
}

object LogEntryRef {
  def from(logEntry: LogEntry, storedEntryUri: URI): LogEntryRef = {
    val taskId = logEntry.taskId
    val executionId = logEntry.executionId
    val jobId: Long = logEntry.jobId
    val chunk: Long = logEntry.chunk
    val lastEntryTimestamp: String = logEntry.lastEntryTimestamp
    LogEntryRef(taskId, executionId, jobId, chunk, lastEntryTimestamp, storedEntryUri)
  }
}


case class JobEntryRef(uri: URI) extends StorageEntryRef
