package com.xebialabs.xlrelease.storage.local

import com.xebialabs.xlrelease.storage.Storage
import grizzled.slf4j.Logging
import org.springframework.util.{FileSystemUtils, StreamUtils}

import java.io._
import java.net.URI
import java.nio.file.{Files, NoSuchFileException, Path, Paths}
import scala.util.Using

class LocalStorage(val config: LocalStorageConfig) extends Storage with Logging {

  createDirectory(null)

  private implicit def implicitConfig: LocalStorageConfig = config

  override def put(uri: URI, data: InputStream): URI = {
    this.createDirectory(uri)
    Using.resource(new FileOutputStream(uri.getFullPath.toFile)) {
      StreamUtils.copy(data, _)
    }
    URI.create(s"$uriScheme://${uri.getPath}")
  }

  override def get(uri: URI): InputStream = {
    new BufferedInputStream(new FileInputStream(URI.create(uri.getPath).getAbsolutePathLocation))
  }

  override def size(uri: URI): Long = {
    try {
      Files.size(URI.create(uri.getPath).getFullPath)
    } catch {
      case _: NoSuchFileException =>
        throw new FileNotFoundException(s"Unable to find file at '$uri'")
      case e: IOException =>
        throw new IOException(s"Unable to find file size at '$uri'", e)
    }
  }

  override def delete(uri: URI): Boolean = {
    val file = URI.create(uri.getPath).getFullPath.toFile
    try {
      FileSystemUtils.deleteRecursively(file)
    } catch {
      case ex: Exception =>
        logger.error(s"Unable to delete ${file.getName}", ex)
        throw ex
    }
  }

  override def deleteIfEmpty(uri: URI): Boolean = {
    val file = URI.create(uri.getPath).getFullPath.toFile
    if (file.exists() && file.isDirectory && file.list().isEmpty) {
      file.delete()
    } else {
      false
    }
  }

  private def createDirectory(append: URI): Unit = {
    val file = if (append != null) {
      val path = Paths.get(config.basePath.toAbsolutePath.toString, append.getPath)
      path.getParent.toFile
    } else {
      new File(config.basePath.toUri)
    }
    if (!file.exists() && !file.mkdirs()) {
      if (!file.exists()) {
        throw new RuntimeException(s"Cannot create directory: ${file.getAbsolutePath}")
      } else {
        // ignore
      }
    }
  }

  override def uriScheme: String = config.uriScheme

  private implicit class UriExtension(uri: URI) {
    def getFullPath(implicit config: LocalStorageConfig): Path = {
      val pathChunks = uri.getPath.split("/")
      val path = if (uri.getPath.matches("^/[a-zA-Z]:.*")) Paths.get("", pathChunks: _*) else Paths.get("/", pathChunks: _*)
      if (path.startsWith(config.basePath.toAbsolutePath.toString)) {
        path
      } else {
        Paths.get(config.basePath.toAbsolutePath.toString, path.toString)
      }
    }

    def getAbsolutePathLocation(implicit config: LocalStorageConfig): String = getFullPath(config).toAbsolutePath.toString
  }

  private def listContents(directory: File, filenameFilter: FilenameFilter): List[URI] = {
    val uri = directory.toPath.toUri.toString
    if (directory.isDirectory) {
      val dirItems: List[String] = directory.list(filenameFilter).toList
      if (dirItems.exists(!_.forall(Character.isDigit))) {
        dirItems.sorted.map(name => URI.create(s"$uri$name"))
      } else {
        dirItems.map(_.toLong).sorted.map(name => URI.create(s"$uri$name"))
      }
    } else {
      List.empty
    }
  }

  override def listDirectories(uri: URI): List[URI] = {
    val file = URI.create(uri.getPath).getFullPath.toFile
    listContents(file, (dir: File, name: String) => {
      new File(dir, name).isDirectory
    })
  }

  override def listFiles(uri: URI): List[URI] = {
    val file = URI.create(uri.getPath).getFullPath.toFile
    listContents(file, (dir: File, name: String) => {
      new File(dir, name).isFile
    })
  }
}
