package com.xebialabs.xlrelease.script;

import java.util.Map;
import java.util.stream.Collectors;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.FolderVariables;
import com.xebialabs.xlrelease.domain.variables.GlobalVariables;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.*;
import com.xebialabs.xlrelease.repository.CiCloneHelper;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.CiIdService;

import static com.xebialabs.xlrelease.domain.Changes.VariablesChanges;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static com.xebialabs.xlrelease.variable.VariableFactory.createVariableByValueType;
import static com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds;
import static java.util.Collections.singletonList;
import static java.util.Optional.ofNullable;

abstract class ScriptVariableProcessorFactory {

    static ScriptVariableProcessorFactory globalVariablesProcessor(CiIdService ciIdService, Task task, GlobalVariables globals) {
        return new GlobalVariablesProcessor(ciIdService, task, globals);
    }

    static ScriptVariableProcessorFactory folderVariablesProcessor(PermissionChecker permissionChecker, Task task, FolderVariables folderVariables) {
        return new FolderVariablesProcessor(permissionChecker, task, folderVariables);
    }

    static ScriptVariableProcessorFactory releaseVariableProcessor(CiIdService ciIdService, Release release) {
        return new ReleaseVariablesProcessor(ciIdService, release);
    }

    VariablesChanges processVariablesAndUpdateInContainer(Map<String, Object> contextVariables, Map<String, Variable> initialVariables) {
        VariablesChanges changes = new VariablesChanges();
        Map<String, Object> decryptedContextVariables = contextVariables.entrySet().stream()
                .collect(toMap(Map.Entry::getKey, entry -> {
                    if (null != entry.getValue()
                            && initialVariables.containsKey(entry.getKey())
                            && entry.getValue() instanceof String
                            && initialVariables.get(entry.getKey()).isPassword()) {
                        return PasswordEncrypter.getInstance().ensureDecrypted((String) entry.getValue());
                    } else {
                        return entry.getValue();
                    }
                }));
        Map<String, Object> decryptedInitialVariableValues = initialVariables.entrySet().stream()
                .collect(toMap(Map.Entry::getKey, e -> {
                    Object value = ofNullable(e.getValue().getValue()).orElse(e.getValue().getEmptyValue());
                    if (value instanceof String && e.getValue().isPassword()) {
                        return PasswordEncrypter.getInstance().ensureDecrypted((String) value);
                    } else {
                        return value;
                    }
                }));

        MapDifference<String, Object> decryptedDifference = Maps.difference(decryptedContextVariables, decryptedInitialVariableValues);

        //Make sure all added & updated variables have updated value and not decrypted
        for (Map.Entry<String, Object> createdVariable : decryptedDifference.entriesOnlyOnLeft().entrySet()) {
            Variable addedVariable = newVariable(createdVariable.getKey(), contextVariables.get(createdVariable.getKey()));

            changes.getOperations().add(createOperation(addedVariable));
            changes.getCreatedVariables().add(addedVariable);
        }

        for (Map.Entry<String, Object> deletedVariable : decryptedDifference.entriesOnlyOnRight().entrySet()) {
            Variable variable = initialVariables.get(withoutVariableSyntax(deletedVariable.getKey()));

            changes.getOperations().add(deleteOperation(variable));
            changes.getDeletedVariables().add(variable);
        }

        for (Map.Entry<String, MapDifference.ValueDifference<Object>> updatedVariable : decryptedDifference.entriesDiffering().entrySet()) {
            Variable variable = initialVariables.get(withoutVariableSyntax(updatedVariable.getKey()));
            Variable original = CiCloneHelper.cloneCi(variable);

            variable.setUntypedValue(contextVariables.get(updatedVariable.getKey()));

            changes.getOperations().add(updateOperation(original, variable));
            changes.getUpdatedVariables().add(variable);
        }

        return changes;
    }

    abstract Variable newVariable(String key, Object value);

    abstract VariableUpdateOperation updateOperation(Variable original, Variable updated);

    abstract VariableCreateOperation createOperation(Variable newVariable);

    abstract VariableDeleteOperation deleteOperation(Variable deletedVariable);

    private static class ReleaseVariablesProcessor extends ScriptVariableProcessorFactory {
        private CiIdService ciIdService;
        private Release release;

        ReleaseVariablesProcessor(CiIdService ciIdService, Release release) {
            this.ciIdService = ciIdService;
            this.release = release;
        }

        @Override
        Variable newVariable(String key, Object value) {
            Variable variable = createVariableByValueType(key, value, false, false);
            release.checkVariableCanBeAdded(variable);
            fixUpVariableIds(release.getId(), singletonList(variable), ciIdService);
            return variable;
        }

        @Override
        VariableUpdateOperation updateOperation(Variable original, Variable updated) {
            return new ReleaseVariableUpdateOperation(original, updated);
        }

        @Override
        VariableCreateOperation createOperation(Variable newVariable) {
            return new ReleaseVariableCreateOperation(newVariable);
        }

        @Override
        VariableDeleteOperation deleteOperation(Variable deletedVariable) {
            return new ReleaseVariableDeleteOperation(deletedVariable);
        }
    }

    private static class GlobalVariablesProcessor extends ScriptVariableProcessorFactory {
        private CiIdService ciIdService;
        private String taskId;
        private GlobalVariables globalVariables;

        GlobalVariablesProcessor(CiIdService ciIdService, Task task, GlobalVariables globalVariables) {
            this.ciIdService = ciIdService;
            this.globalVariables = globalVariables;
            this.taskId = task.getId();
        }

        @Override
        Variable newVariable(String key, Object value) {
            Variable variable = globalVariables.addVariable(createVariableByValueType(key, value, false, true));
            fixUpVariableIds(globalVariables.getId(), globalVariables.getVariables(), ciIdService);
            return variable;
        }

        @Override
        VariableUpdateOperation updateOperation(Variable original, Variable updated) {
            return new GlobalVariableUpdateOperation(original, updated, taskId);
        }

        @Override
        VariableCreateOperation createOperation(Variable newVariable) {
            return new GlobalVariableCreateOperation(newVariable, taskId);
        }

        @Override
        VariableDeleteOperation deleteOperation(Variable deletedVariable) {
            return new GlobalVariableDeleteOperation(deletedVariable, taskId);
        }
    }

    private static class FolderVariablesProcessor extends ScriptVariableProcessorFactory {
        private PermissionChecker permissionChecker;
        private Task task;
        private FolderVariables folderVariables;

        public FolderVariablesProcessor(PermissionChecker permissionChecker, Task task, final FolderVariables folderVariables) {
            this.permissionChecker = permissionChecker;
            this.task = task;
            this.folderVariables = folderVariables;
        }

        @Override
        Variable newVariable(final String key, final Object value) {
            final String folderId = task.getRelease().findFolderId();
            permissionChecker.check(EDIT_FOLDER_VARIABLES, folderId);
            Variable variable = createVariableByValueType(key, value, false, true);
            variable.setFolderId(folderId);
            variable = folderVariables.addVariable(variable);
            return variable;
        }

        @Override
        VariableUpdateOperation updateOperation(final Variable original, final Variable updated) {
            permissionChecker.check(EDIT_FOLDER_VARIABLES, updated.getFolderId());
            return new FolderVariableUpdateOperation(original, updated, task.getId());
        }

        @Override
        VariableCreateOperation createOperation(final Variable newVariable) {
            return new FolderVariableCreateOperation(newVariable, task.getId());
        }

        @Override
        VariableDeleteOperation deleteOperation(final Variable deletedVariable) {
            permissionChecker.check(EDIT_FOLDER_VARIABLES, deletedVariable.getFolderId());
            return new FolderVariableDeleteOperation(deletedVariable, task.getId());
        }
    }
}
