package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.authentication.AuthenticationSupport
import com.xebialabs.xlrelease.scheduler.PreconditionJob
import com.xebialabs.xlrelease.scheduler.workers.Worker.{ExecuteJob, PreconditionExecutionResult, ProcessJobResult}
import com.xebialabs.xlrelease.script._
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class PreconditionWorker(val releaseActorService: ReleaseActorService,
                         val scriptServicesList: java.util.List[PreconditionService])
  extends Worker with AuthenticationSupport with TaskWorkerFailureLogic with Logging {

  private val preconditionsServices: java.util.Map[String, PreconditionService] = scriptServicesList.asScala.map(s => s.engine() -> s).toMap.asJava

  override def execute: ExecuteJob = {
    case PreconditionJob(taskRef) =>
      val task = taskRef.get()
      val result = Try {
        val preconditionEngine = task.getPreconditionType.engine.toLowerCase()
        preconditionsServices.get(preconditionEngine).executePrecondition(task)
      }
      PreconditionExecutionResult(task.getId, task.getExecutionId, result)
  }

  override def processResult: ProcessJobResult = {
    case PreconditionExecutionResult(taskId, executionId, executionResult) =>
      executionResult match {
        case Failure(error) => handleFailure(taskId, executionId, error)
        case Success(result) => result match {
          case ValidPreconditionResult(taskId, executionId, _, _, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.taskPreconditionValidated(taskId, executionId)
            }
          case ExceptionPreconditionResult(taskId, executionId, executionLog, attachmentId, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, None)
            }
          case InvalidPreconditionResult(taskId, executionId, executionLog, _, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.skipTaskDueToPreconditionCheck(taskId, executionId, executionLog)
            }
        }
      }
  }
}
