package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.utils.ArchivedReleaseExporter
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Service

import java.io.File

@Service
class ArchivePurgingScheduleService @Autowired()(archivingService: ArchivingService,
                                                 archivingConfig: ArchivingSettingsManager,
                                                 @Qualifier("reportingJdbcTemplate") reportingJdbcTemplate: JdbcTemplate,
                                                 archivedReleases: ArchivedReleases)
  extends BaseBackgroundJobService[String] {
  val actions = List(
    new PurgeAction(archivingService, archivingConfig, reportingJdbcTemplate, archivedReleases)
  )

  override def jobDisplayName: String = "archive purging"

  override def isJobEnabled(): Boolean = {
    archivingConfig.getPurgingEnabled
  }

  override def getCronSchedule(): String = {
    archivingConfig.getPurgingJobCronSchedule
  }

  override def getPeriodInHours(): Int = {
    archivingConfig.getMaximumArchiveRetentionPeriod
  }

  override def getJobActions(): List[BackgroundJobAction[String]] = {
    actions
  }

  override def getMaxSecondsForJob(): Int = {
    archivingConfig.getPurgingMaxSecondsPerRun
  }

  override def getConfiguredPageSize(): Int = {
    archivingConfig.getPurgingSearchPageSize
  }

  override def getSleepSecondsBetweenItems(): Int = {
    archivingConfig.getPurgingSleepSecondsBetweenReleases
  }
}

class PurgeAction(archivingService: ArchivingService,
                  archivingConfig: ArchivingSettingsManager,
                  reportingJdbcTemplate: JdbcTemplate,
                  archivedReleases: ArchivedReleases) extends BackgroundJobAction[String] with Logging {
  override def queryIdentifiers(jobStartTime: LocalDateTime, jobQueryPeriodInHours: Int, pageSize: Int): List[String] = {
    val expirationDate = jobStartTime.minusHours(jobQueryPeriodInHours).toDate()
    archivingService.findPurgableReleaseIds(expirationDate, pageSize).toList
  }

  override def processIdentifier(identifier: String): Unit = {
    try {
      exportDataIfNeeded(identifier)
      archivingService.purgeArchivedRelease(identifier)
    } catch {
      case e: Exception =>
        logger.error(s"Could not purge [$identifier] from the archive: ${e.getMessage}", e)
    }
  }

  private def exportDataIfNeeded(id: String): Unit = {
    if (archivingConfig.getExportOnPurgeEnabled()) {
      exportData(id)
    }
  }

  private def exportData(id: String): Unit = {
    val exportDir = getExportDir(archivingConfig.getPurgeExportPath())
    val exporter = new ArchivedReleaseExporter(reportingJdbcTemplate, exportDir, archivedReleases)
    val path = exporter.exportReleaseData(id)
    logger.info(s"Data for [${id}] exported to ${path}")
  }

  private def getExportDir(path: String): File = {
    val dir = new File(path)
    if (!dir.exists()) {
      try {
        dir.mkdirs()
        logger.debug(s"Created export directory for purged releases at '${dir.getAbsolutePath}' based on provided path '$path'")
      } catch {
        case e: Exception =>
          val msg = s"Unable to create export directory for purged releases '$path'."
          throw new IllegalStateException(msg, e)
      }
    }

    if (!dir.isDirectory) {
      val msg = s"Purged release storage export directory path '$path' does not point to a directory."
      throw new IllegalStateException(msg)
    }

    dir
  }
}
