package com.xebialabs.xlrelease.utils

import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.xlrelease.utils.FileContentValidation.FileTypes
import org.apache.commons.io.FilenameUtils.wildcardMatch
import org.apache.tika.detect.DefaultDetector
import org.apache.tika.io.{TikaInputStream, FilenameUtils => TikaFilenameUtils}
import org.apache.tika.metadata.{Metadata, TikaCoreProperties}

import java.io.InputStream
import java.nio.file.{Files, StandardCopyOption, StandardOpenOption}
import scala.util.{Try, Using}

case class UploadSettings(shouldAnalyzeContent: Boolean, allowedFileTypes: FileTypes)

object FileContentValidation {
  type FileTypes = Map[String, List[String]]

  def filter(tikaDetector: DefaultDetector, uploadSettings: UploadSettings, filename: String, in: InputStream): Try[InputStream] = {
    val sanitizedFilename = TikaFilenameUtils.normalize(filename)
    for {
      matchedFileTypes <- filterByFileName(uploadSettings, sanitizedFilename)
      filteredInputStream <- filterByContent(tikaDetector, UploadSettings(uploadSettings.shouldAnalyzeContent, matchedFileTypes), sanitizedFilename, in)
    } yield filteredInputStream
  }

  private def filterByFileName(uploadSettings: UploadSettings, filename: String): Try[FileTypes] = Try {
    val matchingFileTypes = uploadSettings.allowedFileTypes.filter { case (_, filenames) => filenames.exists(wildcardMatch(filename, _)) }
    if (matchingFileTypes.isEmpty) {
      fail(s"Filename '$filename' is not allowed")
    }
    matchingFileTypes
  }

  private def filterByContent(tikaDetector: DefaultDetector, uploadSettings: UploadSettings, filename: String, in: InputStream): Try[InputStream] = Try {
    if (uploadSettings.shouldAnalyzeContent) {
      val path = Files.createTempFile("upload", ".tmp")
      Files.copy(in, path, StandardCopyOption.REPLACE_EXISTING)

      Using.resource(TikaInputStream.get(path)) { tikaInputStream =>
        val metadata = new Metadata
        metadata.set(TikaCoreProperties.RESOURCE_NAME_KEY, filename)

        val mediaType = tikaDetector.detect(tikaInputStream, metadata).toString
        val matchedMediaType = uploadSettings.allowedFileTypes.find {
          case (allowedMediaType, _) => wildcardMatch(mediaType, allowedMediaType)
        }
        if (matchedMediaType.isEmpty) {
          Files.delete(path)
          fail(s"The file of MIME type '$mediaType' with filename '$filename' is not allowed")
        }
      }

      Files.newInputStream(path, StandardOpenOption.DELETE_ON_CLOSE)
    } else {
      in
    }
  }

  private def fail(message: String): Unit = throw new AttachmentUploadException(message)
}

@HttpResponseCodeResult(statusCode = 400)
class AttachmentUploadException(msg: String, cause: Throwable = null) extends DeployitException(msg, cause)
