package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.ScriptTask.JYTHON_ENGINE
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.scheduler.FacetCheckJob
import com.xebialabs.xlrelease.scheduler.workers.Worker.{ExecuteJob, FacetCheckExecutionResult, ProcessJobResult}
import com.xebialabs.xlrelease.script.{AwaitFacetCheckResult, FailureFacetCheckResult, ScriptService, SuccessFacetCheckResult}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}

@Component
class FacetCheckWorker(val releaseActorService: ReleaseActorService,
                       val scriptServicesList: java.util.List[ScriptService])
  extends Worker with ScriptServiceSupport with TaskWorkerFailureLogic with Logging {

  override def execute: ExecuteJob = {
    case FacetCheckJob(taskRef) =>
      val task = taskRef.get()
      val result = Try {
        scriptServices.get(JYTHON_ENGINE).executeFacetCheck(task)
      }
      FacetCheckExecutionResult(task.getId, task.getExecutionId, result)
  }

  override def processResult: ProcessJobResult = {
    case FacetCheckExecutionResult(taskId, executionId, executionResult) =>
      executionResult match {
        case Failure(error) => handleFailure(taskId, executionId, error)
        case Success(result) => result match {
          case SuccessFacetCheckResult(taskId, executionId, _, _, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.taskPreconditionValidated(taskId, executionId)
            }
          case FailureFacetCheckResult(taskId, executionId, executionLog, attachmentId, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, None)
            }
          case AwaitFacetCheckResult(taskId, executionId, executionLog, _, delayUntil, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.postponeTaskUntilEnvironmentsAreReserved(Ids.releaseIdFrom(taskId), taskId, delayUntil, executionLog, executionId)
            }
        }
      }
  }

}
