package com.xebialabs.xlrelease.json

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.node.ObjectNode
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.xlrelease.domain.status.FlagStatus
import com.xebialabs.xlrelease.export.TemplateJsonHelper.{flagTask, getPhase, typeNotFound}
import grizzled.slf4j.Logging

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

object JsonKeys {
  val variables = "variables"
  val valueProvider = "valueProvider"
  val status = "status"
  val tiles = "tiles"
  val extensions = "extensions"
  val facets = "facets"
  val pythonScript = "pythonScript" // custom script task property
  val capabilities = "capabilities" // container task property
}

object JsonReplacements extends Logging {

  val unknownScriptTaskTypeName = "xlrelease.UnknownType"
  val unknownTaskTypeName = "xlrelease.UnknownTaskType"
  val originalType = "originalType"
  val backupProperty = "backup"
  val defaultProperties: Set[String] = Set("id", "type", originalType, backupProperty, "$createdBy", "$createdAt", "$lastModifiedBy", "$lastModifiedAt", "$token",
    "$scmTraceabilityDataId")

  def typeFound(typeAlias: String): Boolean = Type.valueOf(typeAlias).exists

  implicit class UnknownTaskOps(val task: ObjectNode) extends AnyVal {

    def isUnknownScriptTaskTypeRegistered: Boolean = {
      task.get("type").asText == unknownScriptTaskTypeName && task.has(originalType) && typeFound(task.get(originalType).asText)
    }

    def isUnknownTaskTypeRegistered: Boolean = {
      task.get("type").asText == unknownTaskTypeName && task.has(originalType) && typeFound(task.get(originalType).asText)
    }

    def replaceNotFoundTask(maybeParent: Option[ObjectNode], maybeRelease: Option[ObjectNode]): Unit = {
      val missingType = task.get("type").asText
      val backup = task.toString
      val taskId = maybeParent.getOrElse(task).get("id").asText()
      val taskTitle = maybeParent.getOrElse(task).get("title").asText()
      logger.debug(s"Missing type found $missingType, backing up $backup for task with id $taskId")
      task.put(backupProperty, backup)
      task.put("originalType", missingType)

      val unknownType = maybeParent match {
        case Some(_) => Type.valueOf(unknownScriptTaskTypeName) // custom script task
        case None => Type.valueOf(unknownTaskTypeName)
      }

      val originalType = task.get("type").asText
      task.put("type", unknownType.toString)

      val propertiesOnMissingTask = unknownType.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toSet
      val propertiesOnUnknownTask = keyExtractor(task)
      propertiesOnUnknownTask
        .filter(p => !propertiesOnMissingTask.contains(p) && !defaultProperties.contains(p))
        .foreach(task.remove)

      val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(maybeParent.getOrElse(task), release) }
      val warning = s"Task '$taskTitle' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' has been " +
        s"replaced by an unknown task. The task of type '$originalType' could not be found because of a missing plugin."
      flagTask(maybeParent.getOrElse(task), warning)
    }

    def recoverUnknownScriptTask(parent: ObjectNode): Unit = {
      val backup: ObjectNode = JsonUtils.objectMapper.createObjectNode()
      val backupObject = JsonUtils.objectMapper.readTree(task.get(backupProperty).asText).asInstanceOf[ObjectNode]
      backup.setAll(backupObject)
      backup.set("customScriptTask", task.get("customScriptTask"))
      backup.set("id", task.get("id"))
      parent.set(JsonKeys.pythonScript, backup)
      parent.put("flagStatus", FlagStatus.OK.toString)
      parent.put("flagComment", "")
    }

    def recoverUnknownTask(): Unit = {
      val backup: ObjectNode = JsonUtils.objectMapper.createObjectNode()
      val backupObject = JsonUtils.objectMapper.readTree(task.get(backupProperty).asText).asInstanceOf[ObjectNode]
      backup.setAll(backupObject)
      backup.set("id", task.get("id"))
      backup.fieldNames().asScala.foreach(property => {
        task.remove(property)
        task.putIfAbsent(property, backup.get(property))
      })
      task.remove(backupProperty)
      task.remove(originalType)
    }

    def ignoreTaskUnknownProperties(maybeParent: Option[ObjectNode], maybeRelease: Option[ObjectNode]): Unit = {
      val registeredType = Type.valueOf(task.get("type").asText)
      val typeProps = registeredType.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
      val jsonProps = keyExtractor(task)
      val unknownProps = jsonProps diff (typeProps concat defaultProperties.toList)
      if (unknownProps.nonEmpty) {
        val taskTitle = maybeParent.getOrElse(task).get("title").asText()
        val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(maybeParent.getOrElse(task), release) }
        val warning = s"Task '$taskTitle' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' has " +
          s"properties that do not match with the known type. Following properties do not match: [${unknownProps.mkString(", ")}]."
        flagTask(maybeParent.getOrElse(task), warning)
        unknownProps.foreach(task.remove)
      }
    }

    def ignoreUnknownEnumValues(maybeParent: Option[ObjectNode], maybeRelease: Option[ObjectNode]): Unit = {
      val registeredType = Type.valueOf(task.get("type").asText)
      val enumProps = registeredType.getDescriptor.getPropertyDescriptors.asScala.filter(_.getKind == PropertyKind.ENUM).toList
      val unknownEnumValuesProps = ListBuffer.empty[String]
      enumProps.foreach { prop =>
        val enumValues = registeredType.getDescriptor.getPropertyDescriptor(prop.getName).getEnumValues
        if (task.has(prop.getName) && !enumValues.contains(task.get(prop.getName).asText)) {
          unknownEnumValuesProps += prop.getName
        }
      }
      if (unknownEnumValuesProps.nonEmpty) {
        val taskTitle = maybeParent.getOrElse(task).get("title").asText()
        val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(maybeParent.getOrElse(task), release) }
        val warning = s"Task '$taskTitle' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' " +
          s"has value that do not match with possible enum values. Following properties do not match: [${unknownEnumValuesProps.mkString(", ")}]."
        flagTask(maybeParent.getOrElse(task), warning)
        unknownEnumValuesProps.foreach(task.remove)
      }
    }

  }

  val systemProps = List("id", "type")

  implicit class JSONArrayOps(val obj: ObjectNode) extends AnyVal {

    def toSeq(key: String): Seq[JsonNode] = {
      if (obj.has(key)) {
        obj.get(key).elements().asScala.toList
      } else {
        Seq()
      }
    }

    def fixJsonArray(key: String): Unit = {
      val originalArray = obj.toSeq(key)
      originalArray.foreach(originalNode => originalNode.asInstanceOf[ObjectNode].fixProperties())
      val filtered = originalArray.filter { obj => typeFound(obj.get("type").asText) }
      if (filtered.length < originalArray.length) {
        (originalArray diff filtered).foreach { notFound =>
          logger.info(s"Missing type found ${notFound.get("type").asText}, ignoring type with id ${notFound.get("id").asText}")
        }
        obj.putArray(key).addAll(filtered.asJava)
      }
    }
  }

  private val standardDashboard = "xlrelease.Dashboard"

  private def assumingDashboard(extension: JsonNode) =
    extension.has(JsonKeys.tiles) && extension.has("rows") && extension.has("columns")

  def keyExtractor(jsonNode: JsonNode): List[String] = {
    jsonNode.fields.asScala.map(_.getKey).toList
  }

  implicit class JSONObjectOps(val parent: ObjectNode) extends AnyVal {
    def fixInnerType(key: String): Unit = {
      if (parent.has(key) && parent.get(key).isInstanceOf[ObjectNode]) {
        val objectNode = parent.get(key).asInstanceOf[ObjectNode]
        val typeName = objectNode.get("type").asText
        if (!typeFound(typeName)) {
          parent.remove(key)
        } else {
          val typeProps = Type.valueOf(typeName).getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
          val jsonProps = keyExtractor(objectNode)
          (jsonProps diff (typeProps concat systemProps)).foreach(objectNode.remove)
        }
      }
    }

    def fixProperties(): Unit = {
      val typeName = parent.get("type").asText
      if (typeFound(typeName)) {
        val typeInSystem = Type.valueOf(typeName)
        val typeProps = typeInSystem.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
        val jsonProps = keyExtractor(parent)
        (jsonProps diff (typeProps concat systemProps)).foreach(parent.remove)
      }
    }

    def fixDashboard(): Unit = {
      if (assumingDashboard(parent)) {
        val dashboardType = parent.get("type").asText
        if (!typeFound(dashboardType)) {
          logger.debug(s"Missing dashboard type found $dashboardType, replacing with $standardDashboard type. Dashboard id ${parent.get("id")}")
          parent.put("type", standardDashboard)
        }
        parent.fixJsonArray(JsonKeys.tiles)
      }
    }

    // scalastyle:off cyclomatic.complexity
    def fixTask(maybeRelease: Option[ObjectNode] = None): Unit = {
      if (parent.has(JsonKeys.pythonScript)) {
        parent.get(JsonKeys.pythonScript).asInstanceOf[ObjectNode] match {
          case pythonScript if typeNotFound(pythonScript) =>
            pythonScript.replaceNotFoundTask(Option(parent), maybeRelease)
          case pythonScript if pythonScript.isUnknownScriptTaskTypeRegistered =>
            pythonScript.recoverUnknownScriptTask(parent)
          case pythonScript =>
            pythonScript.ignoreTaskUnknownProperties(Option(parent), maybeRelease)
            pythonScript.ignoreUnknownEnumValues(Option(parent), maybeRelease)
        }
      } else if (parent.has(JsonKeys.capabilities)) {
        parent match {
          case task if typeNotFound(task) =>
            task.replaceNotFoundTask(None, maybeRelease)
          case task if task.isUnknownTaskTypeRegistered =>
            task.recoverUnknownTask()
          case task =>
            task.ignoreTaskUnknownProperties(None, maybeRelease)
            task.ignoreUnknownEnumValues(None, maybeRelease)
        }
      }
    }
    // scalastyle:on cyclomatic.complexity
  }

}
