package com.xebialabs.xlrelease.api.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.xebialabs.deployit.ServerConfiguration;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.Part;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.overthere.util.OverthereUtils;
import com.xebialabs.xlrelease.api.internal.converters.CustomLogoSettingsConverter;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.configuration.*;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.BaseSettings;
import com.xebialabs.xlrelease.domain.events.ConfigurationCreatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ConfigurationService;
import com.xebialabs.xlrelease.service.DbCredentialsChangeResult;
import com.xebialabs.xlrelease.service.DbCredentialsService;
import com.xebialabs.xlrelease.views.*;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.configuration.ThemeSettings.THEME_SETTINGS_ID;
import static scala.jdk.javaapi.CollectionConverters.asJava;


/**
 * The Digital.ai Release settings.
 */
@Path("/settings")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class SettingsResource {

    private static final Logger logger = LoggerFactory.getLogger(SettingsResource.class);
    public static final String MESSAGE = "message";

    private final PermissionChecker permissions;
    private final ConfigurationRepository configurationRepository;
    private final ConfigurationService configurationService;
    private final CustomLogoSettingsConverter customLogoSettingsConverter;
    private final XLReleaseEventBus eventBus;
    private final XlrConfig xlrConfig;
    private final DbCredentialsService dbCredentialsService;
    private final ServerConfiguration serverConfiguration;

    @Autowired
    public SettingsResource(PermissionChecker permissionChecker,
                            ConfigurationRepository configurationRepository,
                            ConfigurationService configurationService,
                            CustomLogoSettingsConverter customLogoSettingsConverter,
                            XLReleaseEventBus eventBus,
                            XlrConfig xlrConfig,
                            DbCredentialsService dbCredentialsService,
                            ServerConfiguration serverConfiguration) {
        this.permissions = permissionChecker;
        this.configurationRepository = configurationRepository;
        this.configurationService = configurationService;
        this.customLogoSettingsConverter = customLogoSettingsConverter;
        this.eventBus = eventBus;
        this.xlrConfig = xlrConfig;
        this.dbCredentialsService = dbCredentialsService;
        this.serverConfiguration = serverConfiguration;

    }

    @GET
    @Timed
    @Path("reports")
    public ReportsSettingsView getReportsSettings() {
        ReportsSettings reportsSettings = configurationRepository.read(ReportsSettings.REPORTS_SETTINGS_ID);
        return new ReportsSettingsView(reportsSettings);
    }

    @GET
    @Timed
    @Path("polling-interval")
    public PollingSettingsView getPollingInterval() {
        PollingSettings pollingSettings = configurationRepository.read(PollingSettings.POLLING_SETTINGS_ID);
        return new PollingSettingsView(pollingSettings);
    }

    @GET
    @Timed
    @Path("archiving-settings")
    public ArchivingSettingsView getArchivingSettings() {
        ArchivingSettings archivingSettings = configurationRepository.read(ArchivingSettings.ARCHIVING_SETTINGS_ID);
        return new ArchivingSettingsView(archivingSettings);
    }

    @GET
    @Timed
    @Path("release-password-settings")
    public ReleasePasswordSettingsView getReleasePasswordSettings() {
        ReleasePasswordSettings releasePasswordSettings = configurationRepository.read(ReleasePasswordSettings.PASSWORD_SETTINGS_ID);
        return new ReleasePasswordSettingsView(releasePasswordSettings);
    }

    @GET()
    @Timed
    @Path("theme")
    public ThemeSettings getThemeSettings() {
        return configurationService.getThemeSettings();
    }

    @GET
    @Timed
    @Path("system-message")
    public SystemMessageSettings getSystemMessage() {
        if (XlrConfig.getInstance().maintenanceModeEnabled()) {
            SystemMessageSettings maintenanceModeMessage = new SystemMessageSettings();
            maintenanceModeMessage.setId(SystemMessageSettings.SYSTEM_MESSAGE_ID);
            maintenanceModeMessage.setEnabled(true);
            maintenanceModeMessage.setMessage("Release server is running in Maintenance Mode");
            maintenanceModeMessage.setTitle("Default Maintenance Mode Message");
            return maintenanceModeMessage;
        }
        return configurationRepository.read(SystemMessageSettings.SYSTEM_MESSAGE_ID);
    }

    @GET
    @Timed
    @Path("upload")
    public Response getUploadAttachmentSettings() {
        return Response.ok(
                Map.of(
                        "fileTypes", asJava(xlrConfig.server().upload().allowedFileTypes()).keySet(),
                        "maxSize", xlrConfig.server().upload().maxSize(),
                        "enabled", xlrConfig.server().upload().shouldAnalyzeContent()
                )
        ).build();
    }

    @GET
    @Timed
    @Path("ci/{ID:Configuration.*}")
    public Response read(@PathParam("ID") String id) {
        permissions.check(ADMIN);
        if (id.equals(THEME_SETTINGS_ID)) {
            BaseConfiguration themeSettings = getThemeSettings();
            return Response.ok(themeSettings).build();
        } else {
            try {
                BaseConfiguration configuration = configurationRepository.read(id);
                return Response.ok(configuration).build();
            } catch (com.xebialabs.deployit.exception.NotFoundException e) {
                logger.debug(e.getMessage());
                return Response.status(Response.Status.NOT_FOUND).build();
            }
        }
    }

    @POST
    @Timed
    @Path("ci/{ID:Configuration.*}")
    public BaseConfiguration create(@PathParam("ID") String id, BaseConfiguration configurationItem) {
        permissions.check(ADMIN);
        configurationItem.setId(id);

        fixSmtpAuthenticationProperties(configurationItem);

        configurationRepository.create(configurationItem);
        BaseConfiguration conf = configurationRepository.read(id);
        eventBus.publish(new ConfigurationCreatedEvent(conf));
        return conf;
    }

    @PUT
    @Timed
    @Path("ci/{ID:Configuration.*}")
    public BaseConfiguration update(@PathParam("ID") String id, BaseConfiguration configurationItem) {
        permissions.check(ADMIN);
        fixSmtpAuthenticationProperties(configurationItem);
        return configurationService.createOrUpdate(id, configurationItem);
    }

    @DELETE
    @Timed
    @Path("ci/{ID:Configuration.*}")
    public void delete(@PathParam("ID") String id) {
        permissions.check(ADMIN);
        configurationService.delete(id);
    }

    @GET
    @Path("content-type/image")
    public List<String> getCustomLogoContentTypes() {
        return CustomLogoContentType.getBrowserTypes();
    }

    @POST
    @Timed
    @Path("custom-logo/upload")
    @Consumes({MediaType.MULTIPART_FORM_DATA})
    public CustomLogoSettings setCustomLogo(@Context HttpServletRequest request) throws IOException {
        permissions.check(ADMIN);

        try {
            Optional<Part> optionalPart = request.getParts().stream().filter(part -> part.getSubmittedFileName() != null && !part.getSubmittedFileName().isEmpty()).findFirst();
            if (optionalPart.isPresent()) {
                Part logoPart = optionalPart.get();
                return configurationService.saveCustomLogo(customLogoSettingsConverter.from(logoPart.getSubmittedFileName(), logoPart.getContentType(), logoPart.getInputStream()));
            } else {
                throw new BadRequestException("Expected logo file in request");
            }
        } catch (ServletException e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof IllegalStateException stateException) {
                throw stateException;
            }
            logger.error("Unable to upload file", e);
            throw new BadRequestException("Expected multipart content");
        } catch (IllegalStateException e) {
            throw new BadRequestException("Unable to upload file", e);
        }
    }

    @GET
    @Path("custom-logo/download")
    public Response downloadCustomLogo() {
        try {
            CustomLogoSettings customLogoSettings = configurationRepository.read(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID);
            if (customLogoSettings.getContent() == null) {
                return Response.status(Response.Status.NOT_FOUND).build();
            }
            CustomLogoSettingsView customLogoSettingsView = customLogoSettingsConverter.toView(customLogoSettings);

            StreamingOutput output = stream -> {
                InputStream is = new ByteArrayInputStream(customLogoSettingsView.getContent());
                OverthereUtils.write(is, stream);
            };

            return Response.ok(output, customLogoSettingsView.getContentType()).build();
        } catch (com.xebialabs.deployit.exception.NotFoundException e) {
            return Response.status(Response.Status.NOT_FOUND).build();
        }
    }

    @DELETE
    @Path("custom-logo")
    public Response deleteCustomLogo() {
        try {
            CustomLogoSettings customLogoSettings = configurationRepository.read(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID);
            customLogoSettings.setContent(null);
            customLogoSettings.setContentType(null);
            customLogoSettings.setFilename(null);
            configurationService.saveCustomLogo(customLogoSettings);
            return Response.ok().build();
        } catch (com.xebialabs.deployit.exception.NotFoundException e) {
            return Response.status(Response.Status.NOT_FOUND).build();
        }
    }

    @GET
    public List<BaseSettings> getSystemSettings() {
        return configurationService.getSystemSettings();
    }

    @PUT
    @Timed
    public List<BaseSettings> updateSettings(List<BaseSettings> settings) {
        permissions.check(ADMIN);
        configurationService.saveSettings(settings);
        return configurationService.getSystemSettings();
    }

    @GET
    @Path("features")
    public List<FeatureSettings> getFeatureSettings() {
        return configurationService.getFeatureSettings();
    }

    @GET
    @Path("features/{type}")
    public FeatureSettings getFeatureSetting(@PathParam("type") String type) {
        return configurationService.getFeatureSettings(type);
    }

    @PUT
    @Timed
    @Path("features")
    public List<FeatureSettings> updateFeatureSettings(List<FeatureSettings> features) {
        permissions.check(ADMIN);
        configurationService.saveFeatureSettings(features);
        return configurationService.getFeatureSettings();
    }

    @PUT
    @Path("/reloadDbConfig")
    public Response reloadDbConfig(UpdateDBCredentialsForm form) {
        permissions.check(ADMIN);

        String database = form.getDatabase();
        String username = form.getDbUsername();
        String password = form.getDbPassword();

        if (xlrConfig.isLiveDbCredentialUpdateEnabled()) {
            DbCredentialsChangeResult result = dbCredentialsService.validateAndPublishCredentialsChangeRequest(username, password, database);

            if (result.success()) {
                return Response.ok(Map.of(MESSAGE, result.message())).build();
            } else {
                return Response.status(Response.Status.BAD_REQUEST).entity(Map.of(MESSAGE, result.message())).build();
            }
        } else {
            return Response.status(Response.Status.METHOD_NOT_ALLOWED).entity(Map.of(MESSAGE, "This feature is currently disabled")).build();
        }
    }
    @GET
    @Timed
    @Path("ai")
    public Response getAiSettings() {
        return Response.ok(
                Map.of(
                        "enabled", xlrConfig.aiFeature().enabled(),
                        "assistantUrl", xlrConfig.aiFeature().assistantUrl(),
                        "serverUrl", serverConfiguration.getServerUrl(),
                        "uiTimeoutMs", xlrConfig.aiFeature().uiTimeoutMs()
                )
        ).build();
    }

    @GET
    @Timed
    @Path("account-lockout")
    public Response getAccountLockoutSettings() {
        return Response.ok(
                Map.of(
                        "enabled", xlrConfig.isAccountLockoutEnabled(),
                        "maxFailedAttempts", xlrConfig.maxFailedLoginAttempt(),
                        "lockoutDurationMinutes", xlrConfig.accountLockoutDuration().toMinutes()
                )
        ).build();
    }

    /**
     * For any create/update via UI, reset value of accessToken and expiresAt (to generate a new accessToken) in case of OAuth2SmtpAuthentication for SMTP Server
     * Modifications are done via types as SmtpServer lives in notification module
     * TODO: Move create/update for SmtpServer in SmtpResource in xlr-notifications-module
     */
    private void fixSmtpAuthenticationProperties(BaseConfiguration baseConfiguration) {
        if (baseConfiguration.getType().equals(Type.valueOf("xlrelease.SmtpServer"))
                && baseConfiguration.hasProperty("authentication")) {
            BaseConfigurationItem authentication = baseConfiguration.getProperty("authentication");
            if (authentication != null && authentication.getType().equals(Type.valueOf("xlrelease.OAuth2SmtpAuthentication"))) {
                if (authentication.hasProperty("accessToken")) {
                    authentication.setProperty("accessToken", null);
                }
                if (authentication.hasProperty("expiresAt")) {
                    authentication.setProperty("expiresAt", null);
                }
            }
        }
    }
}
