package com.xebialabs.xlrelease.actors.cluster

import com.xebialabs.xlplatform.cluster.XlCluster
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Seed}
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.support.report.ClusterUsageStatistics
import grizzled.slf4j.Logging
import org.apache.pekko.cluster.Cluster

import scala.concurrent.{Await, ExecutionContextExecutor}
import scala.jdk.CollectionConverters._

trait ClusterStatisticsProvider {

  def getClusterStatistics: ClusterUsageStatistics = {
    val statistics = new ClusterUsageStatistics
    val cluster = actorSystemHolder.cluster()
    val mode = XlrConfig.getInstance.cluster.mode
    statistics.setClusterMode(mode.toString)
    statistics.setClustered(true)
    val dbNodes = getActiveDbNodes
    statistics.setClusterDbNodes(dbNodes.asJava)
    statistics.setClusterNumberOfNodes(dbNodes.size)
    statistics.setClusterActorSystem(actorSystemHolder.actorSystemName())
    statistics.setClusterPekkoNodes(
      cluster.state.members.map(member => s"datacenter: ${member.dataCenter}, address: ${member.address.toString}, status: ${member.status}").toList.asJava
    )
    statistics.setClusterLeader(s"datacenter: ${cluster.selfDataCenter}, address: ${cluster.state.leader.map(_.toString).getOrElse("No leader")}")

    statistics
  }

  def actorSystemHolder: ActorSystemHolder

  def getActiveDbNodes: Seq[String]
}


case class XlrPekkoNativeClusterStatisticsProvider(actorSystemHolder: ActorSystemHolder,
                                                   membersService: ClusterMembersService) extends ClusterStatisticsProvider with Logging {

  def getActiveDbNodes: Seq[String] = {
    membersService.listActiveMembers().map(m => s"datacenter: ${m.datacenter}, address: ${m.protocol}://${m.system}@${m.host}:${m.port}")
  }
}

case class XlrLegacyClusterStatisticsProvider(xlrConfig: XlrConfig, actorSystemHolder: ActorSystemHolder) extends ClusterStatisticsProvider with Logging {

  def getActiveDbNodes: Seq[String] = {
    val actorSystem = actorSystemHolder.unmanagedActorSystem
    implicit val context: ExecutionContextExecutor = actorSystem.dispatcher
    val management = XlCluster.get(actorSystem).membershipManagement
    val cluster = Cluster(actorSystem)
    try {
      val seeds = Await.result(management.listActiveSeeds(cluster), xlrConfig.timeouts.releaseActorReceive)
      val dbNodes = seeds match {
        case Data(seeds: Seq[Seed]) => seeds.map(_.address.toString)
        case _ => Seq.empty
      }
      dbNodes
    } catch {
      case e: Throwable =>
        logger.error("Exception encountered querying active seeds.", e)
        Seq.empty
    }
  }
}

case class StandaloneClusterStatisticsProvider(actorSystemHolder: ActorSystemHolder) extends ClusterStatisticsProvider with Logging {

  def getActiveDbNodes: Seq[String] = Seq.empty

  override def getClusterStatistics: ClusterUsageStatistics = {
    val statistics = new ClusterUsageStatistics

    val mode = XlrConfig.getInstance.cluster.mode
    statistics.setClusterMode(mode.toString)

    statistics
  }
}


