package com.xebialabs.xlrelease.config

import com.typesafe.config.ConfigRenderOptions._
import com.typesafe.config.{Config, ConfigException, ConfigFactory, ConfigRenderOptions}
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.xlplatform.cluster.ClusterMode.Full
import com.xebialabs.xlplatform.cluster._
import com.xebialabs.xlplatform.config.ConfigLoader
import com.xebialabs.xlplatform.security.validator.regex.RegexValidator.RegexValidation
import com.xebialabs.xlplatform.settings.XlPlatformSettings
import com.xebialabs.xlrelease.actors.cluster.XlrCluster
import com.xebialabs.xlrelease.config.XlrConfig._
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration
import com.xebialabs.xlrelease.script.jython.JythonSandboxConfiguration
import com.xebialabs.xlrelease.server.jetty.http.{SameSite, SameSiteHelper}
import com.xebialabs.xlrelease.storage.local.LocalStorageConfig
import com.xebialabs.xlrelease.storage.s3.S3StorageConfig
import com.xebialabs.xlrelease.support.cache.config.{CacheSettings, ConfigCacheSettings}
import com.xebialabs.xlrelease.support.config.TypesafeConfigExt._
import grizzled.slf4j.Logging

import java.nio.file.{Path, Paths}
import java.util.concurrent._
import java.util.{List => JList}
import java.{lang, util}
import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.jdk.CollectionConverters._
import scala.language.postfixOps
import scala.util.Try

object XlrConfig {

  var defaultConfigVar: Option[XlrConfig] = None
  var defaultConfigName: String = {
    val overridden: String = System.getProperty("xl.release.config")
    if (overridden == null || overridden.isEmpty) {
      "xl-release.conf"
    } else {
      overridden
    }
  }

  def defaultConfig: XlrConfig = {
    defaultConfigVar match {
      case None =>
        defaultConfigVar = Some(apply(defaultConfigName))
        defaultConfigVar.get
      case Some(config) => config
    }
  }

  lazy val bootConfig: XlrConfig = bootConfig(defaultConfigName)

  def bootConfig(configName: String): XlrConfig = new XlrConfig(configName)

  private val configDisallowed = List("akka")

  private val disallowedIgnoreList = List("pekko.remote.akka.version")

  def apply(configName: String): XlrConfig = {
    val config = new XlrConfig(configName, SecretKeyHolder.get())
    // https://www7.v1host.com/V1Production/defect.mvc/Summary?oidToken=Defect%3A1940958
    // initialisation lazyness is making (some) thread pool initializations to happen when triggered by a script task
    // execution, making use of the wrong (script) AcessControlContext and providing a breeding ground for bugs.
    config.init()
    config
  }

  def getInstance: XlrConfig = defaultConfig

  def reloadConfig(configName: String = defaultConfigName): Unit = {
    defaultConfigVar = Some(apply(configName))
    defaultConfigName = configName
  }

  class DatabaseConfiguration(db: String, rootConfig: Config) {
    implicit val configPath: String = s"xl.$db"
    lazy val config: Config = rootConfig.getConfig(s"xl.$db")
    lazy val dbDriverClassname: String = sysGetString(s"db-driver-classname", config)
    lazy val dbUrl: String = sysGetString(s"db-url", config)
    lazy val dbUsername: String = sysGetString(s"db-username", config)
    lazy val dbPassword: String = sysGetString(s"db-password", config)
    lazy val poolName: String = sysGetString(s"pool-name", config)
    lazy val maxPoolSize: Int = sysGetInt(s"max-pool-size", config)
    lazy val maxLifeTime: Long = sysGetDuration(s"max-life-time", config)
    lazy val idleTimeout: Long = sysGetDuration(s"idle-timeout", config)
    lazy val minimumIdle: Int = sysGetInt(s"minimum-idle", config)
    lazy val connectionTimeout: Long = sysGetDuration(s"connection-timeout", config)
    lazy val leakConnectionThreshold: Long = sysGetDuration(s"leak-connection-threshold", config)
  }

  private def sysGetString(path: String, config: Config)(implicit configPath: String): String = sysGet(path, config)(identity, _.getString)

  private def sysGetInt(path: String, config: Config)(implicit configPath: String): Int = sysGet(path, config)(Integer.parseInt, _.getInt)

  private def sysGetBoolean(path: String, config: Config)(implicit configPath: String): Boolean = sysGet(path, config)(toBoolean, _.getBoolean)

  private def sysGetDuration(path: String, config: Config, timeUnit: TimeUnit = TimeUnit.MILLISECONDS)(implicit configPath: String): Long =
    sysGet(path, config)(Integer.parseInt(_).toLong, c => c.getDuration(_, timeUnit))

  private def sysProps = System.getProperties

  private def sysGet[A](path: String, config: Config)(fromSys: String => A, fromConfig: Config => String => A)(implicit configPath: String): A = {
    val fullPath = if (configPath.isEmpty) path else s"$configPath.$path"
    if (sysProps.containsKey(fullPath)) {
      fromSys(sysProps.getProperty(fullPath))
    } else {
      fromConfig(config).apply(path)
    }
  }

  private def toBoolean(s: String): Boolean = s match {
    case "true" | "True" | "1" | "on" => true
    case _ => false
  }

}


//noinspection ScalaStyle
class XlrConfig(@BeanProperty val rootConfig: Config) extends Logging {

  def this(configName: String, secretKeyHolder: SecretKeyHolder) = {
    this(ConfigFactory.defaultOverrides().withFallback(ConfigLoader.loadSecuredWithDynamic(configName, secretKeyHolder)))
  }

  private[config] def this(configName: String) =
    this(ConfigFactory.defaultOverrides().withFallback(ConfigLoader.loadWithDynamic(configName)))

  private def sysGetString(path: String): String = XlrConfig.sysGetString(path, rootConfig)("")

  @BeanProperty
  lazy val xl: Config = rootConfig.getConfig("xl")

  if (logger.isDebugEnabled) {
    val configRenderOptions: ConfigRenderOptions = defaults.setOriginComments(false).setJson(false)
    logger.debug(xl.root().render(configRenderOptions))
  }

  def init(): Unit = {

    def inDisallowed(property: String): Option[Seq[String]] = {
      Option(configDisallowed.filter(d => {
        val inProperty = property.contains(d) && !disallowedIgnoreList.exists(property.contains)
        if (inProperty) logger.error(s"Configuration property \"$property\" contains disallowed keyword \"$d\"")
        inProperty
      }))
    }

    val configMap = rootConfig.toMap

    val disallowed = configMap.keySet.flatMap(property => inDisallowed(property)).flatten

    val ignored: Boolean = Try(rootConfig.getBoolean("xl.ignore-disallowed")).getOrElse(false)

    if (disallowed.nonEmpty && !ignored) {
      val message = "Unsupported settings detected in configuration." +
        s" The following keywords are disallowed: ${disallowed.mkString(",")}." +
        " See the migration guide for details about fixing your configuration."
      throw new IllegalStateException(message)
    }
  }

  def executors: Executors = Executors

  trait ExecutorSettings {
    def maxThreadsCount: Int

    def shutdownTimeout: Int = 10
  }

  trait Executors {
    def scheduler: ExecutorSettings

    def releaseTrigger: ExecutorSettings

    def timeoutExecutor: ExecutorSettings

    def springScheduler: ExecutorSettings

    def quartzJobExecutor: ExecutorSettings

    def pollingExecutor: ExecutorSettings

    def logFlushExecutor: ExecutorSettings

    def preArchivingExecutor: ExecutorSettings

    def auxiliaryExecutor: ExecutorSettings

    def riskCalculation: ExecutorSettings

    def riskCalculationBatch: ExecutorSettings

    def sseEvents: ExecutorSettings
  }

  object Executors extends Executors {
    object scheduler extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.scheduler.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.scheduler.shutdownTimeout", 10)
    }

    object releaseTrigger extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.releaseTrigger.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.releaseTrigger.shutdownTimeout", 10)
    }

    object timeoutExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.scheduler.maxThreadsCount", 1)
      override lazy val shutdownTimeout: Int = xl.getInt("executors.scheduler.shutdownTimeout", 10)
    }

    object springScheduler extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.springScheduler.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.springScheduler.shutdownTimeout", 10)
    }

    object quartzJobExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.quartzScheduler.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.quartzScheduler.shutdownTimeout", 10)
    }

    object pollingExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.polling.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.polling.shutdownTimeout", 10)
    }

    object logFlushExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.log-flush.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.log-flush.shutdownTimeout", 10)
    }

    object preArchivingExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.preArchiving.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.preArchiving.shutdownTimeout", 10)
    }

    object auxiliaryExecutor extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.auxiliary.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.auxiliary.shutdownTimeout", 10)
    }

    object riskCalculation extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.risk-calculation.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.risk-calculation.shutdownTimeout", 10)
    }

    object riskCalculationBatch extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.risk-calculation-batch.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.risk-calculation-batch.shutdownTimeout", 10)
    }

    object sseEvents extends ExecutorSettings {
      lazy val maxThreadsCount: Int = xl.getInt("executors.sse-events.maxThreadsCount")
      override lazy val shutdownTimeout: Int = xl.getInt("executors.sse-events.shutdownTimeout", 10)
    }

  }

  object license {
    lazy val location: String = xl.getString("license.location")
  }

  object confFile {
    lazy val location: String = xl.getString("conf-file.location")
  }

  object quartz {
    lazy val timezone: String = xl.getString("quartz.timezone", null)
  }

  object timeouts {
    private lazy val timeouts = xl.getConfig("timeouts")
    lazy val systemInitialization: FiniteDuration = timeouts.getDuration("systemInitialization", TimeUnit.SECONDS) seconds
    lazy val systemTermination: FiniteDuration = timeouts.getDuration("systemTermination", TimeUnit.SECONDS) seconds
    lazy val taskSchedulerGraceShutdownPeriod: FiniteDuration = timeouts.getDuration("taskSchedulerGraceShutdownPeriod", TimeUnit.SECONDS) seconds
    lazy val releaseActionResponse: FiniteDuration = timeouts.getDuration("releaseActionResponse", TimeUnit.SECONDS) seconds
    lazy val releaseActorReceive: FiniteDuration = timeouts.getDuration("releaseActorReceive", TimeUnit.MILLISECONDS) millis
    lazy val ddataConsistencyTimeout: FiniteDuration = timeouts.getDuration("ddataConsistencyTimeout", TimeUnit.MILLISECONDS) millis
    lazy val scmConnectorRequestTimeout: FiniteDuration = timeouts.getDuration("scmConnectorRequestTimeout", TimeUnit.SECONDS) seconds
    lazy val messageRetryTimeout: FiniteDuration = timeouts.getDuration("messageRetryTimeout", TimeUnit.SECONDS) seconds
    lazy val messageRetryAttempts: Int = timeouts.getInt("messageRetryAttempts")
    lazy val nodeFailureDetectionPeriod: FiniteDuration = timeouts.getDuration("nodeFailureDetectionPeriod", TimeUnit.MILLISECONDS) millis
    lazy val staleReservedJobDetectionPeriod: FiniteDuration = timeouts.getDuration("staleReservedJobPeriod", TimeUnit.MILLISECONDS) millis
  }

  def timeoutSettings: timeouts.type = timeouts

  def cluster = ClusterConfig(xl.getConfig("cluster"))

  object clusterManagement {
    lazy val clusterManager: String = xl.getConfig("cluster").getString("manager")
  }

  object clusterNode {
    private lazy val node = xl.getConfig("cluster.node")
    lazy val datacenter: String = node.getString("datacenter", "default")
  }

  object sharding {
    lazy val numberOfReleaseShards: Int = xl.getConfig("cluster").getInt("number-of-shards")
  }

  object durations {
    private lazy val durations = xl.getConfig("durations")
    lazy val notifyOverduePlanItemsInterval: FiniteDuration = durations.getDuration("notifyOverduePlanItemsInterval", TimeUnit.MILLISECONDS) millis
    lazy val notifyDueSoonTasksInterval: FiniteDuration = durations.getDuration("notifyDueSoonTasksInterval", TimeUnit.MILLISECONDS) millis
    lazy val notifyUserTokenExpiryInterval: FiniteDuration = durations.getDuration("notifyUserTokenExpiryInterval", TimeUnit.MILLISECONDS) millis
    lazy val customScriptTaskScheduleInterval: FiniteDuration = {
      val interval = durations.getDuration("customScriptTaskScheduleInterval", TimeUnit.MILLISECONDS).millis
      if (interval < 1.second) {
        throw new IllegalArgumentException(
          s"Value for customScriptTaskScheduleInterval cannot be less than 1 second but was $interval")
      }
      interval
    }
    lazy val scriptOutputPollingInterval: FiniteDuration = durations.getDuration("scriptOutputPollingInterval", TimeUnit.MILLISECONDS) millis
    lazy val newWatcherPublishDelay: FiniteDuration = durations.getDuration("newWatcherPublishDelay", TimeUnit.MILLISECONDS) millis
    lazy val watcherUnregisterDelay: FiniteDuration = durations.getDuration("watcherUnregisterDelay", TimeUnit.MILLISECONDS) millis
    lazy val releaseActivationDelay: FiniteDuration = durations.getDuration("releaseActivationDelay", TimeUnit.MILLISECONDS) millis
    lazy val triggerDataPurgeJobInterval: FiniteDuration = durations.getDuration("triggerDataPurgeJobInterval", TimeUnit.MINUTES) minute
    lazy val jobQueuePopulateInterval: FiniteDuration = durations.getDuration("jobQueuePopulateInterval", TimeUnit.MILLISECONDS) millis
    lazy val startupJobsDelay: FiniteDuration = durations.getDuration("startupJobsDelay", TimeUnit.MILLISECONDS) millis
  }

  def pekkoConfig: pekko.type = pekko

  object extensionServletSystem {
    lazy val config = xl.getConfig("extension-servlet-system")
  }

  object pekko {
    lazy val pekkoPersistenceConfig: Config = xl.getConfig("pekko-persistence")
    lazy val kryoSerializationConfig: Config = xl.withOnlyPath("pekko-kryo-serialization")
    lazy val nonClusteredConfig: Config = xl.withOnlyPath("pekko").withFallback(kryoSerializationConfig).withFallback(pekkoPersistenceConfig)
    lazy val pekkoNativeConfig: Config = cluster.config.getConfig("xlr-pekko-native")
    lazy val xlrLegacyConfig: Config = cluster.config.getConfig("xlr-legacy")
    private lazy val clusterRootConfig = cluster.config.withOnlyPath("pekko").withFallback(nonClusteredConfig)

    lazy val clusteredConfig: Config = clusterManagement.clusterManager match {
      case XlrCluster.ClusterManagerConstants.PEKKO_NATIVE => pekkoNativeConfig.withFallback(clusterRootConfig)
      case _ => xlrLegacyConfig.withFallback(clusterRootConfig)
    }
    lazy val config: Config = {
      val modeConfig = if (clusterMode == Full) clusteredConfig else nonClusteredConfig
      modeConfig.withFallback(rootConfig.withOnlyPath("xl.dispatchers"))
    }

    lazy val systemName: String = if (clusterMode == Full) cluster.name else xl.getString("systemName")
    lazy val snapshotAfter: Int = pekkoPersistenceConfig.getInt("snapshot-after")
    lazy val keepNrOfBatches: Int = pekkoPersistenceConfig.getInt("keep-nr-of-batches")

    object discovery {
      private lazy val discoveryConfig = config.getConfig("pekko.discovery")
      lazy val method = discoveryConfig.getString("method")

      object jdbcDiscovery {
        private lazy val jdbcConfig = discoveryConfig.getConfig("jdbc")

        lazy val heartbeatInterval: FiniteDuration = jdbcConfig.getDuration("heartbeat-interval", TimeUnit.MILLISECONDS) millis
        lazy val ttl: FiniteDuration = jdbcConfig.getDuration("ttl", TimeUnit.MILLISECONDS) millis
        lazy val cleanupInterval: FiniteDuration = jdbcConfig.getDuration("cleanup-interval", TimeUnit.MILLISECONDS) millis
      }
    }

    object pekkoCluster {
      private lazy val clusterConfig = config.getConfig("pekko.cluster")
      lazy val downingProviderClass: String = clusterConfig.getString("downing-provider-class")
    }
  }

  object development {
    private lazy val development = xl.getConfig("development")
    lazy val restSlowDownDelay: FiniteDuration = development.getDuration("restSlowDownDelay", TimeUnit.MILLISECONDS) millis
    lazy val keepContainer: Boolean = development.getBoolean("keepContainer", false)
  }

  object api {
    private lazy val api = xl.getConfig("api")
    lazy val corsEnabled: Boolean = api.getBoolean("corsEnabled")
  }

  object maintenance {
    private lazy val api = xl.getConfig("maintenance")
    lazy val maintenanceModeEnabled: Boolean = api.getBoolean("enabled")
  }

  def server: ServerSettings.type = ServerSettings

  object ServerSettings {
    private lazy val server = xl.getConfig("server")

    object http {
      private lazy val http = server.getConfig("http")

      object gzip {
        private lazy val gzip = http.getConfig("gzip")
        lazy val enabled: Boolean = gzip.getBoolean("enabled")
        lazy val minSize: lang.Long = gzip.getBytes("minSize")
        lazy val compression: Int = gzip.getInt("compression")
        lazy val excludedPaths: util.List[String] = gzip.getStringList("excludedPaths")
      }

      object cache {
        private lazy val cache = http.getConfig("cache")
        lazy val enabled: Boolean = cache.getBoolean("enabled")
      }

      object csrf {
        private lazy val csrf = http.getConfig("csrf")
        lazy val enabled: Boolean = csrf.getBoolean("enabled")
      }

      object csp {
        private lazy val csp = http.getConfig("csp")
        lazy val enabled: Boolean = csp.getBoolean("enabled")
        lazy val policyDirectives: String = csp.getString("policyDirectives")
      }

      object cookie {
        private lazy val cookie = http.getConfig("cookie")
        lazy val sameSite: SameSite = SameSiteHelper.getSameSiteValue(cookie.getString("sameSite", "Lax"))
      }

      object allowedHosts {
        private lazy val allowedHosts = http.getConfig("allowed-hosts")
        lazy val enabled: Boolean = allowedHosts.getBoolean("enabled")
        lazy val hostnames: util.List[String] = allowedHosts.getStringList("hostnames")
      }
    }

    object http2 {
      private lazy val http2 = server.getConfig("http2")
      lazy val enabled: Boolean = http2.getBoolean("enabled")
    }

    object ssl {
      private lazy val ssl = server.getConfig("ssl")
      lazy val sniHostCheck: Boolean = ssl.getBoolean("sniHostCheck")
    }

    def upload: UploadSettings.type = UploadSettings

    object UploadSettings {
      private lazy val config = server.getConfig("upload")

      lazy val maxSize: Int = config.getInt("max-size", 100)
      lazy val maxSizeBytes: Int = maxSize * 1024 * 1024
      lazy val shouldAnalyzeContent: Boolean = config.getBoolean("analyze-content", false)

      lazy val allowedFileTypes: Map[String, List[String]] = if (config.hasPathOrNull("allowed-content-types")) {
        if (config.getIsNull("allowed-content-types")) {
          Map.empty
        } else {
          config.getObject("allowed-content-types").asScala.map {
            case (fileType, filePatterns) => (fileType, filePatterns.unwrapped().asInstanceOf[JList[String]].asScala.toList)
          }.toMap
        }
      } else {
        Map("*" -> List("*"))
      }
    }

    def session: SessionSettings.type = SessionSettings

    object SessionSettings {
      private lazy val session = server.getConfig("session")

      lazy val maximumSessions: Int = session.getInt("maximumSessions", -1)
      lazy val exceptionIfMaximumExceeded: Boolean = session.getBoolean("exceptionIfMaximumExceeded", false)

      object storage {
        private lazy val storage = session.getConfig("storage")
        lazy val enabled: Boolean = storage.getBoolean("enabled")
        lazy val cleanupCron: String = storage.getString("cleanup-cron")
      }

      object rememberMeSettings {
        private lazy val rememberMe = session.getConfig("remember-me")
        lazy val key: String = rememberMe.getString("key")
        lazy val tokenValidity: Int = rememberMe.getDuration("tokenValidity", TimeUnit.SECONDS).toInt
      }

    }
  }

  object initialization {
    private lazy val initialization = xl.getConfig("initialization")
    lazy val createSampleTemplates: Boolean = initialization.getBoolean("createSampleTemplates")
  }

  object security extends SecuritySettings with ScriptingSecurity with ElSecurity with SecretComplexity with AcceptEncryptedSecretsSecurity with AccountLockoutSecurity

  trait SecuritySettings {
    lazy val security: Config = xl.getConfig("security")

    lazy val isOidcEnabled: Boolean = if (security.hasPath("auth.providers.oidc.enabled")) {
      security.getBoolean("auth.providers.oidc.enabled")
    } else if (security.hasPath("auth.providers.oidc")) {
      true
    } else {
      false
    }

    private lazy val additionalParametersConfigKey = "auth.providers.oidc.additionalParameters"

    lazy val oidcAdditionalParameters: util.Map[String, Object] = if (security.hasPathOrNull(additionalParametersConfigKey) && !security.getIsNull(additionalParametersConfigKey)) {
      security.getObject(additionalParametersConfigKey).unwrapped()
    } else {
      new util.HashMap[String, Object]()
    }
  }

  trait ScriptingSecurity {
    def security: Config

    object scripting {
      private lazy val scripting = security.getConfig("scripting")

      object sandbox {
        private lazy val sandbox = scripting.getConfig("sandbox")

        lazy val decryptPasswords: Boolean = if (xl.hasPath("features.script.sandbox.decryptPasswords")) { // legacy path
          xl.getBoolean("features.script.sandbox.decryptPasswords")
        } else {
          sandbox.getBoolean("decryptPasswords")
        }

        lazy val isEnabled: Boolean = if (xl.hasPath("features.script.sandbox.enabled")) { // legacy path
          xl.getBoolean("features.script.sandbox.enabled")
        } else {
          sandbox.getBoolean("enabled")
        }

        lazy val autoRefreshScriptPolicy = sandbox.getBoolean("autoRefreshScriptPolicy")

        object jython {
          private lazy val jython = sandbox.getConfig("jython")

          lazy val restrictedModules: util.List[String] = jython.getStringList("restricted-modules")
          lazy val restrictedFunctions: util.List[String] = jython.getStringList("restricted-functions")
          lazy val restrictedAttributes: util.List[String] = jython.getStringList("restricted-attributes")

          lazy val isolatePlugins: Boolean = jython.getBoolean("isolate-plugins")
        }

      }

    }

  }

  trait ElSecurity {
    def security: Config

    object el {
      private lazy val elConfig = security.getConfig("el")

      lazy val restrictedMethods: Map[String, List[String]] = {
        val restrictedMethodsConfig = elConfig.getObject("restricted-methods")
        restrictedMethodsConfig.entrySet().asScala.map { entry =>
          val key = entry.getKey
          val methods = restrictedMethodsConfig.get(key).unwrapped().asInstanceOf[java.util.List[String]].asScala.toList
          key -> methods
        }.toMap
      }
    }
  }

  trait SecretComplexity {
    def security: Config

    object secret {
      private lazy val secretComplexity = security.getConfig("secret-complexity")
      private val validations: JList[String] = secretComplexity.getStringList("regex-validations")
      lazy val regexValidations: List[RegexValidation] = validations.asScala.toList.grouped(2)
        .flatMap {
          case List(regex, message) => Some(RegexValidation(regex.r, message))
          case _ =>
            logger.error("One of the patterns in secret-complexity.regex-validations was not properly formatted.")
            None
        }
        .toList
    }

  }

  trait AccountLockoutSecurity {
    def security: Config
    object accountLockoutSettings {
      private lazy val accountLockout = security.getConfig("account-lockout")
      lazy val enabled: Boolean = accountLockout.getBoolean("enabled")
      lazy val maxLoginFailedAttempts: Int = accountLockout.getInt("max-login-failed-attempts")
      lazy val lockoutDuration: FiniteDuration = accountLockout.getDuration("lockout-duration", TimeUnit.MINUTES) minutes

      lazy val whiteListUsername: JList[String] = accountLockout.getStringList("whitelist-username")
    }
  }

  trait AcceptEncryptedSecretsSecurity {
    def security: Config

    object acceptEncryptedSecrets {
      private lazy val acceptEncrypted = security.getConfig("accept-encrypted-secrets")
      lazy val enabled: Boolean = acceptEncrypted.getBoolean("enabled", false)
    }

  }

  object features extends FeaturesSettings

  trait FeaturesSettings {
    private lazy val features = xl.getConfig("features")

    object phases {
      lazy val phasesConfig: Config = features.getConfig("phases")

      object defunctDelete {
        lazy val defunctDeleteConfig: Config = phasesConfig.getConfig("defunct-delete")
        lazy val allowed: Boolean = defunctDeleteConfig.getBoolean("allowed", false)
      }
    }

    object releaseOwner {
      lazy val releaseOwnerConfig: Config = features.getConfig("release-owner")

      object taskTransition {
        lazy val taskTransitionConfig: Config = releaseOwnerConfig.getConfig("task-transition")
        lazy val taskTransitionAllowed: Boolean = taskTransitionConfig.getBoolean("allowed")
      }
    }

    object plugins {
      lazy val plugins: Config = features.getConfig("plugins")
      lazy val jdk17Compatibility: Config = plugins.getConfig("check-jdk17-compatibility")

      lazy val checkJdk17CompatibilityOnPluginInstall: Boolean = jdk17Compatibility.getBoolean("on-plugin-install", true)
      lazy val checkJdk17CompatibilityOnProductUpdate: Boolean = jdk17Compatibility.getBoolean("on-product-upgrade", true)
    }

    object releaseActors {
      lazy val config: Config = features.getConfig("release-actors")

      lazy val releaseCacheEnabled: Boolean = config.getBoolean("cache.enabled")
    }

    object userLastActiveActor {
      lazy val config: Config = features.getConfig("user-last-active-actor")

      lazy val maxBufferSize: Int = config.getInt("max-buffer-size")

      lazy val batchSize: Int = config.getInt("batch-size")

      lazy val flushInterval: FiniteDuration = config.getDuration("flush-interval", TimeUnit.MILLISECONDS) millis

      require(batchSize > 0, "xl.features.user-last-active-actor.batch-size must be a positive integer")
      require(maxBufferSize > 0, "xl.features.user-last-active-actor.max-buffer-size must be a positive integer")
      require(flushInterval.toSeconds >= 1, "xl.features.user-last-active-actor.flush-interval should be at least 1 second")
    }

    object taskExecutionEntryDebounceActor {
      lazy val config: Config = features.getConfig("task-execution-entry-debounce-actor")

      lazy val flushInterval: FiniteDuration = config.getDuration("flush-interval", TimeUnit.MILLISECONDS) millis

      require(flushInterval.toMillis >= 100, "xl.features.task-execution-entry-debounce-actor.flush-interval should be at least 100 millis")
    }

    object userProfile {
      private lazy val userProfile = features.getConfig("user-profile")
      lazy val emailRegex: String = userProfile.getString("email-regex")
    }

    object notifications {
      lazy val config: Config = features.getConfig("notifications")

      lazy val dueSoonEnabled: Boolean = config.getBoolean("due-soon.enabled", true)

      lazy val overdueEnabled: Boolean = config.getBoolean("overdue.enabled", true)

      lazy val userTokenExpiryEnabled: Boolean = config.getBoolean("user-token-expiry.enabled", true)

      lazy val notifyInactiveUsers: Boolean = config.getBoolean("notify-inactive-users.enabled", false)
    }

    object gateActor {
      lazy val config: Config = features.getConfig("gate-actor")

      lazy val enabled: Boolean = config.getBoolean("enabled", true)

      lazy val graceDuration: FiniteDuration = config.getDuration("graceDuration", TimeUnit.MILLISECONDS) millis

      lazy val unstuckInterval: FiniteDuration = config.getDuration("unstuckInterval", TimeUnit.MILLISECONDS) millis
    }

    @Deprecated(forRemoval = true)
    object lockTask {
      lazy val config: Config = features.getConfig("lock-task")

      lazy val assignmentEnabled: Boolean = config.getBoolean("assignment.enabled", true)
    }

    object webhooks {
      private lazy val webhooksConfig: Config = features.getConfig("webhooks")

      object retentionPolicy {
        private lazy val config: Config = webhooksConfig.getConfig("retention-policy")

        lazy val schedule: String = config.getString("schedule")
      }

    }

    object serialization {
      private lazy val serializationConfig: Config = features.getConfig("serialization")
      lazy val failOnUnresolvableReferences: Boolean = serializationConfig.getBoolean("failOnUnresolvableReferences")
      private lazy val jacksonConfig: Config = serializationConfig.getConfig("jackson");
      lazy val maxStringLength: Int = jacksonConfig.getInt("maxStringLength")
    }

    object analytics {
      private lazy val analyticsConfig: Config = features.getConfig("analytics")
      lazy val pendoQueueSize: Int = analyticsConfig.getInt("pendoQueueSize")
      lazy val pendoMetricsCron: String = analyticsConfig.getString("pendoMetricsCron", "0 0 3 * * *")
      lazy val blacklist: JList[String] = analyticsConfig.getStringList("blacklist")
    }

    object liveDbCredentialUpdate {
      lazy val liveDbConfig: Config = features.getConfig("liveDbCredentialUpdate")
      lazy val enabled: Boolean = liveDbConfig.getBoolean("enabled", false)
    }

    object databaseProxy {
      lazy val databaseProxyConfig: Config = features.getConfig("database-proxy")
    }

    object provision {
      private lazy val provisionConfig: Config = features.getConfig("provision")

      object folders {
        private lazy val foldersConfig: Config = provisionConfig.getConfig("folders")

        lazy val enabled: Boolean = foldersConfig.getBoolean("enabled")

        object defaultContentFolder {
          private lazy val defaultContentFolderConfig: Config = foldersConfig.getConfig("default-content-folder")

          lazy val folderTitle: String = defaultContentFolderConfig.getString("folder-title")
          lazy val repositoryUrl: String = defaultContentFolderConfig.getString("repository-url")
          lazy val repositoryTitle: String = defaultContentFolderConfig.getString("repository-title")
          lazy val checkScmBranchOnStart: Boolean = defaultContentFolderConfig.getBoolean("check-scm-branch-on-start")
        }
      }
    }

    object datacenter {
      private lazy val datacenterConfig: Config = features.getConfig("datacenter")

      object stateCheck {
        private lazy val stateCheckConfig: Config = datacenterConfig.getConfig("state-check")
        lazy val enabled: Boolean = stateCheckConfig.getBoolean("enabled", false)
        lazy val interval: FiniteDuration = stateCheckConfig.getDuration("interval", TimeUnit.MILLISECONDS) millis
      }

      object stateChangeCoolDown {
        private lazy val coolDownConfig: Config = datacenterConfig.getConfig("state-change-cool-down")
        lazy val enabled: Boolean = coolDownConfig.getBoolean("enabled", true)
        lazy val interval: FiniteDuration = coolDownConfig.getDuration("interval", TimeUnit.SECONDS) seconds
        lazy val onlyIfUserChanges: Boolean = coolDownConfig.getBoolean("only-if-user-changes", true)
      }
    }

    object ai {
      private lazy val aiConfig: Config = features.getConfig("ai")

      lazy val enabled: Boolean = aiConfig.getBoolean("enabled", true)
      lazy val assistantUrl: String = aiConfig.getString("assistant-url")
    }
  }

  private def localStorageEngineConfig(): LocalStorageConfig = {
    LocalStorageConfig(
      basePath = Paths.get(sysGetString("xl.reporting.engine.location")),
      uriScheme = StorageConfiguration.URI_SCHEME_LOCAL_STORAGE,
    )
  }

  private def s3StorageGenericConfig(config: Config): S3StorageConfig = {
    val baseKey = config.getString("s3.base-key").stripSuffix("/")
    S3StorageConfig(
      region = config.getString("s3.region"),
      bucketName = config.getString("s3.bucket"),
      baseKey = baseKey.stripPrefix("/").stripSuffix("/"), // strip leading and trailing slashes if any
      endpointUri = config.getString("s3.endpoint-uri"),
      uriScheme = StorageConfiguration.URI_SCHEME_S3_STORAGE,
    )
  }

  trait ReportEngineConfiguration {
    def config: Config

    private lazy val engineConfig: Config = config.getConfig("engine")

    lazy val engine: EngineConfig.type = EngineConfig

    object EngineConfig {
      lazy val maxThreadsCount: Int = engineConfig.getInt("maxThreadsCount", 10)
      lazy val shutdownTimeout: Int = engineConfig.getInt("shutdownTimeout", 10)
      lazy val storageType: String = engineConfig.getString("type")
      lazy val localStorageConfig = localStorageEngineConfig()
      lazy val s3StorageConfig = s3StorageGenericConfig(engineConfig)
      lazy val cleanUpInterval: Long = engineConfig.getDuration("cleanUpInterval", TimeUnit.MILLISECONDS)
    }
  }

  object database extends DatabaseConfiguration("database", rootConfig)

  object reporting extends DatabaseConfiguration("reporting", rootConfig) with ReportEngineConfiguration

  def metrics: MetricsSettings.type = MetricsSettings

  object MetricsSettings {
    private lazy val metrics = xl.getConfig("metrics")
    lazy val enabled: Boolean = metrics.getBoolean("enabled")
  }

  def repository: RepositoryConfig.type = RepositoryConfig

  object RepositoryConfig {
    private lazy val repository = xl.getConfig("repository")

    def config: Config = repository

    lazy val workDir: String = {
      try {
        sysGetString("xl.repository.workDir")
      } catch {
        case _: ConfigException.Missing => "work"
      }
    }

    def defaults: DefaultConfig.type = DefaultConfig

    @Deprecated(forRemoval = true)
    lazy val decryptPasswords: Boolean = repository.getBoolean("decryptPasswords")

    object DefaultConfig {
      private lazy val default = xl.getConfig("default")

      def config: Config = default
    }

  }

  object upgrader {
    val VERIFY_MODE = "verify"
    val APPLY_MODE = "apply"

    private lazy val config = xl.getConfig("upgrader")

    lazy val forceRemoveMissingTypes: Boolean = config.getBoolean("forceRemoveMissingTypes")

    // REL-7726 and REL-7258 - parallel updates can cause a lock
    lazy val parallel: Boolean = config.getBoolean("parallel", true)
    lazy val batchSize: Int = config.getInt("batch-size", 20)
    lazy val mode: String = config.getString("mode")
    lazy val skip: util.List[String] = if (config.hasPath("skip")) config.getStringList("skip") else Seq.empty[String].asJava
  }

  def getCacheSettings(defaultConfigName: String): CacheSettings = new ConfigCacheSettings(cache.config, defaultConfigName)

  def getCacheSettings(): CacheSettings = new ConfigCacheSettings(cache.config)

  object cache {
    val config = xl.getConfig("cache")

    def ttl(cacheName: String): FiniteDuration = if (config.hasPath(s"$cacheName.ttl")) {
      config.getDuration(s"$cacheName.ttl", TimeUnit.MINUTES) minutes
    } else {
      FiniteDuration(15, TimeUnit.MINUTES)
    }

    def maxSize(cacheName: String): Int = config.getInt(s"$cacheName.max-size", 1000)

    object securityUser {
      private lazy val securityUser: Config = config.getConfig("security-user")
      lazy val enabled: Boolean = securityUser.getBoolean("enabled", false)
    }

    object externalUserGroup {
      private lazy val userGroup: Config = config.getConfig("external-user-groups")
      lazy val enabled: Boolean = userGroup.getBoolean("enabled", true)
    }
  }

  def getTimeUnit(timeUnitInString: String): TimeUnit = timeUnitInString match {
    case "seconds" => TimeUnit.SECONDS
    case "minutes" => TimeUnit.MINUTES
    case "hours" => TimeUnit.HOURS
    case "days" => TimeUnit.DAYS
    case _ => TimeUnit.MINUTES
  }

  private def localStorageJobLogConfig(config: Config): LocalStorageConfig = {
    LocalStorageConfig(
      basePath = Paths.get(config.getString("directory")),
      uriScheme = StorageConfiguration.URI_SCHEME_LOCAL_STORAGE,
    )
  }

  def jobLog: JobLogConfig.type = JobLogConfig

  object JobLogConfig {
    private lazy val jobLog = xl.getConfig("job-log")
    lazy val storageType: String = jobLog.getString("type")
    lazy val localStorageConfig = localStorageJobLogConfig(jobLog)
    lazy val s3StorageConfig = s3StorageGenericConfig(jobLog)
    lazy val cleanupInterval: Long = jobLog.getDuration("cleanup-interval", TimeUnit.MILLISECONDS)
    lazy val maxItemsPerCleanup: Int = jobLog.getInt("max-items-per-cleanup")
  }


  // accessors for Java/spring context files

  def cacheTtl(cacheName: String): FiniteDuration = this.cache.ttl(cacheName)

  def cacheMaxSize(cacheName: String): Int = this.cache.maxSize(cacheName)

  def xlrRepositoryJdbcUrl: String = this.database.dbUrl

  def xlrRepositoryDriver: String = this.database.dbDriverClassname

  def xlrRepositoryUsername: String = this.database.dbUsername

  def xlrRepositoryPassword: String = this.database.dbPassword

  def reportingDriver: String = this.reporting.dbDriverClassname

  def reportingUrl: String = this.reporting.dbUrl

  def reportingUsername: String = this.reporting.dbUsername

  def reportingPassword: String = this.reporting.dbPassword

  def reportingMaxPoolSize: Int = this.reporting.maxPoolSize

  def reportingPoolName: String = this.reporting.poolName

  def reportingMaxLifetime: Long = this.reporting.maxLifeTime

  def reportingIdleTimeout: Long = this.reporting.idleTimeout

  def reportingMinimumIdle: Int = this.reporting.minimumIdle

  def reportingConnectionTimeout: Long = this.reporting.connectionTimeout

  def reportingLeakConnectionThreshold: Long = this.reporting.leakConnectionThreshold

  def reportingEngineLocation: Path = this.reporting.engine.localStorageConfig.basePath

  def serverExtension_rootPath: String = XlPlatformSettings.apply(rootConfig).ServerExtension.rootPath

  def confFile_location: String = this.confFile.location

  def license_location: String = this.license.location

  def isClusterEnabled: Boolean = cluster.mode != ClusterMode.Standalone

  def clusterMode: ClusterMode = cluster.mode

  def isActiveOnStartup: Boolean = cluster.mode == ClusterMode.Standalone

  def systemTerminationSeconds: Long = this.timeouts.systemTermination.toSeconds

  def taskSchedulerGraceShutdownPeriodSeconds: Long = this.timeouts.taskSchedulerGraceShutdownPeriod.toSeconds

  def taskSchedulerGraceShutdownPeriod: Long = this.timeouts.taskSchedulerGraceShutdownPeriod.toSeconds

  def development_restSlowDownDelay: FiniteDuration = this.development.restSlowDownDelay

  def api_corsEnabled: Boolean = this.api.corsEnabled

  def maintenanceModeEnabled: Boolean = this.maintenance.maintenanceModeEnabled

  def server_http_gzip_enabled: Boolean = this.ServerSettings.http.gzip.enabled

  def server_http_gzip_minSize: lang.Long = this.ServerSettings.http.gzip.minSize

  def server_http_gzip_compression: Int = this.ServerSettings.http.gzip.compression

  def server_http_gzip_excludedPaths: util.List[String] = this.ServerSettings.http.gzip.excludedPaths

  def server_http_cache_enabled: Boolean = this.ServerSettings.http.cache.enabled

  def server_http_csrf_enabled: Boolean = this.ServerSettings.http.csrf.enabled

  def server_http_csp_enabled: Boolean = this.ServerSettings.http.csp.enabled

  def server_http_csp_policyDirectives: String = this.ServerSettings.http.csp.policyDirectives

  def server_http_allowed_hosts_enabled: Boolean = this.ServerSettings.http.allowedHosts.enabled

  def server_http_allowed_hostnames: util.List[String] = this.ServerSettings.http.allowedHosts.hostnames

  def server_http_cookie_sameSite: SameSite = this.ServerSettings.http.cookie.sameSite

  def server_http2_enabled: Boolean = this.ServerSettings.http2.enabled

  def server_ssl_sniHostCheck: Boolean = this.ServerSettings.ssl.sniHostCheck

  def server_session_maximum_sessions: Int = this.ServerSettings.session.maximumSessions

  def server_session_exception_if_maximum_exceeded: Boolean = this.ServerSettings.session.exceptionIfMaximumExceeded

  def server_session_remember_me_key: String = this.ServerSettings.session.rememberMeSettings.key

  def server_session_remember_me_token_validity_seconds: Int = this.ServerSettings.session.rememberMeSettings.tokenValidity

  def server_session_storage_enabled: Boolean = this.ServerSettings.session.storage.enabled

  def server_session_storage_cleanupCron: String = this.ServerSettings.session.storage.cleanupCron

  def initialization_createSampleTemplates: Boolean = this.initialization.createSampleTemplates

  def durations_customScriptTaskScheduleInterval: FiniteDuration = this.durations.customScriptTaskScheduleInterval

  def durations_scriptOutputPollingInterval: FiniteDuration = this.durations.scriptOutputPollingInterval

  def repository_decryptPasswords: Boolean = this.repository.decryptPasswords

  def pluginsConfig: Config = this.features.plugins.plugins

  def isScriptSandboxEnabled: Boolean = this.security.scripting.sandbox.isEnabled

  def isNotifyInactiveUsers: Boolean = this.features.notifications.notifyInactiveUsers

  @Deprecated
  def isScriptSandboxDecryptPasswords: Boolean = this.security.scripting.sandbox.decryptPasswords

  def isAutoRefreshScriptPolicy: Boolean = this.security.scripting.sandbox.autoRefreshScriptPolicy

  def regexPasswordValidationList: List[RegexValidation] = this.security.secret.regexValidations

  def isOidcEnabled: Boolean = this.security.isOidcEnabled

  def oidcAdditionalParameters(): util.Map[String, Object] = this.security.oidcAdditionalParameters

  @Deprecated(forRemoval = true)
  def isLockTaskAssignmentDisabled: Boolean = !this.features.lockTask.assignmentEnabled

  def jythonSandboxConfiguration(): JythonSandboxConfiguration = {
    val jythonScripting = this.security.scripting.sandbox.jython
    JythonSandboxConfiguration(
      jythonScripting.restrictedModules,
      jythonScripting.restrictedFunctions,
      jythonScripting.restrictedAttributes
    )
  }

  def isReleaseOwnerTaskTransitionAllowed: Boolean = this.features.releaseOwner.taskTransition.taskTransitionAllowed

  def isLiveDbCredentialUpdateEnabled: Boolean = this.features.liveDbCredentialUpdate.enabled

  def upgradesToSkip(): util.List[String] = this.upgrader.skip

  def isSecurityUserCacheEnabled: Boolean = this.cache.securityUser.enabled

  def isUserGroupCacheEnabled: Boolean = this.cache.externalUserGroup.enabled

  def isDefunctPhaseDeleteAllowed: Boolean = this.features.phases.defunctDelete.allowed

  def userProfile_emailRegex: String = this.features.userProfile.emailRegex

  def maxStringLength: Integer = features.serialization.maxStringLength

  def upgraderMode: String = upgrader.mode

  def isVerifyUpgraderMode: Boolean = upgraderMode == upgrader.VERIFY_MODE

  def isAccountLockoutEnabled: Boolean = this.security.accountLockoutSettings.enabled

  def maxFailedLoginAttempt: Int = this.security.accountLockoutSettings.maxLoginFailedAttempts

  def accountLockoutDuration: FiniteDuration = this.security.accountLockoutSettings.lockoutDuration

  def whiteListUser: util.List[String] = this.security.accountLockoutSettings.whiteListUsername

  def aiFeature: features.ai.type = features.ai
}
