package com.xebialabs.xlrelease.limits;

import java.util.Optional;
import java.util.function.IntSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import com.xebialabs.xlrelease.configuration.TenantLimit;
import com.xebialabs.xlrelease.exception.RateLimitReachedException;
import com.xebialabs.xlrelease.repository.TenantLimitRepository;
import com.xebialabs.xlrelease.security.authority.TenantAuthority;
import com.xebialabs.xlrelease.utils.TenantContext;

/**
 * Central utility for enforcing tenant limits across XL Release.
 *
 * @since 25.3.0
 */
@Service
public class LimitEnforcer {

    private static final Logger logger = LoggerFactory.getLogger(LimitEnforcer.class);

    private final TenantLimitRepository tenantLimitRepository;

    public LimitEnforcer(TenantLimitRepository tenantLimitRepository) {
        this.tenantLimitRepository = tenantLimitRepository;
    }

    /**
     * Enum defining different types of limits that can be enforced
     */
    public enum LimitType {
        // Team & Folder limits
        FOLDERS("Folders", "folders"),
        TEAM_MEMBERS("Team members", "team members"),
        FOLDER_TEAMS("Folder teams", "folder teams"),

        // Connections
        CONNECTIONS("Connections", "connections"),

        // Templates and Releases
        TEMPLATES("Templates", "templates"),
        RELEASES("Running releases", "running releases"),
        TEMPLATE_PHASES("Template phases", "template phases"),
        TEMPLATE_TASKS("Template tasks", "template tasks"),
        RELEASE_PHASES("Release phases", "release phases"),
        RELEASE_TASKS("Release tasks", "release tasks"),

        // Workflows
        WORKFLOW_TEMPLATES("Workflow templates", "workflow templates"),
        WORKFLOW_EXECUTIONS("Workflow executions", "workflow executions"),
        WORKFLOW_PHASES("Workflow phases", "workflow phases"),
        WORKFLOW_TASKS("Workflow tasks", "workflow tasks"),
        WORKFLOW_EXECUTION_PHASES("Workflow execution phases", "workflow execution phases"),
        WORKFLOW_EXECUTION_TASKS("Workflow execution tasks", "workflow execution tasks"),

        // Triggers
        TRIGGERS("Triggers", "triggers"),

        // Patterns and Deliveries
        PATTERNS("Patterns", "patterns"),
        DELIVERIES("Deliveries", "deliveries"),

        // Applications and Environments
        APPLICATIONS("Applications", "applications"),
        ENVIRONMENTS("Environments", "environments"),

        // Dashboards
        DASHBOARDS("Dashboards", "dashboards"),
        DASHBOARD_TILES("Dashboard tiles", "dashboard tiles"),

        // Variables
        GLOBAL_VARIABLES("Global variables", "global variables"),
        FOLDER_VARIABLES("Folder variables", "folder variables"),
        RELEASE_VARIABLES("Release variables", "release variables"),
        VARIABLE_VALUE_SIZE("Variable value size", "variable value size");


        private final String displayName;
        private final String resourceName;

        LimitType(String displayName, String resourceName) {
            this.displayName = displayName;
            this.resourceName = resourceName;
        }

        public String getDisplayName() {
            return displayName;
        }

        public String getResourceName() {
            return resourceName;
        }
    }

    /**
     * Enforces a specific limit by checking current count against configured limit.
     *
     * @param tenantId             The tenant id for which limit should be checked
     * @param limitType            The type of limit to validate
     * @param increment            Number of items to be added
     * @param currentCountSupplier Supplier to get current count (lazy evaluation)
     * @throws RateLimitReachedException if the action would exceed the limit
     */
    public void enforceLimit(String tenantId, LimitType limitType, int increment, IntSupplier currentCountSupplier) {
        enforceLimit(tenantId, limitType, increment, currentCountSupplier, Optional.empty());
    }

    /**
     * Enforces a specific limit by checking current count against configured limit.
     *
     * @param tenantId             The tenant id for which limit should be checked
     * @param limitType            The type of limit to validate
     * @param increment            Number of items to be added
     * @param currentCountSupplier Supplier to get current count (lazy evaluation)
     * @param limitExceededMessage The custom error message when limit exceeds
     * @throws RateLimitReachedException if the action would exceed the limit
     */
    public void enforceLimit(String tenantId, LimitType limitType, int increment, IntSupplier currentCountSupplier, Optional<String> limitExceededMessage) {
        if (isGlobalAdmin()) {
            logger.debug("Global admin bypassing {} limit (system-wide bypass)", limitType.getResourceName());
            return;
        }

        TenantLimit tenantLimit = tenantLimitRepository.getEffectiveLimit(tenantId);
        if (!tenantLimit.isEnabled()) {
            logger.debug("Limit enforcement is disabled for tenant: {}", tenantId);
            return;
        }

        int limit = getConfiguredLimit(limitType, tenantLimit);
        if (limit < 0) {
            logger.debug("Unlimited {} allowed for tenant: {}", limitType.getResourceName(), tenantId);
            return;
        }

        int currentCount = currentCountSupplier.getAsInt();
        if ((currentCount + increment) > limit) {
            String message = buildLimitExceededMessage(limitType, limit, limitExceededMessage);
            throw new RateLimitReachedException(message);
        }

        logger.debug("Limit enforcement passed for tenant {}: {} {}/{}", tenantId, limitType.getResourceName(), currentCount, limit);
    }

    /**
     * Gets the configured limit for a specific limit type.
     *
     * @param limitType The type of limit to retrieve
     * @return The configured limit value
     */
    private int getConfiguredLimit(LimitType limitType, TenantLimit tenantLimit) {
        return switch (limitType) {
            case FOLDERS -> tenantLimit.getMaxFolders();
            case TEAM_MEMBERS -> tenantLimit.getMaxTeamMembers();
            case FOLDER_TEAMS -> tenantLimit.getMaxTeamsPerFolder();
            case CONNECTIONS -> tenantLimit.getMaxConnections();
            case TEMPLATES -> tenantLimit.getMaxTemplates();
            case RELEASES -> tenantLimit.getMaxRunningReleases();
            case WORKFLOW_TEMPLATES -> tenantLimit.getMaxWorkflowTemplates();
            case WORKFLOW_EXECUTIONS -> tenantLimit.getMaxWorkflowExecutions();
            case TRIGGERS -> tenantLimit.getMaxTriggers();
            case PATTERNS -> tenantLimit.getMaxPatterns();
            case DELIVERIES -> tenantLimit.getMaxDeliveries();
            case TEMPLATE_PHASES -> tenantLimit.getMaxTemplatePhases();
            case TEMPLATE_TASKS -> tenantLimit.getMaxTemplateTasks();
            case RELEASE_PHASES -> tenantLimit.getMaxReleasePhases();
            case RELEASE_TASKS -> tenantLimit.getMaxReleaseTasks();
            case WORKFLOW_PHASES -> tenantLimit.getMaxWorkflowPhases();
            case WORKFLOW_TASKS -> tenantLimit.getMaxWorkflowTasks();
            case WORKFLOW_EXECUTION_PHASES -> tenantLimit.getMaxWorkflowExecutionPhases();
            case WORKFLOW_EXECUTION_TASKS -> tenantLimit.getMaxWorkflowExecutionTasks();
            case APPLICATIONS -> tenantLimit.getMaxApplications();
            case ENVIRONMENTS -> tenantLimit.getMaxEnvironments();
            case DASHBOARDS -> tenantLimit.getMaxDashboards();
            case DASHBOARD_TILES -> tenantLimit.getMaxDashboardTiles();
            case GLOBAL_VARIABLES -> tenantLimit.getMaxGlobalVariables();
            case FOLDER_VARIABLES -> tenantLimit.getMaxFolderVariables();
            case RELEASE_VARIABLES -> tenantLimit.getMaxReleaseVariables();
            case VARIABLE_VALUE_SIZE -> tenantLimit.getMaxVariableValueSize();
        };
    }

    /**
     * Builds a descriptive error message for limit exceeded scenarios.
     */
    private String buildLimitExceededMessage(LimitType limitType, int limit, Optional<String> limitExceededMessage) {
        return limitExceededMessage.orElseGet(() -> String.format("%s limit of %d reached. You cannot add more %s.",
                limitType.getDisplayName(), limit, limitType.getDisplayName().toLowerCase()));
    }

    private boolean isGlobalAdmin() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null || !authentication.isAuthenticated()) {
            return false;
        }

        // Cache system tenant ID to avoid repeated Scala interop calls
        String systemTenantId = TenantContext.SYSTEM_TENANT_ID();

        // Single pass check - only need to find SYSTEM_TENANT_ID
        // The authentication system guarantees only global admins have this
        for (GrantedAuthority authority : SecurityContextHolder.getContext().getAuthentication().getAuthorities()) {
            if (authority instanceof TenantAuthority && systemTenantId.equals(authority.getAuthority())) {
                return true;
            }
        }
        return false;
    }
}
