package com.xebialabs.xlrelease.domain.tasks.task;

import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateValidator;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdater;
import com.xebialabs.xlrelease.security.TaskGranularPermissions;
import com.xebialabs.xlrelease.service.TaskConcurrencyService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

@Component
public class DefaultTaskUpdater implements TaskUpdater {

    @Autowired(required = false)
    private List<TaskUpdateValidator> validators = Collections.emptyList();

    @Autowired
    private TaskGranularPermissions taskGranularPermissions;

    @Autowired
    protected TaskConcurrencyService taskConcurrencyService;

    @Override
    public Class<? extends Task> getTaskClass() {
        return Task.class;
    }

    @Override
    public Changes update(final Task original, final Task updated, Set<TaskUpdateDirective> updateDirectives) {
        return updateAll(original, updated, updateDirectives);
    }

    private Changes updateAll(final Task original, final Task updated, Set<TaskUpdateDirective> updateDirectives) {
        taskConcurrencyService.checkConcurrentModification(original, updated, updateDirectives);

        validators.forEach(v -> v.checkTask(original, updated));
        Changes changes = new Changes();
        boolean didUpdateHappen = false;

        if (hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            original.setTeam(updated.getTeam());
            original.setWatchers(updated.getWatchers());
            updateVariableMapping(original, updated);

            if (!Objects.equals(original.getOwner(), updated.getOwner())) {
                original.setOwner(updated.getOwner());
            }
            didUpdateHappen = true;
        }

        didUpdateHappen = updateTaskTags(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskPostponeDuringBlackout(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskDates(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskFlag(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskDescriptionAndTitle(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskFailureHandler(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskPrecondition(original, updated, updateDirectives) || didUpdateHappen;
        didUpdateHappen = updateTaskPreconditionType(original, updated, updateDirectives) || didUpdateHappen;

        if (didUpdateHappen) {
            taskConcurrencyService.updateLastModifiedDetails(original);
            changes.update(original);
        }
        return changes;
    }

    protected void updateVariableMapping(final Task original, final Task updated) {
        original.getVariableMapping().keySet().removeIf(k ->
                !updated.getVariableMapping().containsKey(k)
        );
        original.getVariableMapping().putAll(updated.getVariableMapping());
    }

    private boolean updateTaskPrecondition(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if ((!original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_PRECONDITION)) ||
                (original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TEMPLATE_TASK_PRECONDITION)) ||
                hasDirectiveToUpdateAllProperties(original, updateDirectives)
        ) {
            original.setPrecondition(updated.getPrecondition());
            return true;
        }
        return false;
    }

    private boolean updateTaskPreconditionType(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if ((!original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_PRECONDITION_TYPE)) ||
                (original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TEMPLATE_TASK_PRECONDITION_TYPE)) ||
                hasDirectiveToUpdateAllProperties(original, updateDirectives)
        ) {
            original.setPreconditionType(updated.getPreconditionType());
            return true;
        }
        return false;
    }


    private boolean updateTaskFailureHandler(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if ((!original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_FAILURE_HANDLER)) ||
                (original.getRelease().isTemplate() && updateDirectives.contains(TaskUpdateDirective.UPDATE_TEMPLATE_TASK_FAILURE_HANDLER)) ||
                hasDirectiveToUpdateAllProperties(original, updateDirectives)
        ) {
            original.setFailureHandler(updated.getFailureHandler());
            original.setTaskFailureHandlerEnabled(updated.isTaskFailureHandlerEnabled());
            original.setTaskRecoverOp(updated.getTaskRecoverOp());
            return true;
        }
        return false;
    }

    protected boolean hasDirectiveToUpdateAllProperties(final Task original, final Set<TaskUpdateDirective> updateDirectives) {
        return taskGranularPermissions.hasDirectiveToUpdateAllProperties(original.getRelease().getId(), updateDirectives);
    }

    private boolean updateTaskTags(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_TAGS) || hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            original.setTags(updated.getTags());
            return true;
        }
        return false;
    }

    private boolean updateTaskDates(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_DATES) || hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            resetDueSoonAndOverdueNotifications(original, updated);
            original.updateDates(updated.getScheduledStartDate(), updated.getDueDate(), updated.getPlannedDuration());
            original.setWaitForScheduledStartDate(updated.isWaitForScheduledStartDate());
            return true;
        }
        return false;
    }

    private void resetDueSoonAndOverdueNotifications(final Task original, final Task updated) {
        boolean dueDateChanged = original.getDueDate() != updated.getDueDate();
        boolean overdueShouldBeNotifiedAgain = original.isOverdueNotified() && !updated.isOverdue();
        boolean dueSoonShouldBeNotifiedAgain = original.isDueSoonNotified() && !updated.isDueSoon();
        if (dueDateChanged) {
            if (overdueShouldBeNotifiedAgain) {
                original.setOverdueNotified(false);
            }
            if (dueSoonShouldBeNotifiedAgain) {
                original.setDueSoonNotified(false);
            }
        }
    }

    private boolean updateTaskFlag(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_FLAG) || hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            original.setFlagStatus(updated.getFlagStatus());
            original.setFlagComment(updated.getFlagComment());
            return true;
        }
        return false;
    }


    private boolean updateTaskDescriptionAndTitle(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_DESCRIPTION_AND_TITLE) || hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            original.setTitle(updated.getTitle());
            original.setDescription(updated.getDescription());
            return true;
        }
        return false;
    }

    private boolean updateTaskPostponeDuringBlackout(Task original, Task updated, Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_TASK_BLACKOUT_POSTPONE) || hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            if (original.isCheckAttributes() && !updated.isCheckAttributes() && original.isPostponedUntilEnvironmentsAreReserved()) {
                // the checkbox was cleared, thus we need to:
                // 1. remove "postponedUntilEnvironmentsAreReserved" flag
                // 2. return old scheduled date
                // 3. clear originalScheduledStartDate
                original.setPostponedUntilEnvironmentsAreReserved(false);
                updated.setScheduledStartDate(updated.getOriginalScheduledStartDate());
                updated.setOriginalScheduledStartDate(null);
            }
            original.setDelayDuringBlackout(updated.isDelayDuringBlackout());
            original.setCheckAttributes(updated.isCheckAttributes());
            return true;
        }
        return false;
    }
}
