package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.security.UserService;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult;
import com.xebialabs.xlrelease.api.v1.forms.UserAccount;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.service.Users;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.utils.SortSupport;
import com.xebialabs.xlrelease.utils.TenantContext$;
import com.xebialabs.xlrelease.views.UserView;
import com.xebialabs.xlrelease.views.users.UserFilters;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;
import static jakarta.ws.rs.core.Response.Status.NOT_FOUND;

/**
 * The user accounts (either internal or external) known to Digital.ai Release.
 */
@Path("/users")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class UserAccountResource {
    private final UserService userService;
    private final Users users;
    private final UserProfileService userProfileService;
    private final PermissionChecker permissionChecker;
    private final SessionService sessionService;
    private final UserAccountValidator userAccountValidator;

    @Autowired
    public UserAccountResource(UserService userService, Users users, UserProfileService userProfileService,
                               PermissionChecker permissionChecker, SessionService sessionService, UserAccountValidator userAccountValidator) {
        this.userService = userService;
        this.users = users;
        this.userProfileService = userProfileService;
        this.permissionChecker = permissionChecker;
        this.sessionService = sessionService;
        this.userAccountValidator = userAccountValidator;
    }

    @GET
    public Page<UserAccount> find(
            @BeanParam UserFilters userFilters,
            @DefaultValue("0") @QueryParam(PAGE) Integer page,
            @DefaultValue("50") @QueryParam(RESULTS_PER_PAGE) Integer resultsPerPage,
            @Context UriInfo uriInfo) {
        permissionChecker.check(EDIT_SECURITY);
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        Pageable pageable = PageRequest.of(page, resultsPerPage, SortSupport.toSort(uriInfo));
        Page<UserProfile> userProfiles = userProfileService.searchUserAccounts(userFilters, pageable);
        List<String> internalNames = users.getRepositoryUsernames();
        return userProfiles.map(p -> {
            var name = extractName(p.getId());
            var isInternal = isInternalUser(internalNames, p.getName());
            return new UserAccount(name, p, isInternal);
        });
    }

    @GET
    @Timed
    @Path("names")
    public List<UserView> getUsernames() {
        List<UserProfile> allUserProfiles = userProfileService.findAll(false);
        String currentTenantId = TenantContext$.MODULE$.getTenant();
        if (TenantContext$.MODULE$.isSystemTenant()) {
            return allUserProfiles.stream()
                    .map(profile -> new UserView(profile.getId(), profile.getFullName(), profile.getEmail()))
                    .toList();
        } else {
            return allUserProfiles.stream()
                    .filter(userProfile -> userProfile.getTenantId().equals(currentTenantId))
                    .map(profile -> new UserView(profile.getId(), profile.getFullName(), profile.getEmail()))
                    .toList();
        }
    }

    @POST
    public void create(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        userAccountValidator.check(account);
        userProfileService.validate(account.toUserProfile());
        checkNoUserProfileAlreadyExists(account.getUsername());
        boolean created = insertUser(account);
        if (!created) {
            throw new UserAlreadyExistsException(account.getUsername());
        } else {
            userProfileService.save(account.toUserProfile());
        }
    }

    @PUT
    @Timed
    public Response update(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        String username = account.getUsername();
        checkArgument(isNotBlank(username), "User name cannot be empty.");

        UserProfile profile = userProfileService.findByUsername(username);

        if (profile == null) {
            return Response.status(NOT_FOUND).build();
        }

        profile.setEmail(account.getEmail());
        profile.setFullName(account.getFullName());
        profile.setLoginAllowed(account.isLoginAllowed());
        userProfileService.updateProfile(profile);

        if (account.hasPassword() && users.userExistsInRepository(username)) {
            userAccountValidator.checkPassword(account.getPassword());
            if (isUsernameLoggedIn(username)) {
                userService.modifyPassword(username, account.getPassword(), account.getPreviousPassword());
            } else {
                userService.modifyPassword(username, account.getPassword());
            }
            sessionService.disconnect(username);
        }

        return Response.ok().build();
    }

    @DELETE
    @Timed
    public void delete(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        String username = account.getUsername();

        checkArgument(isNotBlank(username), "User name cannot be empty.");
        checkArgument(!isAdmin(username), "Admin user cannot be deleted.");

        userService.delete(username);
        userProfileService.deleteByUsername(username);
        sessionService.disconnect(username);
    }

    @POST
    @Timed
    @Path("validatePassword")
    public List<PasswordValidationResult> validatePassword(UserAccount account) {
        String password = account.getPassword() == null ? "" : account.getPassword();
        return userAccountValidator.validatePassword(password);
    }

    private boolean isUsernameLoggedIn(String username) {
        return username.equalsIgnoreCase(User.AUTHENTICATED_USER.getName());
    }

    /**
     * Checks for existing external users before creating internal user.
     *
     * @param username
     */
    private void checkNoUserProfileAlreadyExists(String username) {
        boolean userProfileExists = userProfileService.findByUsername(username) != null;
        if (userProfileExists) {
            throw new UserAlreadyExistsException(username);
        }
    }

    /**
     * @return false if the account already existed
     */
    private boolean insertUser(UserAccount account) {
        try {
            userService.create(account.getUsername(), account.getPassword());
        } catch (UserAlreadyExistsException e) {
            return false;
        }
        return true;
    }

    private boolean isInternalUser(List<String> internalNames, String name) {
        for (String internalName : internalNames) {
            if (name.equalsIgnoreCase(internalName)) {
                return true;
            }
        }
        return false;
    }

    private static String extractName(String id) {
        return id.replace(UserProfileService.ROOT + "/", "");
    }

}
