package com.xebialabs.xlrelease.service.accountlock

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.service.{UserProfileService, Users}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Isolation.READ_COMMITTED
import org.springframework.transaction.annotation.Propagation.REQUIRED
import org.springframework.transaction.annotation.Transactional

import java.util.Date
import scala.concurrent.duration.FiniteDuration


@Service
class AccountLockProtectionService(xlrConfig: XlrConfig,
                                   userProfileService: UserProfileService,
                                   eventBus: EventBus,
                                   usersRepository: Users) extends Logging {

  // Configuration - these should ideally come from xl-release.conf
  private val MaxFailedAttempts = xlrConfig.maxFailedLoginAttempt
  private val accountLockoutDuration: FiniteDuration = xlrConfig.accountLockoutDuration
  private val accountLockoutEnabled: Boolean = xlrConfig.isAccountLockoutEnabled

  /**
   * Records a failed login attempt for a username.
   * This method handles the complete business logic for account locking.
   *
   * @param username The username that failed authentication
   * @return Updated UserProfile, or null if user not found
   */
  @Transactional(value = "xlrRepositoryTransactionManager",
    propagation = REQUIRED,
    isolation = READ_COMMITTED,
    noRollbackFor = Array(classOf[NotFoundException]))
  def recordFailedAttempt(username: String): UserProfile = {
    val now = new Date()
    val unlockThresholdTime = new Date(now.getTime - accountLockoutDuration.toMillis)

    // Step 1: Get user profile with row lock (prevents race conditions)
    val userProfile: UserProfile = userProfileService.findByUsernameForUpdate(username)

    if (userProfile == null) {
      logger.error(s"[AccountLockProtectionService][$username]: User profile for [$username] not found. Cannot record failed login attempt.")
      null
    } else {
      val currentCount = userProfile.getLastFailedLoginAttemptCount
      val currentLocked = userProfile.isAccountLocked
      val lastFailedAt = Option(userProfile.getLastFailedLoginAt)

      // Check if we should skip DB update
      if (shouldSkipUpdate(currentLocked, lastFailedAt, unlockThresholdTime)) {
        val minutesRemaining = getMinutesRemaining(lastFailedAt.get, now)
        val event = AutoLockThresholdNotReachedEvent(currentCount, minutesRemaining)
        event.user = username
        eventBus.publish(event)
        userProfile
      } else {
        // Calculate new state
        val shouldReset = lastFailedAt.isEmpty || lastFailedAt.exists(_.before(unlockThresholdTime))
        logger.debug(s"[AccountLockProtectionService][$username] - Calculating new state: " +
          s"currentCount=$currentCount, currentLocked=$currentLocked, shouldReset=$shouldReset")
        val (newCount, newLocked) = calculateNewState(currentCount, currentLocked, shouldReset, MaxFailedAttempts, username)

        // Update database
        val rowsUpdated = userProfileService.updateFailedLoginAttempt(username, newCount, newLocked, now, true)

        if (rowsUpdated > 0) {
          userProfile.setLastFailedLoginAttemptCount(newCount)
          userProfile.setAccountLocked(newLocked)
          userProfile.setLastFailedLoginAt(now)
          // Log warnings
          logStateChanges(username, currentCount, currentLocked, newCount, newLocked)
        } else {
          logger.error(s"[AccountLockProtectionService][$username]: Failed to update failed login attempt for user [$username]. No rows affected.")
        }
        userProfile
      }
    }
  }

  /**
   * Calculates new failed attempt count and lock status based on current state.
   *
   * @param currentCount     Current failed attempt count
   * @param currentLocked    Current lock status
   * @param shouldReset      Whether threshold has passed (auto-unlock condition)
   * @param maxFailedAttempt Maximum allowed failed attempts
   * @return Tuple of (newCount, newLocked)
   */
  private def calculateNewState(currentCount: Int, currentLocked: Boolean, shouldReset: Boolean,
                                maxFailedAttempt: Int, username: String): (Int, Boolean) = {
    if (currentLocked && shouldReset) {
      // Locked account, threshold passed → unlock and reset
      logger.debug(s"[AccountLockProtectionService][$username]: Account locked and threshold passed. Unlocking and resetting counter to 1.")
      (1, false)
    } else if (currentLocked) {
      // Locked account, threshold NOT passed → freeze counter
      logger.debug(s"[AccountLockProtectionService][$username]: Account locked, threshold not passed. Freezing counter at $currentCount.")
      (currentCount, true)
    } else {
      // Unlocked account
      val incrementedCount = currentCount + 1
      val shouldLock = incrementedCount >= maxFailedAttempt

      if (shouldLock) {
        logger.debug(s"[AccountLockProtectionService][$username]: Incrementing counter to $incrementedCount and locking account.")
        (incrementedCount, true)
      } else {
        logger.debug(s"[AccountLockProtectionService][$username]: Incrementing counter to $incrementedCount. Account remains unlocked.")
        (incrementedCount, false)
      }
    }
  }

  /**
   * Logs appropriate warnings based on state changes.
   */
  private def logStateChanges(username: String, currentCount: Int, currentLocked: Boolean,
                              newCount: Int, newLocked: Boolean): Unit = {
    // Log when account gets locked
    if (!currentLocked && newLocked) {
      val autoUnlockMinutes = accountLockoutDuration.toMinutes
      logger.warn(s"[AccountLockProtectionService][$username]: Account locked for user [$username] after reaching $newCount failed attempts. " +
        s"Auto-unlock in $autoUnlockMinutes minutes.")
      val event = AccountLockedEvent(newCount, autoUnlockMinutes)
      event.user = username
      eventBus.publish(event)
    }

    // Log when account gets unlocked
    if (currentLocked && !newLocked) {
      val event = AccountUnlockAndFailedAttemptsResetEvent(newCount)
      event.user = username
      eventBus.publish(event)
    }

    // Log normal increment
    if (!currentLocked && !newLocked && newCount > currentCount) {
      val event = FailedLoginAttemptEvent(newCount, MaxFailedAttempts)
      event.user = username
      eventBus.publish(event)
    }
  }


  /**
   * Determines if database update should be skipped.
   * Skip when account is locked and auto-unlock threshold has not passed.
   */
  private def shouldSkipUpdate(currentLocked: Boolean, lastFailedAt: Option[Date], unlockThresholdTime: Date): Boolean = {
    currentLocked && lastFailedAt.exists(_.after(unlockThresholdTime))
  }

  /**
   * Helper method to calculate remaining minutes until auto-unlock
   */
  private def getMinutesRemaining(lastFailedAt: Date, now: Date): Long = {
    val unlockTime = new Date(lastFailedAt.getTime + accountLockoutDuration.toMillis)
    val remainingMillis = unlockTime.getTime - now.getTime
    Math.max(0, remainingMillis / (60 * 1000))
  }

  def isAccountLocked(username: String): Boolean = {
    if (!accountLockoutEnabled) {
      logger.debug(s"[AccountLockProtectionService][$username] - Account lockout not enabled.")
      false
    } else {
      val userProfile = userProfileService.findByUsername(username)
      if (userProfile == null || !userProfile.isAccountLocked) {
        logger.debug(s"[AccountLockProtectionService][$username] - User not found or Account not locked.")
        false
      } else {
        val now = new Date()
        val lastFailedAt = Option(userProfile.getLastFailedLoginAt)
        val unlockThresholdTime = new Date(now.getTime - accountLockoutDuration.toMillis)
        val isLockedAndThresholdNotReached = lastFailedAt.exists(_.after(unlockThresholdTime))
        if (isLockedAndThresholdNotReached) {
          logger.debug(s"[AccountLockProtectionService][$username] - Account locked: ${userProfile.isAccountLocked}, " +
            s"threshold NOT reached: $isLockedAndThresholdNotReached. Returning true.")
        } else {
          logger.debug(s"[AccountLockProtectionService][$username] - Account locked: ${userProfile.isAccountLocked}, " +
            s"threshold reached: ${!isLockedAndThresholdNotReached}. Returning false.")
        }
        isLockedAndThresholdNotReached
      }
    }
  }

  def resetFailedAttemptsAfterSuccessLogin(username: String): Unit = {
    if (accountLockoutEnabled && usersRepository.userExistsInRepository(username)) {
      val userProfile = userProfileService.findByUsername(username)
      if(userProfile!=null) {
        if (userProfile.isAccountLocked || userProfile.getLastFailedLoginAttemptCount > 0) {
          val rowsUpdated = userProfileService.updateFailedLoginAttempt(username, 0, false, null, true)
          if (rowsUpdated > 0) {
            logger.debug(s"[AccountLockProtectionService][$username] - Reset failed login attempts and unlock account if locked.")
            val event = AccountUnlockedAfterSuccessfulLoginEvent()
            event.user = username
            eventBus.publish(event)
          } else {
            logger.error(s"[AccountLockProtectionService][$username]: Failed to reset failed login attempts. No rows affected.")
          }
        } else {
          logger.debug(s"[AccountLockProtectionService][$username] - No reset needed. Account not locked and failed attempts is 0.")
        }
      } else {
        logger.debug("[AccountLockProtectionService][$username] - No reset needed. User profile not found.")
      }
    }
  }

  def unlockAccount(userProfile: UserProfile): Unit = {
    if(userProfile!= null) {
      val username = userProfile.getCanonicalId
      if (userProfile.isAccountLocked) {
        val rowsUpdated = userProfileService.updateFailedLoginAttempt(username, 0, false, null, true)
        if (rowsUpdated > 0) {
          logger.debug(s"[AccountLockProtectionService][$username] - Admin Action - Manually unlocked account and reset failed login attempts.")
          val event = AccountUnlockEvent()
          event.user = username
          eventBus.publish(event)
        } else {
          logger.error(s"[AccountLockProtectionService][$username]: Failed to unlock account. No rows affected.")
        }
      } else {
        logger.debug(s"[AccountLockProtectionService][$username] - No unlock needed. Account is not locked.")
      }
    } else {
      logger.debug("[AccountLockProtectionService][$username] - No unlock needed. User profile not found.")
    }
  }

  def publishAuthenticationFailureLockEvent(username: String): Unit = {
    val event = AuthenticationFailureLockEvent()
    event.user = username
    eventBus.publish(event)
  }

}
